\name{[-methods}
\docType{methods}

\alias{[-methods}
\alias{[}

\title{Subsetting methods for microarray objects}


\description{
Subsetting methods were defined for the microarray classes, \code{"\link{marrayInfo}"}, \code{"\link{marrayLayout}"}, \code{"\link{marrayRaw}"}, \code{"\link{marrayNorm}"}, \code{"\link{marraySpots}"}, and \code{"\link{marrayTwo}"}. These methods create instances of the given class, for a subset of spots and/or arrays in a batch.
}

\section{Methods}{\describe{
\item{x = ANY}{generic method.}
\item{x = marrayInfo}{\code{x[i, j]} extract object of class \code{"\link{marrayInfo}"} for spots or arrays with indices i and labels with indices j.}
\item{x = marrayLayout}{\code{x[i]} extract object of class \code{"\link{marrayLayout}"} for spots with indices i.}
\item{x = marrayRaw}{\code{x[i, j]} extract object of class \code{"\link{marrayRaw}"} for spots with indices i and arrays with indices j.}
\item{x = marrayTwo}{\code{x[i, j]} extract object of class \code{"\link{marrayTwo}"} for spots with indices i and arrays with indices j.}
\item{x = marraySpots}{\code{x[i]} extract object of class \code{"\link{marraySpots}"} for spots with indices i.}
\item{x = marrayNorm}{\code{x[i, j]} extract object of class \code{"\link{marrayNorm}"} for spots with indices i and arrays with indices j.}
}}

\keyword{methods}


\eof
\name{cbind}
\alias{cbind.marrayRaw}
\alias{cbind.marrayNorm}
\alias{rbind.marrayInfo}

\title{Combine marrayRaw,  marrayNorm or marrayInfo Objects}
\description{
  Combine a series of \code{marrayRaw}, \code{marrayNorm} and
  \code{marrayInfo} objects.
}

\usage{
\method{cbind}{marrayRaw}(\dots, deparse.level=1)
\method{cbind}{marrayNorm}(\dots, deparse.level=1)
\method{rbind}{marrayInfo}(\dots, deparse.level=1)
}
\arguments{
  \item{\dots}{\code{marrayRaw} objects or \code{marrayNorm} objects}
  \item{deparse.level}{not currently used, see \code{\link[base]{cbind}} in the base package}
}
\details{
\code{cbind} combines data objects assuming the same gene lists but
different arrays. 
\code{rbind} combines data objects assuming equivalent arrays, i.e., the same RNA targets, but different genes.

For \code{cbind}, the matrices o f expression data from the individual objects are cbinded.
The data.frames of target information, if they exist, are rbinded.
The combined data object will preserve any additional components or attributes found in the first object to be combined.
For \code{rbind}, the matrices of expression data are rbinded while the target information, in any, is unchanged.
}
\value{
}
\author{Jean Yang}
\seealso{
  \code{\link[base]{cbind}} in the base package.
}
\examples{
}
\keyword{manip}

\eof
\name{coerce-methods}
\docType{methods}


\alias{coerce-methods}
\alias{coerce}
\alias{as}

\title{Coerce an object to belong to a given microarray class}

\description{
Coercing methods were defined to convert microarray objects of one class into objects of another class, e.g., instances of the  \code{"\link{marrayRaw}"} class into instances of the \code{"\link{marrayNorm}"} class.
}

\section{Methods}{\describe{

\item{from = marrayRaw, to = marrayNorm}{convert an object of class \code{"\link{marrayRaw}"} into an object of class \code{"\link{marrayNorm}"}.}
\item{from = marrayRaw, to = exprSet}{convert an object of class \code{"\link{marrayRaw}"} into an object of class \code{"\link{exprSet-class}"}.}
\item{from = marrayRaw, to = marraySpots}{convert an object of class \code{"\link{marrayRaw}"} into an object of class \code{"\link{marraySpots}"}.}
\item{from = marrayNorm, to = exprSet}{convert an object of class \code{"\link{marrayNorm}"} into an object of class \code{"\link{exprSet-class}"}.}
\item{from = marrayNorm, to = marraySpots}{convert an object of class \code{"\link{marrayNorm}"} into an object of class \code{"\link{marraySpots}"}.}

}}

\keyword{methods}



\eof
\name{maCompCoord}
\alias{maCompCoord}
%- Also NEED an `\alias' for EACH other topic documented here.
\title{Generate grid and spot matrix coordinates}
\description{
This function generates grid and spot matrix coordinates from ranges of
rows and columns for the grid and spot matrices. Spots on the array are numbered consecutively starting
from the top left grid and the top left spot within each grid.
}
\usage{
maCompCoord(grows, gcols, srows, scols)
}
%- maybe also `usage' for other objects documented here.
\arguments{
  \item{grows}{numeric vector of grid rows.}
  \item{gcols}{numeric vector of grid columns.}
  \item{srows}{numeric vector of spot rows.}
  \item{scols}{numeric vector of spot columns.}
}

\value{

a matrix of spot four-coordinates, with rows corresponding to
  spots and columns to grid row, grid column, spot row, and spot column coordinates.


}

\author{Yee Hwa (Jean) Yang, Sandrine Dudoit, \url{http://www.stat.berkeley.edu/~sandrine}.}

\seealso{\code{\link{marrayLayout}}, \code{\link{maCoord2Ind}},
\code{\link{maInd2Coord}}, \code{\link{maCompInd}}.}


\examples{

maCompCoord(1:2,1,1:4,1:3)


}
\keyword{manip}
\keyword{methods}

\eof
\name{maCompInd}
\alias{maCompInd}
%- Also NEED an `\alias' for EACH other topic documented here.
\title{Generate spot indices}
\description{
This function generates spot indices from ranges of rows and columns for
the grid and spot matrices. Spots on the array are numbered consecutively starting
from the top left grid and the top left spot within each grid.

}
\usage{
maCompInd(grows, gcols, srows, scols, L)
}
%- maybe also `usage' for other objects documented here.
\arguments{
  \item{grows}{numeric vector of grid rows.}
  \item{gcols}{numeric vector of grid columns.}
  \item{srows}{numeric vector of spot rows.}
  \item{scols}{numeric vector of spot columns.}
  \item{L}{object of class \code{"\link{marrayLayout}"}.}
}

\value{
a numeric vector of spot indices.
}

\author{Yee Hwa (Jean) Yang, Sandrine Dudoit, \url{http://www.stat.berkeley.edu/~sandrine}.}

\seealso{\code{\link{marrayLayout}}, \code{\link{maCoord2Ind}},
\code{\link{maInd2Coord}}, \code{\link{maCompCoord}}.}


\examples{
L <- new("marrayLayout", maNgr=4, maNgc=4, maNsr=22, maNsc=24)
maCompInd(1:2,1,1:4,1:3,L)


}
\keyword{manip}
\keyword{methods}

\eof
\name{maCompPlate}
\alias{maCompPlate}
%- Also NEED an `\alias' for EACH other topic documented here.
\title{Generate plate IDs}
\description{
This function generates plate IDs from the dimensions of the grid and
spot matrices. Note that this function only applies to arrays with a
regular plate layout, where the number of spots is a multiple of the
number of wells on a plate (usually 96 or 384) and each well contributes 
exactly one spot. It should thus be used with caution.}

\usage{
maCompPlate(x, n=384)
}
%- maybe also `usage' for other objects documented here.
\arguments{
  \item{x}{object of class \code{"\link{marrayLayout}"}, \code{"\link{marrayRaw}"}, \code{"\link{marrayNorm}"},  \code{"\link{marraySpots}"}, or  \code{"\link{marrayTwo}"}.}
  \item{n}{object of class "numeric", number of wells in each plate, usually 384 or 96.}
}
\details{
Having plate IDs may be useful for the purpose of normalization. Normalization by plate can be done using the function \code{\link{maNormMain}}.
}
\value{
a vector of plate IDs (\code{\link{factor}}).
}

\author{Yee Hwa (Jean) Yang, Sandrine Dudoit, \url{http://www.stat.berkeley.edu/~sandrine}.}

\seealso{\code{\link{marrayLayout}}, \code{\link{marrayRaw}}, \code{\link{marrayNorm}}, \code{\link{marraySpots}}, \code{\link{marrayTwo}}.}

\examples{

L<-new("marrayLayout", maNgr=4, maNgc=4, maNsr=22, maNsc=24)
plate<-maCompPlate(L,384)
table(plate)
maPlate(L)<-plate

}
\keyword{methods}% at least one, from doc/KEYWORDS


\eof
\name{maCoord2Ind}
\alias{maCoord2Ind}
%- Also NEED an `\alias' for EACH other topic documented here.
\title{Convert grid and spot matrix coordinates to spot indices}
\description{
This functions converts grid and spot matrix coordinates (four
coordinates) to spot
indices, where spots on the array are numbered consecutively starting
from the top left grid and the top left spot within each grid.
}
\usage{
maCoord2Ind(x, L)
}
%- maybe also `usage' for other objects documented here.
\arguments{
  \item{x}{a matrix of spot four-coordinates, with rows corresponding to
  spots and columns to grid row, grid column, spot row, and spot column coordinates.}
  \item{L}{an object of class \code{"\link{marrayLayout}"}.}
}

\value{
 a numeric vector of spot indices.
}

\author{Yee Hwa (Jean) Yang, Sandrine Dudoit, \url{http://www.stat.berkeley.edu/~sandrine}.}

\seealso{\code{\link{marrayLayout}}, \code{\link{maInd2Coord}},
\code{\link{maCompCoord}}, \code{\link{maCompInd}}.}

\examples{
L <- new("marrayLayout", maNgr=4, maNgc=4, maNsr=22, maNsc=24)
coord<-cbind(rep(2,4),rep(1,4),rep(1,4),1:4)
maCoord2Ind(coord, L)


}
\keyword{manip}
\keyword{methods}

\eof
\name{maDotsMatch}
\alias{maDotsMatch}

\title{Replace default arguments of a function by user supplied values}

\description{
  This function may be used to replace default arguements for any
  functions to user supplied parameters.}


\usage{
maDotsMatch(dots, defaults)
}

\arguments{
  \item{dots}{List of user supplied argements, e.g. from \code{list(...)}.}
  \item{defaults}{List of  formal arguments of a function, e.g. from the
    function \code{\link{formals}}.} 
}

\value{
  \item{args}{List of argument of a function.}
}

\author{Jean Yee Hwa Yang}

\seealso{\code{\link{maDefaultPar}}, \code{\link{maDotsDefaults}}}

\examples{
dots<-list(x=1:10, y=11:20)
argsfun <- maDotsMatch(dots, formals(args(plot)))
do.call("plot", argsfun)
}
\keyword{misc}% at least one, from doc/KEYWORDS


\eof
\name{maGenControls}
\alias{maGenControls}
\alias{controlCode}

\title{Generating a vector recording the control status of the spotted probe sequences. }
\description{
  ControlCode is a matrix representing certain regular expression
  pattern and the control status of the spotted probe sequences.
  This function uses `grep' searches for matches to `pattern' (its first argument)
  within the character vector `x' (second argument). 
}

\usage{
maGenControls(Gnames, controlcode, id = "ID")
}

\arguments{
  \item{Gnames}{An object of class \code{matrix}, \code{data.frame} or \code{marrayInfo}
    which contains  description of spotted probe sequences.}
  \item{controlcode}{A character matrix of n by 2 columns. The first
    column contains a few regular expression of spotted probe sequences
    and the second column contains the corresponding control status.}
  \item{id}{the column number of column name in \code{Gnames} that
    contains description of each spot on the array.}
}

\value{
  A vector of characters recording the control status of the spotted probe sequences.
}

\author{Jean Yee Hwa Yang}

\seealso{\code{\link{grep}}}

\examples{
data(swirl)
maControls(swirl) <- maGenControls(maGnames(swirl), id="Name")
table(maControls(swirl))
}
\keyword{manip}% at least one, from doc/KEYWORDS


\eof
\name{maGeneTable}
\alias{maGeneTable}
%- Also NEED an `\alias' for EACH other topic documented here.
\title{Table of spot coordinates and gene names}
\description{
This function produces a table of spot coordinates and gene names for objects of class \code{"\link{marrayRaw}"}, \code{"\link{marrayNorm}"}, and \code{"\link{marraySpots}"}.
}
\usage{
maGeneTable(object)
}
%- maybe also `usage' for other objects documented here.
\arguments{
  \item{object}{microarray object of class \code{"\link{marrayRaw}"}, \code{"\link{marrayNorm}"}, or \code{"\link{marraySpots}"}.}
}

\value{
  an object of class \code{\link{data.frame}}, with rows corresponding to spotted probe sequences. The first four columns are the grid matrix and spot matrix coordinates, and the remaining columns are the spot descriptions stored in the \code{maGnames} slot of the microarray object.
}

\author{Yee Hwa (Jean) Yang, Sandrine Dudoit, \url{http://www.stat.berkeley.edu/~sandrine}.}


\seealso{\code{\link{marrayInfo}}, \code{\link{marrayLayout}}, \code{\link{marrayRaw}}, \code{\link{marrayNorm}},  \code{\link{marraySpots}}, \code{\link{maCompCoord}}.}


\examples{
# Example uses swirl dataset, for description type ? swirl

data(swirl)

tab<-maGeneTable(swirl)
tab[1:10,]

}

\keyword{methods}% at least one, from doc/KEYWORDS


\eof
\name{maInd2Coord}
\alias{maInd2Coord}
%- Also NEED an `\alias' for EACH other topic documented here.
\title{Convert spot indices to grid and spot matrix coordinates}
\description{
This functions converts spot indices to grid and spot matrix coordinates (four
coordinates), where spots on the array are numbered consecutively starting
from the top left grid and the top left spot within each grid.
}
\usage{
maInd2Coord(x, L)
}
%- maybe also `usage' for other objects documented here.
\arguments{
  \item{x}{a numeric vector of spot indices.}
  \item{L}{an object of class \code{"\link{marrayLayout}"}.}
}
\value{
a matrix of spot four-coordinates, with rows corresponding to
  spots and columns to grid row, grid column, spot row, and spot column coordinates.
}

\author{Yee Hwa (Jean) Yang, Sandrine Dudoit, \url{http://www.stat.berkeley.edu/~sandrine}.}

\seealso{\code{\link{marrayLayout}}, \code{\link{maCoord2Ind}},
\code{\link{maCompCoord}}, \code{\link{maCompInd}}.}

\examples{

L <- new("marrayLayout", maNgr=4, maNgc=4, maNsr=22, maNsc=24)
maInd2Coord(c(1:10,529:538), L)


}
\keyword{manip}
\keyword{methods}

\eof
\name{maNum2Logic}
\alias{maNum2Logic}
%- Also NEED an `\alias' for EACH other topic documented here.
\title{Convert a numeric vector of indices to a logical vector}
\description{
This function converts a numeric vector of indices to a logical
vector. It is used for subsetting purposes.
}
\usage{
maNum2Logic(n=length(subset), subset=TRUE)
}
%- maybe also `usage' for other objects documented here.
\arguments{
  \item{n}{the length of the logical vector to be produced.}
  \item{subset}{a numeric vector of indices. A logical vector may also
  be supplied, in which case it is also the value of the function.}
}

\value{
 a logical vector.
}


\author{Sandrine Dudoit, \url{http://www.stat.berkeley.edu/~sandrine}.}


\examples{
maNum2Logic(10, 1:3)
}
\keyword{manip}% at least one, from doc/KEYWORDS

\eof
\name{marrayInfo-class}
\docType{class}

\alias{marrayInfo-class}
\alias{marrayInfo}

\alias{maLabels}
\alias{maLabels<-}
\alias{maInfo}
\alias{maInfo<-}
\alias{maNotes}
\alias{maNotes<-}

\alias{[,marrayInfo-method}
\alias{maGnames<-,marrayRaw,marrayInfo-method}
\alias{maGnames<-,marrayNorm,marrayInfo-method}
\alias{maGnames<-,marraySpots,marrayInfo-method}
\alias{maInfo<-,marrayInfo,data.frame-method}
\alias{maInfo,marrayInfo-method}
\alias{maLabels<-,marrayInfo,character-method}
\alias{maLabels<-,marrayInfo,numeric-method}
\alias{maLabels,marrayInfo-method}
\alias{maNotes<-,marrayInfo,character-method}
\alias{maNotes,marrayInfo-method}
\alias{maTargets<-,marrayRaw,marrayInfo-method}
\alias{maTargets<-,marrayNorm,marrayInfo-method}
\alias{maTargets<-,marrayTwo,marrayInfo-method}
\alias{print,marrayInfo-method}
\alias{rbind,marrayInfo-method}
\alias{show,marrayInfo-method}

\title{Class "marrayInfo", description of target samples or spotted probe sequences}

\description{This class is used to store information on target samples hybridized to a batch of arrays or probe sequences spotted onto these arrays. It is not specific to the microarray context.}


\section{Objects from the Class}{
Objects can be created by calls of the form \code{  new('marrayInfo',}\cr
\code{    maLabels = ...., # Object of class character}\cr
\code{    maInfo   = ...., # Object of class data.frame}\cr
\code{    maNotes  = ...., # Object of class character}\cr
\code{  )}
}

\section{Slots}{
  \describe{
    \item{\code{maLabels}:}{Object of class \code{"character"}, vector of spot or array labels.}
    \item{\code{maInfo}:}{Object of class \code{"data.frame"}. If the object of class \code{"marrayInfo"} is used to describe probe sequences, rows of \code{maInfo} correspond to spots and columns to various gene identifiers and annotations. 
If the object of class \code{"marrayInfo"} is used to describe target samples hybridized to the arrays, rows of \code{maInfo} correspond to arrays and columns to various descriptions of the hybridizations, e.g., names of Cy3 and Cy5 samples, labels for the arrays etc.}
    \item{\code{maNotes}:}{Object of class \code{"character"}, any notes on the target samples or spotted probe sequences.}
  }
}

\section{Methods}{
  \describe{
    \item{[}{\code{signature(x = "marrayInfo")}: subsetting operator for spots on the array or arrays in the batch, ensures that all slots are subset properly.}
    \item{maGnames<-}{\code{signature(object = "marrayRaw", value = "marrayInfo")}: slot assignment method.}
    \item{maGnames<-}{\code{signature(object = "marrayNorm", value = "marrayInfo")}: slot assignment method.}
    \item{maGnames<-}{\code{signature(object = "marraySpots", value = "marrayInfo")}: slot assignment method.}
    \item{maInfo}{\code{signature(object = "marrayInfo")}: slot accessor method.}
    \item{maInfo<-}{\code{signature(object = "marrayInfo", value = "data.frame")}: slot assignment method.}
    \item{maLabels}{\code{signature(object = "marrayInfo")}: slot accessor method.}
    \item{maLabels<-}{\code{signature(object = "marrayInfo", value = "character")}: slot assignment method.}
    \item{maLabels<-}{\code{signature(object = "marrayInfo", value = "numeric")}: slot assignment method.}
    \item{maNotes}{\code{signature(object = "marrayInfo")}: slot accessor method.}
    \item{maNotes<-}{\code{signature(object = "marrayInfo", value = "character")}: slot assignment method.}
    \item{maTargets<-}{\code{signature(object = "marrayRaw", value = "marrayInfo")}: slot assignment method.}
    \item{maTargets<-}{\code{signature(object = "marrayNorm", value = "marrayInfo")}: slot assignment method.}
    \item{maTargets<-}{\code{signature(object = "marrayTwo", value = "marrayInfo")}: slot assignment method.}
    \item{print}{\code{signature(x = "marrayInfo")}: print method for \code{"\link{marrayInfo}"} class.}
  }
}

\references{S. Dudoit and Y. H. Yang. (2002). Bioconductor R packages for exploratory analysis and normalization of cDNA microarray data. In G. Parmigiani, E. S. Garrett, R. A. Irizarry and S. L. Zeger, editors, \emph{The Analysis of Gene Expression Data: Methods and Software}, Springer, New York.
}


\author{Sandrine Dudoit, \url{http://www.stat.berkeley.edu/~sandrine}.}

\seealso{\code{\link{marrayLayout}}, \code{\link{marrayRaw}}, \code{\link{marrayNorm}}, \code{\link{marraySpots}}, \code{\link{marrayTwo}}, \code{\link{exprSet-class}}, \code{\link{print-methods}}, \code{\link{[-methods}}.}

\examples{
## See marrayRaw
}
\keyword{classes}

\eof
\name{marrayLayout-class}
\docType{class}

\alias{marrayLayout-class}
\alias{marrayLayout}

\alias{maNgr}
\alias{maNgr<-}
\alias{maNgc}
\alias{maNgc<-}
\alias{maNsr}
\alias{maNsr<-}
\alias{maNsc}
\alias{maNsc<-}
\alias{maNspots}
\alias{maNspots<-}
\alias{maSub}
\alias{maSub<-}
\alias{maPlate}
\alias{maPlate<-}
\alias{maControls}
\alias{maControls<-}

\alias{[,marrayLayout-method}
\alias{maControls<-,marrayLayout-method}
\alias{maControls,marrayLayout-method}
\alias{maGridCol,marrayLayout-method}
\alias{maGridRow,marrayLayout-method}
\alias{maLayout<-,marrayRaw,marrayLayout-method}
\alias{maLayout<-,marrayNorm,marrayLayout-method}
\alias{maLayout<-,marraySpots,marrayLayout-method}
\alias{maLayout<-,marrayTwo,marrayLayout-method}
\alias{maNgc<-,marrayLayout,numeric-method}
\alias{maNgc,marrayLayout-method}
\alias{maNgr<-,marrayLayout,numeric-method}
\alias{maNgr,marrayLayout-method}
\alias{maNotes<-,marrayLayout,character-method}
\alias{maNotes,marrayLayout-method}
\alias{maNsc<-,marrayLayout,numeric-method}
\alias{maNsc,marrayLayout-method}
\alias{maNspots<-,marrayLayout,numeric-method}
\alias{maNspots,marrayLayout-method}
\alias{maNsr<-,marrayLayout,numeric-method}
\alias{maNsr,marrayLayout-method}
\alias{maPlate<-,marrayLayout-method}
\alias{maPlate,marrayLayout-method}
\alias{maPrintTip,marrayLayout-method}
\alias{maSpotCol,marrayLayout-method}
\alias{maSpotRow,marrayLayout-method}
\alias{maSub<-,marrayLayout,logical-method}
\alias{maSub<-,marrayLayout,numeric-method}
\alias{maSub,marrayLayout-method}
\alias{print,marrayLayout-method}
\alias{show,marrayLayout-method}

%\alias{maNotes}

\alias{maPrintTip}
\alias{maGridRow}
\alias{maGridCol}
\alias{maSpotRow}
\alias{maSpotCol}


\title{Class "marrayLayout", classes and methods for layout parameters of cDNA microarrays}
\description{This class is used to keep track of important layout parameters for two-color cDNA microarrays. It contains slots for: the total number of spotted probe sequences on the array, the dimensions of the spot and grid matrices, the plate origin of the probes, information on spotted control sequences (e.g. probe sequences which should have equal abundance in the two target samples, such as housekeeping genes). The terms \emph{print-tip-group}, \emph{grid}, \emph{spot matrix}, and \emph{sector} are used interchangeably and refer to a set of spots printed using the same print-tip.}


\section{Objects from the Class}{
Objects can be created by calls of the form \code{  new('marrayLayout',}\cr
\code{    maNgr      = ...., # Object of class numeric}\cr
\code{    maNgc      = ...., # Object of class numeric}\cr
\code{    maNsr      = ...., # Object of class numeric}\cr
\code{    maNsc      = ...., # Object of class numeric}\cr
\code{    maNspots   = ...., # Object of class numeric}\cr
\code{    maSub      = ...., # Object of class logical}\cr
\code{    maPlate    = ...., # Object of class factor}\cr
\code{    maControls = ...., # Object of class factor}\cr
\code{    maNotes    = ...., # Object of class character}\cr
\code{  )}
}

\section{Slots}{
  \describe{
    \item{\code{maNgr}:}{Object of class \code{"numeric"}, number of rows for the grid matrix.}
    \item{\code{maNgc}:}{Object of class \code{"numeric"}, number of columns for the grid matrix.}
    \item{\code{maNsr}:}{Object of class \code{"numeric"}, number of rows for the spot matrices.}
    \item{\code{maNsc}:}{Object of class \code{"numeric"}, number of columns for the spot matrices.}
    \item{\code{maNspots}:}{Object of class \code{"numeric"}, total number of spots on the array, equal to \eqn{maNgr x maNgc x maNsr x maNsc}.}
    \item{\code{maSub}:}{Object of class \code{"logical"},  indicating which spots are currently being considered.}
    \item{\code{maPlate}:}{Object of class \code{"factor"}, recording the plate origin of the spotted probe sequences.}
    \item{\code{maControls}:}{Object of class \code{"factor"}, recording the control status of the spotted probe sequences.}
    \item{\code{maNotes}:}{Object of class \code{"character"}, any notes concerning the microarray layout, e.g., printing conditions.}
  }
}

\section{Methods}{
  \describe{
    \item{[}{\code{signature(x = "marrayLayout")}: subsetting operator for spots on the array, ensures that all slots are subset properly.}
    \item{maControls<-}{\code{signature(object = "marrayLayout")}: slot assignment method.} 
    \item{maControls}{\code{signature(object = "marrayLayout")}: slot accessor method.}
    \item{maGridCol}{\code{signature(object = "marrayLayout")}: method which computes a vector of grid column coordinates for each spot.}
    \item{maGridRow}{\code{signature(object = "marrayLayout")}: method which computes a vector of grid row coordinates for each spot.}
    \item{maLayout<-}{\code{signature(object = "marrayRaw", value = "marrayLayout")}: slot assignment method.} 
    \item{maLayout<-}{\code{signature(object = "marrayNorm", value = "marrayLayout")}: slot assignment method.} 
    \item{maLayout<-}{\code{signature(object = "marrayTwo", value = "marrayLayout")}: slot assignment method.} 
    \item{maLayout<-}{\code{signature(object = "marraySpots", value = "marrayLayout")}: slot assignment method.} 
    \item{maNgc}{\code{signature(object = "marrayLayout")}: slot accessor method.}
 \item{maNgc<-}{\code{signature(object = "marrayLayout", value = "numeric")}: slot assignment method.}
    \item{maNgr}{\code{signature(object = "marrayLayout")}: slot accessor method.}
\item{maNgr<-}{\code{signature(object = "marrayLayout", value = "numeric")}: slot assignment method.}
    \item{maNotes}{\code{signature(object = "marrayLayout")}: slot accessor method.}
    \item{maNotes<-}{\code{signature(object = "marrayLayout", value = "character")}: slot assignment method.} 
    \item{maNsc}{\code{signature(object = "marrayLayout")}: slot accessor method.}
    \item{maNsc<-}{\code{signature(object = "marrayLayout", value = "numeric")}: slot assignment method.}
    \item{maNspots}{\code{signature(object = "marrayLayout")}: slot accessor method.}
 \item{maNspots<-}{\code{signature(object = "marrayLayout", value = "numeric")}: slot assignment method.}
    \item{maNsr}{\code{signature(object = "marrayLayout")}: slot accessor method.}
\item{maNsr<-}{\code{signature(object = "marrayLayout", value = "numeric")}: slot assignment method.}
    \item{maPlate}{\code{signature(object = "marrayLayout")}: slot accessor method.}
    \item{maPlate<-}{\code{signature(object = "marrayLayout")}: slot assignment method.} 
    \item{maPrintTip}{\code{signature(object = "marrayLayout")}: method which computes a vector of print-tip-group indices for each spot.}
    \item{maSpotCol}{\code{signature(object = "marrayLayout")}: method which computes a vector of spot column coordinates for each spot.}
    \item{maSpotRow}{\code{signature(object = "marrayLayout")}: method which computes a vector of spot row coordinates for each spot.}
    \item{maSub}{\code{signature(object = "marrayLayout")}: slot accessor method.}
    \item{maSub<-}{\code{signature(object = "marrayLayout", value = "logical")}: slot assignment method.} 
    \item{maSub<-}{\code{signature(object = "marrayLayout", value = "numeric")}: slot assignment method.} 
    \item{print}{\code{signature(x = "marrayLayout")}: print method for \code{"\link{marrayLayout}"} class.}
  }
}

\references{S. Dudoit and Y. H. Yang. (2002). Bioconductor R packages for exploratory analysis and normalization of cDNA microarray data. In G. Parmigiani, E. S. Garrett, R. A. Irizarry and S. L. Zeger, editors, \emph{The Analysis of Gene Expression Data: Methods and Software}, Springer, New York.
}

\author{Sandrine Dudoit, \url{http://www.stat.berkeley.edu/~sandrine}.}

\seealso{\code{\link{marrayRaw}}, \code{\link{marrayNorm}}, \code{\link{marrayInfo}},   \code{\link{marraySpots}}, \code{\link{marrayTwo}}, \code{\link{print-methods}}, \code{\link{[-methods}}.}

\examples{
## See marrayRaw

}

\keyword{classes}

\eof
\name{marrayNorm-class}
\docType{class}

\alias{marrayNorm-class}
\alias{marrayNorm}

\alias{maA}
\alias{maA<-}
\alias{maM}
\alias{maM<-}
\alias{maMloc}
\alias{maMloc<-}
\alias{maMscale}
\alias{maMscale<-}

\alias{[,marrayNorm-method}
\alias{cbind,marrayNorm-method}
\alias{coerce,marrayRaw,marrayNorm-method}
\alias{coerce,marrayNorm,exprSet-method}
\alias{coerce,marrayNorm,marraySpots-method}
\alias{maA<-,marrayNorm,matrix-method}
\alias{maA,marrayNorm-method}
\alias{maControls<-,marrayNorm-method}
\alias{maControls,marrayNorm-method}
%\alias{maGnames<-,marrayNorm,marrayInfo-method}
\alias{maGnames,marrayNorm-method}
\alias{maGridCol,marrayNorm-method}
\alias{maGridRow,marrayNorm-method}
%\alias{maLayout<-,marrayNorm,marrayLayout-method}
\alias{maLayout,marrayNorm-method}
\alias{maMloc<-,marrayNorm,matrix-method}
\alias{maMloc,marrayNorm-method}
\alias{maM<-,marrayNorm,matrix-method}
\alias{maM,marrayNorm-method}
\alias{maMscale<-,marrayNorm,matrix-method}
\alias{maMscale,marrayNorm-method}
\alias{maNgc<-,marrayNorm,numeric-method}
\alias{maNgc,marrayNorm-method}
\alias{maNgr<-,marrayNorm,numeric-method}
\alias{maNgr,marrayNorm-method}
\alias{maNormCall,marrayNorm-method}
\alias{maNotes<-,marrayNorm,character-method}
\alias{maNotes,marrayNorm-method}
\alias{maNsamples,marrayNorm-method}
\alias{maNsc<-,marrayNorm,numeric-method}
\alias{maNsc,marrayNorm-method}
\alias{maNspots<-,marrayNorm,numeric-method}
\alias{maNspots,marrayNorm-method}
\alias{maNsr<-,marrayNorm,numeric-method}
\alias{maNsr,marrayNorm-method}
\alias{maPlate<-,marrayNorm-method}
\alias{maPlate,marrayNorm-method}
\alias{maPrintTip,marrayNorm-method}
\alias{maSpotCol,marrayNorm-method}
\alias{maSpotRow,marrayNorm-method}
\alias{maSub<-,marrayNorm-method}
\alias{maSub,marrayNorm-method}
%\alias{maTargets<-,marrayNorm,marrayInfo-method}
\alias{maTargets,marrayNorm-method}
\alias{maW<-,marrayNorm,matrix-method}
\alias{maW,marrayNorm-method}
\alias{print,marrayNorm-method}
\alias{show,marrayNorm-method}

%\alias{maW}

%\alias{maLayout}
%\alias{maNgr}
%\alias{maNgc}
%\alias{maNsr}
%\alias{maNsc}
%\alias{maNspots}
%\alias{maSub}
%\alias{maPlate}
%\alias{maControls}

%\alias{maPrintTip}
%\alias{maGridRow}
%\alias{maGridCol}
%\alias{maSpotRow}
%\alias{maSpotCol}

%\alias{maGnames}
%\alias{maTargets}
%\alias{maNotes}
\alias{maNormCall}

\title{Class "marrayNorm", classes and methods for post-normalization cDNA microarray intensity data}


\description{
This class represents post-normalization intensity data for a batch of cDNA microarrays. A \emph{batch of arrays} consists of a collection of arrays with the same layout  (\code{"\link{marrayLayout}"}). The class contains slots for the average log-intensities A, the normalized log-ratios M, the location and scale normalization values, the layout of the arrays, and descriptions of the target samples hybridized to the arrays and probe sequences spotted onto the arrays.}


\section{Objects from the Class}{
Objects can be created by calls of the form 
\code{  new('marrayNorm',}\cr
\code{    maA        = ...., # Object of class matrix}\cr
\code{    maM        = ...., # Object of class matrix}\cr
\code{    maMloc     = ...., # Object of class matrix}\cr
\code{    maMscale   = ...., # Object of class matrix}\cr
\code{    maW        = ...., # Object of class matrix}\cr
\code{    maLayout   = ...., # Object of class marrayLayout}\cr
\code{    maGnames   = ...., # Object of class marrayInfo}\cr
\code{    maTargets  = ...., # Object of class marrayInfo}\cr
\code{    maNotes    = ...., # Object of class character}\cr
\code{    maNormCall = ...., # Object of class call}\cr
\code{  )}
}


\section{Slots}{
  \describe{
    \item{\code{maA}:}{Object of class \code{"matrix"}, average log-intensities (base 2) A, rows correspond to spotted probe sequences, columns to arrays in the batch.}
    \item{\code{maM}:}{Object of class \code{"matrix"}, intensity log-ratios (base 2) M, rows correspond to spotted probe sequences, columns to arrays in the batch.}
    \item{\code{maMloc}:}{Object of class \code{"matrix"}, location normalization values, rows correspond to spotted probe sequences, columns to arrays in the batch.}
    \item{\code{maMscale}:}{Object of class \code{"matrix"},  scale normalization values, rows correspond to spotted probe sequences, columns to arrays in the batch.}
    \item{\code{maW}:}{Object of class \code{"matrix"}, spot quality weights, rows correspond to spotted probe sequences, columns to arrays in the batch.}
    \item{\code{maLayout}:}{Object of class \code{"\link{marrayLayout}"}, layout parameters for cDNA microarrays.}
    \item{\code{maGnames}:}{Object of class \code{"\link{marrayInfo}"}, description of spotted probe sequences.}
    \item{\code{maTargets}:}{Object of class \code{"\link{marrayInfo}"}, description of target samples hybridized to the arrays.}
    \item{\code{maNotes}:}{Object of class \code{"character"}, any notes concerning the microarray experiments, e.g. hybridization or scanning conditions.}
    \item{\code{maNormCall}:}{Object of class \code{"call"}, function call for normalizing the batch of arrays.}
  }
}

\section{Methods}{
  \describe{
    \item{[}{\code{signature(x = "marrayNorm")}:  subsetting operator for spots on the array and arrays in the batch, ensures that all slots are subset properly.}
    \item{coerce}{\code{signature(from = "marrayRaw", to = "marrayNorm")}: coerce an object of class \code{"\link{marrayRaw}"} into an object of class \code{\link{marrayNorm}}. }
    \item{coerce}{\code{signature(from = "marrayNorm", to = "exprSet")}: coerce an object of class \code{"\link{marrayNorm}"} into an object of class \code{\link{exprSet-class}}. }
 \item{coerce}{\code{signature(from = "marrayNorm", to = "marraySpots")}: coerce an object of class \code{"\link{marrayNorm}"} into an object of class \code{"\link{marraySpots}"}. }
    \item{maA}{\code{signature(object = "marrayNorm")}: slot accessor method.}
    \item{maA<-}{\code{signature(object = "marrayNorm", value = "matrix")}: slot assignment method.} 
    \item{maControls<-}{\code{signature(object = "marrayNorm")}: slot assignment method.} 
    \item{maControls}{\code{signature(object = "marrayNorm")}: slot accessor method.}
    \item{maGnames}{\code{signature(object = "marrayNorm")}: slot accessor method.}
    \item{maGnames<-}{\code{signature(object = "marrayNorm", value = "marrayInfo")}: slot assignment method.} 
    \item{maGridCol}{\code{signature(object = "marrayNorm")}: method which computes a vector of grid column coordinates for each spot.}
    \item{maGridRow}{\code{signature(object = "marrayNorm")}: method which computes a vector of grid row coordinates for each spot.}
    \item{maLayout}{\code{signature(object = "marrayNorm")}: slot accessor method.}
    \item{maLayout<-}{\code{signature(object = "marrayNorm", value = "marrayLayout")}: slot assignment method.} 
    \item{maM}{\code{signature(object = "marrayNorm")}: slot accessor method.}
    \item{maM<-}{\code{signature(object = "marrayNorm", value = "matrix")}: slot assignment method.} 
    \item{maMloc}{\code{signature(object = "marrayNorm")}: slot accessor method.}
    \item{maMloc<-}{\code{signature(object = "marrayNorm", value = "matrix")}: slot assignment method.} 
    \item{maMscale}{\code{signature(object = "marrayNorm")}: slot accessor method.}
    \item{maMscale<-}{\code{signature(object = "marrayNorm", value = "matrix")}: slot assignment method.} 
    \item{maNgc}{\code{signature(object = "marrayNorm")}: slot accessor method.}
 \item{maNgc<-}{\code{signature(object = "marrayNorm", value = "numeric")}: slot assignment method.} 
    \item{maNgr}{\code{signature(object = "marrayNorm")}: slot accessor method.}
\item{maNgr<-}{\code{signature(object = "marrayNorm", value = "numeric")}: slot assignment method.} 
    \item{maNormCall}{\code{signature(object = "marrayNorm")}: slot accessor method.}
    \item{maNotes}{\code{signature(object = "marrayNorm")}: slot accessor method.}
    \item{maNotes<-}{\code{signature(object = "marrayNorm", value = "character")}: slot assignment method.} 
    \item{maNsamples}{\code{signature(object = "marrayNorm")}: slot accessor method.}
    \item{maNsc}{\code{signature(object = "marrayNorm")}: slot accessor method.}
\item{maNsc<-}{\code{signature(object = "marrayNorm", value = "numeric")}: slot assignment method.} 
    \item{maNspots}{\code{signature(object = "marrayNorm")}: slot accessor method.}
\item{maNspots<-}{\code{signature(object = "marrayNorm", value = "numeric")}: slot assignment method.} 
    \item{maNsr}{\code{signature(object = "marrayNorm")}: slot accessor method.}
\item{maNsr<-}{\code{signature(object = "marrayNorm", value = "numeric")}: slot assignment method.} 
    \item{maPlate}{\code{signature(object = "marrayNorm")}: slot accessor method.}
    \item{maPlate<-}{\code{signature(object = "marrayNorm")}: slot assignment method.} 
    \item{maPrintTip}{\code{signature(object = "marrayNorm")}: method which computes a vector of print-tip-group indices for each spot.}
    \item{maSpotCol}{\code{signature(object = "marrayNorm")}: method which computes a vector of spot column coordinates for each spot.}
    \item{maSpotRow}{\code{signature(object = "marrayNorm")}: method which computes a vector of spot row coordinates for each spot.}
    \item{maSub}{\code{signature(object = "marrayNorm")}: slot accessor method.}
    \item{maSub<-}{\code{signature(object = "marrayNorm")}: slot assignment method.} 
    \item{maTargets}{\code{signature(object = "marrayNorm")}: slot accessor method.}
    \item{maTargets<-}{\code{signature(object = "marrayNorm", value = "marrayInfo")}: slot assignment method.}  
    \item{maW}{\code{signature(object = "marrayNorm")}: slot accessor method.}
    \item{maW<-}{\code{signature(object = "marrayNorm", value = "matrix")}: slot assignment method.} 
    \item{print}{\code{signature(x = "marrayNorm")}: print method for \code{"\link{marrayNorm}"} class.}
  }
}

\references{S. Dudoit and Y. H. Yang. (2002). Bioconductor R packages for exploratory analysis and normalization of cDNA microarray data. In G. Parmigiani, E. S. Garrett, R. A. Irizarry and S. L. Zeger, editors, \emph{The Analysis of Gene Expression Data: Methods and Software}, Springer, New York.
}

\author{Sandrine Dudoit, \url{http://www.stat.berkeley.edu/~sandrine}.}

\seealso{\code{\link{marrayLayout}}, \code{\link{marrayRaw}}, \code{\link{marrayInfo}}, \code{\link{marraySpots}}, \code{\link{marrayTwo}}, \code{\link{exprSet-class}}, \code{\link{print-methods}}, \code{\link{[-methods}}, \code{\link{coerce-methods}}.}

\examples{
# Examples use swirl dataset, for description type ? swirl

# data(swirl)

# Normalization is done using functions from the marrayNorm package
# library(marrayNorm)

# Within-print-tip-group loess location normalization
# mnorm<-maNorm(swirl[,2:3],norm="p")

# Object of class marrayNorm for the second and third swirl arrays
# mnorm

# Function call
# maNormCall(mnorm)

#  Object of class marrayInfo -- Probe sequences
# maGnames(mnorm)

#  Object of class marrayInfo -- Target samples
# maTargets(mnorm)

# Density plot of log-ratios M for third array
# plot(density(maM(mnorm[,2])), lwd=2, col=2, main="Density plots of log-ratios M")
# lines(density(maM(swirl[,3])), lwd=2)
# abline(v=0)
# legend(2,1,c("Pre-normalization","Post-normalization"))


}

\keyword{classes}

\eof
\name{marrayRaw-class}
\docType{class}

\alias{marrayRaw-class}
\alias{marrayRaw}

\alias{maRf}
\alias{maRf<-}
\alias{maGf}
\alias{maGf<-}
\alias{maRb}
\alias{maRb<-}
\alias{maGb}
\alias{maGb<-}
\alias{maW}
\alias{maW<-}
\alias{maLayout}
\alias{maLayout<-}

\alias{[,marrayRaw-method}
\alias{cbind,marrayRaw-method}
%\alias{coerce,marrayRaw,marrayNorm-method}
\alias{coerce,marrayRaw,exprSet-method}
\alias{coerce,marrayRaw,marraySpots-method}
\alias{maA,marrayRaw-method}
\alias{maControls<-,marrayRaw-method}
\alias{maControls,marrayRaw-method}
\alias{maGb<-,marrayRaw,matrix-method}
\alias{maGb<-,marrayRaw,NULL-method}
\alias{maGb,marrayRaw-method}
\alias{maGf<-,marrayRaw,matrix-method}
\alias{maGf,marrayRaw-method}
%\alias{maGnames<-,marrayRaw,marrayInfo-method}
\alias{maGnames,marrayRaw-method}
\alias{maGridCol,marrayRaw-method}
\alias{maGridRow,marrayRaw-method}
%\alias{maLayout<-,marrayRaw,marrayLayout-method}
\alias{maLayout,marrayRaw-method}
\alias{maLG,marrayRaw-method}
\alias{maLR,marrayRaw-method}
\alias{maM,marrayRaw-method}
\alias{maNgc<-,marrayRaw,numeric-method}
\alias{maNgc,marrayRaw-method}
\alias{maNgr<-,marrayRaw,numeric-method}
\alias{maNgr,marrayRaw-method}
\alias{maNotes<-,marrayRaw,character-method}
\alias{maNotes,marrayRaw-method}
\alias{maNsamples,marrayRaw-method}
\alias{maNsc<-,marrayRaw,numeric-method}
\alias{maNsc,marrayRaw-method}
\alias{maNspots<-,marrayRaw,numeric-method}
\alias{maNspots,marrayRaw-method}
\alias{maNsr<-,marrayRaw,numeric-method}
\alias{maNsr,marrayRaw-method}
\alias{maPlate<-,marrayRaw-method}
\alias{maPlate,marrayRaw-method}
\alias{maPrintTip,marrayRaw-method}
\alias{maRb<-,marrayRaw,matrix-method}
\alias{maRb<-,marrayRaw,NULL-method}
\alias{maRb,marrayRaw-method}
\alias{maRf<-,marrayRaw,matrix-method}
\alias{maRf,marrayRaw-method}
\alias{maSpotCol,marrayRaw-method}
\alias{maSpotRow,marrayRaw-method}
\alias{maSub<-,marrayRaw-method}
\alias{maSub,marrayRaw-method}
%\alias{maTargets<-,marrayRaw,marrayInfo-method}
\alias{maTargets,marrayRaw-method}
\alias{maW<-,marrayRaw,matrix-method}
\alias{maW,marrayRaw-method}
\alias{print,marrayRaw-method}
\alias{show,marrayRaw-method}

%\alias{maNgr}
%\alias{maNgc}
%\alias{maNsr}
%\alias{maNsc}
%\alias{maNspots}
%\alias{maSub}
%\alias{maPlate}
%\alias{maControls}

%\alias{maPrintTip}
%\alias{maGridRow}
%\alias{maGridCol
%\alias{maSpotRow}
%\alias{maSpotCol}

\alias{maGnames}
\alias{maGnames<-}
\alias{maTargets}
\alias{maTargets<-}
%\alias{maNotes}

\alias{maLR}
\alias{maLG}

%\alias{maA}
%\alias{maM}

\alias{maNsamples}


\title{Class "marrayRaw", classes and methods for pre-normalization cDNA microarray intensity data}

\description{This class represents pre-normalization intensity data for a batch of cDNA microarrays. A \emph{batch of arrays} consists of a collection of arrays with the same layout  (\code{"\link{marrayLayout}"}). The class contains slots for the green (Cy3) and red (Cy5) foreground and background intensities, the layout of the arrays, and descriptions of the target samples hybridized to the arrays and probe sequences spotted onto the arrays.}

\section{Objects from the Class}{
Objects can be created by calls of the form 
\code{  new('marrayRaw',}\cr
\code{    maRf      = ...., # Object of class matrix}\cr
\code{    maGf      = ...., # Object of class matrix}\cr
\code{    maRb      = ...., # Object of class matrix}\cr
\code{    maGb      = ...., # Object of class matrix}\cr
\code{    maW       = ...., # Object of class matrix}\cr
\code{    maLayout  = ...., # Object of class marrayLayout}\cr
\code{    maGnames  = ...., # Object of class marrayInfo}\cr
\code{    maTargets = ...., # Object of class marrayInfo}\cr
\code{    maNotes   = ...., # Object of class character}\cr
\code{  )}
}

\section{Slots}{
  \describe{
    \item{\code{maRf}:}{Object of class \code{"matrix"}, red foreground intensities, rows correspond to spotted probe sequences, columns to arrays in the batch.}
    \item{\code{maGf}:}{Object of class \code{"matrix"}, green foreground intensities, rows correspond to spotted probe sequences, columns to arrays in the batch. }
    \item{\code{maRb}:}{Object of class \code{"matrix"}, red background intensities, rows correspond to spotted probe sequences, columns to arrays in the batch. }
    \item{\code{maGb}:}{Object of class \code{"matrix"}, green background intensities, rows correspond to spotted probe sequences, columns to arrays in the batch. }
    \item{\code{maW}:}{Object of class \code{"matrix"}, spot quality weights, rows correspond to spotted probe sequences, columns to arrays in the batch.}
    \item{\code{maLayout}:}{Object of class \code{"\link{marrayLayout}"}, layout parameters for the cDNA microarrays.}
    \item{\code{maGnames}:}{Object of class \code{"\link{marrayInfo}"}, description of spotted probe sequences.}
    \item{\code{maTargets}:}{Object of class \code{"\link{marrayInfo}"},  description of target samples hybridized to the arrays.}
    \item{\code{maNotes}:}{Object of class \code{"character"},  any notes concerning the microarray experiments, e.g. hybridization or scanning conditions.}
  }
}

\section{Methods}{
  \describe{
    \item{[}{\code{signature(x = "marrayRaw")}: subsetting operator for spots on the array and arrays in the batch, ensures that all slots are subset properly.}
    \item{coerce}{\code{signature(from = "marrayRaw", to = "marrayNorm")}: coerce an object of class \code{"\link{marrayRaw}"} into an object of class \code{"\link{marrayNorm}"}. }
    \item{coerce}{\code{signature(from = "marrayRaw", to = "exprSet")}:  coerce an object of class \code{"\link{marrayRaw}"} into an object of class \code{"\link{exprSet-class}"}. }
\item{coerce}{\code{signature(from = "marrayRaw", to = "marraySpots")}:  coerce an object of class \code{"\link{marrayRaw}"} into an object of class \code{"\link{marraySpots}"}. }
    \item{maA}{\code{signature(object = "marrayRaw")}: function which computes average log-intensities (base 2) A for an object of class \code{"\link{marrayRaw}"}.}
    \item{maControls<-}{\code{signature(object = "marrayRaw")}: slot assignment method.} 
    \item{maControls}{\code{signature(object = "marrayRaw")}: slot accessor method.}
    \item{maGb}{\code{signature(object = "marrayRaw")}: slot accessor method.}
    \item{maGb<-}{\code{signature(object = "marrayRaw", value = "matrix")}: slot assignment method.} 
    \item{maGb<-}{\code{signature(object = "marrayRaw", value = "NULL")}: slot assignment method.} 
    \item{maGf}{\code{signature(object = "marrayRaw")}: slot accessor method.}
    \item{maGf<-}{\code{signature(object = "marrayRaw", value = "matrix")}: slot assignment method.}
    \item{maGnames}{\code{signature(object = "marrayRaw")}: slot accessor method.}
    \item{maGnames<-}{\code{signature(object = "marrayRaw", value = "marrayInfo")}: slot assignment method.}
    \item{maGridCol}{\code{signature(object = "marrayRaw")}: method which computes a vector of grid column coordinates for each spot.}
    \item{maGridRow}{\code{signature(object = "marrayRaw")}: method which computes a vector of grid row coordinates for each spot.}
    \item{maLayout}{\code{signature(object = "marrayRaw")}: slot accessor method.}
    \item{maLayout<-}{\code{signature(object = "marrayRaw", value = "marrayLayout")}: slot assignment method. }
    \item{maLG}{\code{signature(object = "marrayRaw")}: method which computes green log-intensities (base 2) for an object of class \code{"\link{marrayRaw}"}.}
    \item{maLR}{\code{signature(object = "marrayRaw")}: method  which computes red log-intensities (base 2) for an object of class \code{"\link{marrayRaw}"}.}
    \item{maM}{\code{signature(object = "marrayRaw")}: method which computes intensity log-ratios (base 2) M for an object of class \code{"\link{marrayRaw}"}.}
    \item{maNgc}{\code{signature(object = "marrayRaw")}: slot accessor method.}
 \item{maNgc<-}{\code{signature(object = "marrayRaw", value = "numeric")}:  slot assignment method.}
    \item{maNgr}{\code{signature(object = "marrayRaw")}: slot accessor method.}
 \item{maNgr<-}{\code{signature(object = "marrayRaw", value = "numeric")}:  slot assignment method.}
    \item{maNotes}{\code{signature(object = "marrayRaw")}: slot accessor method.}
    \item{maNotes<-}{\code{signature(object = "marrayRaw", value = "character")}: slot assignment method.}
    \item{maNsamples}{\code{signature(object = "marrayRaw")}: slot accessor method.}
    \item{maNsc}{\code{signature(object = "marrayRaw")}: slot accessor method.}
\item{maNsc<-}{\code{signature(object = "marrayRaw", value = "numeric")}:  slot assignment method.}
    \item{maNspots}{\code{signature(object = "marrayRaw")}: slot accessor method.}
\item{maNspots<-}{\code{signature(object = "marrayRaw", value = "numeric")}:  slot assignment method.}
    \item{maNsr}{\code{signature(object = "marrayRaw")}: slot accessor method.}
\item{maNsr<-}{\code{signature(object = "marrayRaw", value = "numeric")}:  slot assignment method.}
    \item{maPlate}{\code{signature(object = "marrayRaw")}: slot accessor method.}
    \item{maPlate<-}{\code{signature(object = "marrayRaw")}: slot assignment method.}
    \item{maPrintTip}{\code{signature(object = "marrayRaw")}: method which computes a vector of print-tip-group indices for each spot.}
    \item{maRb}{\code{signature(object = "marrayRaw")}: slot accessor method.}
    \item{maRb<-}{\code{signature(object = "marrayRaw", value = "matrix")}: slot assignment method.}
    \item{maRb<-}{\code{signature(object = "marrayRaw", value = "NULL")}: slot assignment method. }
    \item{maRf}{\code{signature(object = "marrayRaw")}: slot accessor method. }
    \item{maRf<-}{\code{signature(object = "marrayRaw", value = "matrix")}: slot assignment method. }
    \item{maSpotCol}{\code{signature(object = "marrayRaw")}: method which computes a vector of spot column coordinates for each spot.}
    \item{maSpotRow}{\code{signature(object = "marrayRaw")}: method which computes a vector of spot row coordinates for each spot.}
    \item{maSub}{\code{signature(object = "marrayRaw")}: slot accessor method.}
    \item{maSub<-}{\code{signature(object = "marrayRaw")}: slot assignment method.}
    \item{maTargets}{\code{signature(object = "marrayRaw")}: slot accessor method.}
    \item{maTargets<-}{\code{signature(object = "marrayRaw", value = "marrayInfo")}: slot assignment method. }
    \item{maW}{\code{signature(object = "marrayRaw")}: slot accessor method.}
    \item{maW<-}{\code{signature(object = "marrayRaw", value = "matrix")}: slot assignment method.}
    \item{print}{\code{signature(x = "marrayRaw")}:  print method for \code{"\link{marrayRaw}"} class.}
  }
}

\references{S. Dudoit and Y. H. Yang. (2002). Bioconductor R packages for exploratory analysis and normalization of cDNA microarray data. In G. Parmigiani, E. S. Garrett, R. A. Irizarry and S. L. Zeger, editors, \emph{The Analysis of Gene Expression Data: Methods and Software}, Springer, New York.
}

\author{Sandrine Dudoit, \url{http://www.stat.berkeley.edu/~sandrine}.}

\seealso{\code{\link{marrayLayout}}, \code{\link{marrayNorm}}, \code{\link{marrayInfo}}, \code{\link{marraySpots}}, \code{\link{marrayTwo}}, \code{\link{exprSet-class}}, \code{\link{print-methods}}, \code{\link{[-methods}}, \code{\link{coerce-methods}}.}


\examples{
# Examples use swirl dataset, for description type ? swirl

data(swirl)

# Object of class marrayRaw for the 4 swirl arrays
swirl

# Object of class marrayLayout
maLayout(swirl)

# Access only the first 100 spots of the third array
swirl[1:100,3]

# Accessor methods -- How many spots on the array
maNspots(swirl)

# Density plot of log-ratios M for third array
plot(density(maM(swirl[,3])))

# Assignment methods -- Replace maNotes slot
maNotes(swirl)
maNotes(swirl)<-"This is a zebrafish microarray"
maNotes(swirl)

# Coerce methods -- Convert marrayRaw object into exprSet
e<-as(swirl, "exprSet")
e

}
\keyword{classes}

\eof
\name{marraySpots-class}
\docType{class}

\alias{marraySpots-class}
\alias{marraySpots}

\alias{[,marraySpots-method}
%\alias{coerce,marrayRaw,marraySpots-method}
%\alias{coerce,marrayNorm,marraySpots-method}
\alias{maControls<-,marraySpots-method}
\alias{maControls,marraySpots-method}
%\alias{maGnames<-,marraySpots,marrayInfo-method}
\alias{maGnames,marraySpots-method}
\alias{maGridCol,marraySpots-method}
\alias{maGridRow,marraySpots-method}
%\alias{maLayout<-,marraySpots,marrayLayout-method}
\alias{maLayout,marraySpots-method}
\alias{maNgc<-,marraySpots,numeric-method}
\alias{maNgc,marraySpots-method}
\alias{maNgr<-,marraySpots,numeric-method}
\alias{maNgr,marraySpots-method}
\alias{maNsc<-,marraySpots,numeric-method}
\alias{maNsc,marraySpots-method}
\alias{maNspots<-,marraySpots,numeric-method}
\alias{maNspots,marraySpots-method}
\alias{maNsr<-,marraySpots,numeric-method}
\alias{maNsr,marraySpots-method}
\alias{maPlate<-,marraySpots-method}
\alias{maPlate,marraySpots-method}
\alias{maPrintTip,marraySpots-method}
\alias{maSpotCol,marraySpots-method}
\alias{maSpotRow,marraySpots-method}
\alias{maSub<-,marraySpots-method}
\alias{maSub,marraySpots-method}
\alias{print,marraySpots-method}
\alias{show,marraySpots-method}

%\alias{maGnames}

%\alias{maLayout}
%\alias{maNgr}
%\alias{maNgc}
%\alias{maNsr}
%\alias{maNsc}
%\alias{maNspots}
%\alias{maSub}
%\alias{maPlate}
%\alias{maControls}

\title{Class "marraySpots", classes and methods for probe sequences spotted on microarrays}

\description{This class stores information on the spotted probe sequences for a batch of arrays.
 A \emph{batch of arrays} consists of a collection of arrays with the same layout  (\code{"\link{marrayLayout}"}). The class contains slots for the layout of the arrays, and a description of the probe sequences spotted onto the arrays.}


\section{Objects from the Class}{
Objects can be created by calls of the form 
\code{  new('marraySpots',}\cr
\code{    maGnames = ...., # Object of class marrayInfo}\cr
\code{    maLayout  = ...., # Object of class marrayLayout}\cr
}
\section{Slots}{
  \describe{
    \item{\code{maGnames}:}{Object of class \code{"\link{marrayInfo}"},  description of spotted probe sequences.}
    \item{\code{maLayout}:}{Object of class \code{"\link{marrayLayout}"}, layout parameters for the cDNA microarrays.}
  }
}

\section{Methods}{
  \describe{
    \item{[}{\code{signature(x = "marraySpots")}:  subsetting operator for spots on the array, ensures that all slots are subset properly.}
\item{coerce}{\code{signature(from = "marrayRaw", to = "marraySpots")}: coerce an object of class \code{"\link{marrayRaw}"} into an object of class \code{"\link{marraySpots}"}. }
    \item{coerce}{\code{signature(from = "marrayNorm", to = "marraySpots")}: coerce an object of class \code{"\link{marrayNorm}"} into an object of class \code{"\link{marraySpots}"}. } 
   \item{maControls}{\code{signature(object = "marraySpots")}: slot accessor method.}
    \item{maControls<-}{\code{signature(object = "marraySpots")}:  slot assignment method.} 
    \item{maGnames}{\code{signature(object = "marraySpots")}: slot accessor method.}
    \item{maGnames<-}{\code{signature(object = "marraySpots", value = "marrayInfo")}: slot assignment method.} 
\item{maGridCol}{\code{signature(object = "marraySpots")}:  method which computes a vector of grid column coordinates for each spot.}
    \item{maGridRow}{\code{signature(object = "marraySpots")}: method which computes a vector of grid row coordinates for each spot.}
    \item{maLayout}{\code{signature(object = "marraySpots")}:  slot accessor method.}
    \item{maLayout<-}{\code{signature(object = "marraySpots", value = "marrayLayout")}: slot assignment method.} 
 \item{maNgc}{\code{signature(object = "marraySpots")}: slot accessor method.}
    \item{maNgc<-}{\code{signature(object = "marraySpots", value = "numeric")}: slot assignment method.} 
    \item{maNgr}{\code{signature(object = "marraySpots")}: slot accessor method.}
    \item{maNgr<-}{\code{signature(object = "marraySpots", value = "numeric")}: slot assignment method.} 
    \item{maNsc}{\code{signature(object = "marraySpots")}: slot accessor method.}
    \item{maNsc<-}{\code{signature(object = "marraySpots", value = "numeric")}: slot assignment method.} 
    \item{maNspots}{\code{signature(object = "marraySpots")}: slot accessor method.}
    \item{maNspots<-}{\code{signature(object = "marraySpots", value = "numeric")}: slot assignment method.} 
    \item{maNsr}{\code{signature(object = "marraySpots")}: slot accessor method.}
    \item{maNsr<-}{\code{signature(object = "marraySpots", value = "numeric")}: slot assignment method.} 
    \item{maPlate}{\code{signature(object = "marraySpots")}: slot accessor method.}
    \item{maPlate<-}{\code{signature(object = "marraySpots")}: slot assignment method. }
    \item{maPrintTip}{\code{signature(object = "marraySpots")}: method which computes a vector of print-tip-group indices for each spot.}
    \item{maSpotCol}{\code{signature(object = "marraySpots")}: method which computes a vector of spot column coordinates for each spot.}
    \item{maSpotRow}{\code{signature(object = "marraySpots")}: method which computes a vector of spot row coordinates for each spot.}
    \item{maSub}{\code{signature(object = "marraySpots")}: slot accessor method.}
    \item{maSub<-}{\code{signature(object = "marraySpots")}: slot assignment method.} 
    \item{print}{\code{signature(x = "marraySpots")}: print method for \code{"\link{marraySpots}"} class.}
  }
}


\references{S. Dudoit and Y. H. Yang. (2002). Bioconductor R packages for exploratory analysis and normalization of cDNA microarray data. In G. Parmigiani, E. S. Garrett, R. A. Irizarry and S. L. Zeger, editors, \emph{The Analysis of Gene Expression Data: Methods and Software}, Springer, New York.
}

\author{Sandrine Dudoit, \url{http://www.stat.berkeley.edu/~sandrine}.}

\seealso{\code{\link{marrayLayout}}, \code{\link{marrayRaw}}, \code{\link{marrayNorm}}, \code{\link{marrayInfo}}, \code{\link{marrayTwo}}, \code{\link{print-methods}}, \code{\link{[-methods}}.}


\examples{
## See marrayRaw
}
\keyword{classes}

\eof
\name{marrayTwo-class}
\docType{class}

\alias{marrayTwo-class}
\alias{marrayTwo}

\alias{maX}
\alias{maX<-}
\alias{maY}
\alias{maY<-}

\alias{[,marrayTwo-method}
\alias{maControls<-,marrayTwo-method}
\alias{maControls,marrayTwo-method}
\alias{maGridCol,marrayTwo-method}
\alias{maGridRow,marrayTwo-method}
%\alias{maLayout<-,marrayTwo,marrayLayout-method}
\alias{maLayout,marrayTwo-method}
\alias{maNgc<-,marrayTwo,numeric-method}
\alias{maNgc,marrayTwo-method}
\alias{maNgr<-,marrayTwo,numeric-method}
\alias{maNgr,marrayTwo-method}
\alias{maNsamples,marrayTwo-method}
\alias{maNsc<-,marrayTwo,numeric-method}
\alias{maNsc,marrayTwo-method}
\alias{maNspots<-,marrayTwo,numeric-method}
\alias{maNspots,marrayTwo-method}
\alias{maNsr<-,marrayTwo,numeric-method}
\alias{maNsr,marrayTwo-method}
\alias{maPlate<-,marrayTwo-method}
\alias{maPlate,marrayTwo-method}
\alias{maPrintTip,marrayTwo-method}
\alias{maSpotCol,marrayTwo-method}
\alias{maSpotRow,marrayTwo-method}
\alias{maSub<-,marrayTwo-method}
\alias{maSub,marrayTwo-method}
%\alias{maTargets<-,marrayTwo,marrayInfo-method}
\alias{maTargets,marrayTwo-method}
\alias{maX<-,marrayTwo,matrix-method}
\alias{maX,marrayTwo-method}
\alias{maY<-,marrayTwo,matrix-method}
\alias{maY,marrayTwo-method}
\alias{print,marrayTwo-method}
\alias{show,marrayTwo-method}

%\alias{maLayout}
%\alias{maNgr}
%\alias{maNgc}
%\alias{maNsr}
%\alias{maNsc}
%\alias{maNspots}
%\alias{maSub}
%\alias{maPlate}
%\alias{maControls}

%\alias{maTargets}


\title{Class "marrayTwo", classes and methods for microarray spot statistics}

\description{This class represents pre- or post-normalization intensity data for a batch of cDNA microarrays. A \emph{batch of arrays} consists of a collection of arrays with the same layout  (\code{"\link{marrayLayout}"}). The class contains slots for two types of spot statistics (e.g. the raw green (Cy3) and red (Cy5) foreground  intensities, or normalized log-ratios M and average intensities A), the layout of the arrays, and a description of the target samples hybridized to the arrays}


\section{Objects from the Class}{
Objects can be created by calls of the form 
\code{  new('marrayRaw',}\cr
\code{    maX      = ...., # Object of class matrix}\cr
\code{    maY      = ...., # Object of class matrix}\cr
\code{    maLayout  = ...., # Object of class marrayLayout}\cr
\code{    maTargets = ...., # Object of class marrayInfo}\cr
\code{  )}
}

\section{Slots}{
  \describe{
    \item{\code{maX}:}{Object of class \code{"matrix"}, any spot statistic, rows correspond to spotted probe sequences, columns to arrays in the batch.}
    \item{\code{maY}:}{Object of class \code{"matrix"}, any spot statistic, rows correspond to spotted probe sequences, columns to arrays in the batch.}
    \item{\code{maLayout}:}{Object of class \code{"\link{marrayLayout}"},  layout parameters for the cDNA microarrays.}
    \item{\code{maTargets}:}{Object of class \code{"\link{marrayInfo}"}, description of target samples hybridized to the arrays.}
  }
}

\section{Methods}{
  \describe{
    \item{[}{\code{signature(x = "marrayTwo")}: subsetting operator for spots on the array and arrays in the batch, ensures that all slots are subset properly.}
    \item{maControls}{\code{signature(object = "marrayTwo")}: slot accessor method.}
    \item{maControls<-}{\code{signature(object = "marrayTwo")}:  slot assignment method.} 
   \item{maGridCol}{\code{signature(object = "marrayTwo")}:  method which computes a vector of grid column coordinates for each spot.}
    \item{maGridRow}{\code{signature(object = "marrayTwo")}: method which computes a vector of grid row coordinates for each spot.}
    \item{maLayout}{\code{signature(object = "marrayTwo")}:  slot accessor method.}
    \item{maLayout<-}{\code{signature(object = "marrayTwo", value = "marrayLayout")}: slot assignment method.} 
 \item{maNgc}{\code{signature(object = "marrayTwo")}: slot accessor method.}
    \item{maNgc<-}{\code{signature(object = "marrayTwo", value = "numeric")}: slot assignment method.} 
    \item{maNgr}{\code{signature(object = "marrayTwo")}: slot accessor method.}
    \item{maNgr<-}{\code{signature(object = "marrayTwo", value = "numeric")}: slot assignment method.} 
    \item{maNsamples}{\code{signature(object = "marrayTwo")}: slot accessor method.}
    \item{maNsc}{\code{signature(object = "marrayTwo")}: slot accessor method.}
    \item{maNsc<-}{\code{signature(object = "marrayTwo", value = "numeric")}: slot assignment method.} 
    \item{maNspots}{\code{signature(object = "marrayTwo")}: slot accessor method.}
    \item{maNspots<-}{\code{signature(object = "marrayTwo", value = "numeric")}: slot assignment method.} 
    \item{maNsr}{\code{signature(object = "marrayTwo")}: slot accessor method.}
    \item{maNsr<-}{\code{signature(object = "marrayTwo", value = "numeric")}: slot assignment method.} 
    \item{maPlate}{\code{signature(object = "marrayTwo")}: slot accessor method.}
    \item{maPlate<-}{\code{signature(object = "marrayTwo")}: slot assignment method. }
    \item{maPrintTip}{\code{signature(object = "marrayTwo")}: method which computes a vector of print-tip-group indices for each spot.}
    \item{maSpotCol}{\code{signature(object = "marrayTwo")}: method which computes a vector of spot column coordinates for each spot.}
    \item{maSpotRow}{\code{signature(object = "marrayTwo")}: method which computes a vector of spot row coordinates for each spot.}
    \item{maSub}{\code{signature(object = "marrayTwo")}: slot accessor method.}
    \item{maSub<-}{\code{signature(object = "marrayTwo")}: slot assignment method.} 
    \item{maTargets}{\code{signature(object = "marrayTwo")}: slot accessor method.}
    \item{maTargets<-}{\code{signature(object = "marrayTwo", value = "marrayInfo")}: slot assignment method.} 
    \item{maX}{\code{signature(object = "marrayTwo")}: slot accessor method.}
    \item{maX<-}{\code{signature(object = "marrayTwo", value = "matrix")}: slot assignment method.} 
    \item{maY}{\code{signature(object = "marrayTwo")}: slot accessor method.}
    \item{maY<-}{\code{signature(object = "marrayTwo", value = "matrix")}: slot assignment method.} 
    \item{print}{\code{signature(x = "marrayTwo")}: print method for \code{"\link{marrayTwo}"} class.}
  }
}
\references{S. Dudoit and Y. H. Yang. (2002). Bioconductor R packages for exploratory analysis and normalization of cDNA microarray data. In G. Parmigiani, E. S. Garrett, R. A. Irizarry and S. L. Zeger, editors, \emph{The Analysis of Gene Expression Data: Methods and Software}, Springer, New York.
}

\author{Sandrine Dudoit, \url{http://www.stat.berkeley.edu/~sandrine}.}

\seealso{\code{\link{marrayLayout}}, \code{\link{marrayRaw}}, \code{\link{marrayNorm}}, \code{\link{marrayInfo}}, \code{\link{marraySpots}}, \code{\link{print-methods}}, \code{\link{[-methods}}.}


\examples{
## See marrayRaw
}
\keyword{classes}

\eof
\name{print-methods}
\docType{methods}

\alias{print-methods}
\alias{show-methods}
\alias{print}

\title{Printing methods for microarray objects}

\description{
Print methods were defined for the microarray classes, \code{"\link{marrayInfo}"}, \code{"\link{marrayLayout}"}, \code{"\link{marrayRaw}"}, \code{"\link{marrayNorm}"}, \code{"\link{marraySpots}"}, and \code{"\link{marrayTwo}"}. These methods produce summaries of the intensity and textual data stored in different classes of microarray objects.
}

\section{Methods}{\describe{
\item{x = ANY}{generic print method}
\item{x = marrayLayout}{for an object of class \code{"\link{marrayLayout}"}, the method prints main layout parameters such as the number of spots and the dimensions of the spot and grid matrices.}
\item{x = marrayInfo}{for an object of class \code{"\link{marrayInfo}"}, the method prints the first 10 rows of the \code{"maInfo"} and \code{"maLabels"} slots.}
\item{x = marrayRaw}{for an object of class \code{"\link{marrayRaw}"}, the method prints a short description of the microarray layout \code{"maLayout"} and the target samples hybridized to the arrays \code{"maTargets"}, and a summary of the distribution of the log-ratio statistics \code{"maM"}.}
\item{x = marrayNorm}{for an object of class \code{"\link{marrayNorm}"}, the method prints a short description of the microarray layout \code{"maLayout"} and the target samples hybridized to the arrays \code{"maTargets"}, and a summary of the distribution of the log-ratio statistics \code{"maM"}.}
\item{x = marrayTwo}{for an object of class \code{"\link{marrayTwo}"}, the method prints a short description of the microarray layout \code{"maLayout"}, and a summary of the distribution of the statistics \code{"maX"} and \code{"maY"}.}
\item{x = marraySpots}{for an object of class \code{"\link{marraySpots}"}, the method prints a short description of the microarray layout \code{"maLayout"} and the probe sequences spotted on the arrays \code{"maGnames"}.}
}}
\keyword{methods}


\eof
\name{swirl}
\alias{swirl}
\alias{swirl.1.spot}
\alias{swirl.2.spot}
\alias{swirl.3.spot}
\alias{swirl.4.spot}
\alias{fish.gal}
\alias{SwirlSample.txt}
\alias{SwirlSample}
\alias{last.warning}

\title{Gene expression data from Swirl zebrafish cDNA microarray experiment}

\description{
  The \code{swirlRaw} dataset consists of an object \code{swirl}
  of class \code{"\link{marrayRaw}"}, which represents
  pre-normalization intensity data for a batch of cDNA microarrays.  \cr

This experiment was carried out
using zebrafish as a model organism to study early development in
vertebrates. Swirl is a point mutant in the BMP2 gene that affects the
dorsal/ventral body axis. Ventral fates such as blood are reduced, whereas
dorsal structures such as somites and notochord are expanded. A goal of
the Swirl experiment is to identify genes with altered expression in
the swirl mutant compared to wild-type zebrafish. Two sets of dye-swap experiments were performed, for a total of four replicate hybridizations. For
each of these hybridizations, target cDNA from the swirl mutant was labeled using
one of the Cy3 or Cy5 dyes and the target cDNA wild-type mutant was labeled using
the other dye. Target cDNA was hybridized to microarrays containing
8,448 cDNA probes, including 768 controls spots (e.g. negative, positive,
and normalization controls spots). Microarrays were printed using \eqn{4 \times 4} print-tips and are thus partitioned into a \eqn{4 \times 4} grid matrix. Each grid consists of a \eqn{22
\times 24} spot matrix that was printed with a single print-tip. Here, spot row and plate coordinates should coincide, as each row of spots corresponds to probe sequences from the same 384 well-plate.\cr


Each of the four hybridizations produced
a pair of 16-bit images, which were processed using the image analysis software package \code{Spot}. Raw images of the Cy3 and Cy5 fluorescence intensities for all fourhybridizations are available at \url{http://fgl.lsa.berkeley.edu/Swirl/index.html}.the dataset includes four output files
\code{swirl.1.spot},
\code{swirl.2.spot},
\code{swirl.3.spot}, and
\code{swirl.4.spot}
from the \code{Spot} package.  Each of these files contains
8,448 rows and 30 columns; rows correspond to spots and columns to
different statistics from the \code{Spot} image analysis output. The file
\code{fish.gal} is a gal file generated by the \code{GenePix} program; it contains information on individual probe sequences, such as gene names, spot ID, spot coordinates.  Hybridization information for the mutant and wild-type target samples is stored in \code{SwirlSample.txt}.


}

\usage{data(swirl)}

\source{These data were provided by Katrin Wuennenberg-Stapleton from the Ngai Lab at UC Berkeley. The swirl embryos for this experiment were provided by David Kimelman and David
Raible at the University of Washington.}

\keyword{datasets}



\eof
