\name{DFS}
\alias{DFS}
\alias{DFS,graph,character-method}
\title{Depth First Search  }
\description{
  This function implements algorithm 4.2.1 of Gross and Yellen. The
  input is a \code{graph} and a \code{node} to start from. It returns a
  standard vertex labeling of \code{graph}. This is a vector with
  elements corresponding to the nodes of \code{graph} and with values
  that correspond to point in the depth first search the node is
  visited.
}
\usage{
DFS(object, node, checkConn=FALSE)
}
\arguments{
  \item{object}{An instance of the \code{graph} class. }
  \item{node}{A \code{character} indicating the starting node. }
  \item{checkConn}{A \code{logical} indicating whether the connectivity
    of the graph should be checked. }
}
\details{
  This function implements algorithm 4.2.1 of Gross and Yellen. Specific
  details are given there.

  It requires that the graph be connected. This can be checked, but is
  expensive and is only done when requested.

  A faster and mostly likely better implementation of depth first
  searching is given by \code{\link[RBGL]{dfs}} in the \pkg{RBGL}
  package.  
}
\value{
  A vector with names given by the nodes of \code{graph} whose values
  are \code{0} to one less than the number of nodes. These indices
  indicate the point at which the node will be visited.
}
\references{\emph{Graph Theory and its Applications}, J. Gross and J. Yellen. }
\author{R. Gentleman }

\seealso{\code{\link{boundary}}}

\examples{
  RNGkind("Mersenne-Twister")
  set.seed(123)
  g1 <- randomGraph(letters[1:10], 1:4, p=.3)
  RNGkind()
  DFS(g1, "a")
}
\keyword{manip}

\eof
\name{Graph to Matrix coersions}
\alias{coerce,graphNEL,matrix-method}

\title{Coercing matrices to graphs}
\description{
Functions for converting graph objects into matrices
}
\details{
A \code{graphNEL} can be directly coerced into a \code{adjacency} matrix
using the \code{as} method. If the graph is \code{undirected} then the
matrix will by symmetric.

}
\value{
  An adjacency 
  For \code{aM2bpG}, an object of class \code{graphNEL}.  For
  \code{ftM2adjM}, an adjacency matrix representation of the coerced graph.
}

\author{Byron Ellis}

\examples{

From <- matrix(sample(c(0,1),16,replace=TRUE),ncol=4,nrow=4)
dimnames(From) <- list(LETTERS[1:4],LETTERS[1:4])
To <- as(From,"graphNEL")
Back <- as(To,"matrix")
if(sum(From == Back) != 16) warning("conversion failed")

}

\keyword{graphs }

\eof
\name{integrinMediatedCellAdhesion}
\alias{IMCA}
\alias{IMCAAttrs}
\alias{IMCAGraph}
\alias{IMCALocusLink}
\docType{data}
\title{KEGG Integrin Mediated Cell Adhesion graph}
\description{
  A graph representing the integrin-mediated cell adhesion pathway from
  KEGG, as well as a data.frame of attributes for use in plotting the
  graph with \code{Rgraphviz} and a list to compare the nodes with their
  respective LocusLink IDs.  
}
\usage{
data(integrinMediatedCellAdhesion)
}
\details{
  The \code{integrinMediatedCellAdhesion} data set contains three objects:
  
  The first is \code{IMCAGraph}, which is an object of class
  \code{graph-NEL} and represents the hsa04510 graph from \code{KEGG}.

  The second is \code{IMCAAttrs}, which is a data.frame with two columns,
  and a row for every node in \code{IMCAGraph}.  The first column lists
  what color the node is represented with on the \code{KEGG} site.  The
  second column lists the type of the node - either \code{genesym} or
  \code{text}.  Most nodes are of type \code{genesym} as they represent
  genes, but some of the nodes in the \code{KEGG} graph were not genes
  and thus those nodes are of type \code{text}.

  The third, \code{IMCALocusLink} is a named list where the names
  correspond to the node names in \code{IMCAGraph}.  The values of the
  list are the LocusLink IDs that correspond to that node in the KEGG graph.
}
\source{
\url{http://www.genome.ad.jp/kegg/pathway/hsa/hsa04510.html}
}
\examples{
data(integrinMediatedCellAdhesion)
if (require("Rgraphviz") & interactive())
  plot(IMCAGraph)
}
\keyword{datasets}

\eof
\name{acc-methods}
\docType{methods}
\alias{acc-methods}
\alias{acc,graph-method}
\alias{acc,clusterGraph-method}
\alias{acc}
\alias{acc,graph,character-method}
\alias{acc,clusterGraph,character-method}
\title{Methods for Accessibility Lists}

\description{
 This generic function takes an object that inherits from the \code{graph} 
class and a node in that graph and returns a vector containing information
about all other nodes that are accessible from the given node. The
methods are vectorized so that \code{index} can be a vector.
}
\usage{
\S4method{acc}{graph,character}(object, index)
\S4method{acc}{clusterGraph,character}(object, index)

}
\arguments{
  \item{object}{An instance of the appropriate graph class.}
  \item{index}{A character vector specifying the nodes for which
    accessibilty information is wanted.}
}

\value{
  The methods should return a named list of integer vectors. The
  \code{names} of the list correspond to the names of the supplied
  nodes. For each element of the list the returned vector is named. The
  names of the vector elements correspond to the nodes that are
  accessible from the given node. The values in the vector indicate how
  many edges are between the given node and the node in the return vector.
}
\section{Methods}{\describe{
    \item{object = graph}{An object of class graph.}
    \item{object = clusterGraph}{An instance of the \code{clusterGraph}
      class.} 
\item{index}{A \code{character} vector of indices corresponding to nodes in the
  graph.}  }
}

\examples{
 set.seed(123)
 gR3 <- randomGraph(LETTERS[1:10], M<-1:2, p=.5)
 acc(gR3, "A")
 acc(gR3, c("B", "D"))
}

\keyword{methods}

\eof
\name{addEdge}
\alias{addEdge}
\title{ addEdge }
\description{
  A function to add an edge to a graph.
}
\usage{
addEdge(from, to, graph, weights)
}
\arguments{
  \item{from}{The node the edge starts at }
  \item{to}{The node the edge goes to. }
  \item{graph}{The graph that the edge is being added to. }
  \item{weights}{A vector of weights, one for each edge. }
}
\details{
  Both \code{from} and \code{to} can be vectors. They need not be the
  same length (if not the standard rules for replicating the shorter one
  are used). Edges are added to the graph between the supplied nodes.

  The \code{weights} are given for each edge.

  The implementation is a bit too oriented towards the \code{graphNEL}
  class and will likely change in the next release to accomodate more
  general graph classes.

  If the graph is undirected then the edge is bidirectional (and only
  needs to be added once). For directed graphs the edge is directional.
}
\value{
  A new instance of a graph object with the same class as \code{graph}
  but with the indicated edges added.
}
\author{R. Gentleman}
\seealso{\code{\link{addNode}},\code{\link{removeEdge}},
    \code{\link{removeNode}} }

  \examples{
V <- LETTERS[1:4]
edL2 <- vector("list", length=4)
names(edL2) <- V
for(i in 1:4)
  edL2[[i]] <- list(edges=c(2,1,2,1)[i], weights=sqrt(i))
gR2 <- new("graphNEL", nodes=V, edgeL=edL2, edgemode="directed")

gX <- addEdge("A", "C", gR2, 1)

gR3 <- randomEGraph(letters[10:14], .4)
gY <- addEdge("n", "l", gR3, 1)

}
\keyword{ manip }

\eof
\name{addNode}
\alias{addNode}
\alias{addNode,character,graphNEL,missing-method}
\alias{addNode,character,graphNEL,list-method}
\alias{addNode,character,distGraph,list-method}
\title{ addNode }
\description{
  Add one or more nodes to a graph.
}
\usage{
addNode(node, object, edges)
}
\arguments{
  \item{node}{A character vector of node names. }
  \item{object}{A \code{graph} }
  \item{edges}{A named list of edges.} 
}
\details{
  The supplied \code{node}s are added to the set of nodes of the
  \code{object}.

  If \code{edges} are provided then their must be the
  same number as there are \code{node}s and the must be in the same
  order. The elements of the \code{edges} list are vectors. They can be
  character vectors of node labels for nodes in \code{object} and if so
  then they are added with unit weights. If the vector is numeric then
  it must be named (with labels corresponding to nodes in the
  \code{object}) and the values are taken to be the edge weights.
  
  When the \code{object} is a \code{distGraph} then the \code{edges} must 
  be supplied and they must contain appropriate distances for all nodes 
  both those in \code{object} and those supplied.
}
\value{
  A new graph of the same class as \code{object} with the supplied node
  added to the set of nodes.
}
\author{R. Gentleman }
\seealso{\code{\link{removeNode}}, \code{\link{removeEdge}},
  \code{\link{addEdge}}}

  \examples{

  V <- LETTERS[1:4]
  edL1 <- vector("list", length=4)
  names(edL1) <- V
  for(i in 1:4)
     edL1[[i]] <- list(edges=c(2,1,4,3)[i], weights=sqrt(i))
  gR <- new("graphNEL", nodes=V, edgeL=edL1)
  gX <- addNode("X", gR)

set.seed(123)
g1 <- randomGraph(letters[1:10], 1:4, p=.3)
g2 <- addNode("z", g1, edges=list(c("a", "h", "g")))

}
\keyword{ manip }

\eof
\name{adj-methods}
\docType{methods}
\alias{adj}
\title{ Methods for finding the adjacency list for selected nodes. } 
\alias{adj-methods}
\description{
 This generic function takes an object that inherits from the \code{graph}
class and a node in that graph and returns a vector containing information
about all other nodes that are adjacent to the given node.
This means that they are joined to the given node by an edge.
The accessibility list, \code{\link{acc}} is the list of all nodes that can
be reached from a specified node.
}
\value{
  The methods return vector of nodes that are adjacent to the 
 specified node.
}

\section{Methods}{\describe{
\item{object = graph}{An object that inherits from glass \code{graph}}
\item{index}{An index (could be multiple) which can be either the
  integer offset for the node(s) or their labels.}
}}
\seealso{\code{\link{acc-methods}}}
\examples{
 set.seed(123)
 gR3 <- randomGraph(LETTERS[1:4], M<-1:2, p=.5)
 adj(gR3, "A")
 adj(gR3, c(2,3))
}


\keyword{methods}

\eof
\name{apoptosisGraph}
\alias{apopGraph}
\alias{apopAttrs}
\alias{apopLocusLink}
\docType{data}
\title{KEGG apoptosis pathway graph}
\description{
  A graph representing the apoptosis pathway from
  KEGG, as well as a data.frame of attributes for use in plotting the
  graph with \code{Rgraphviz} and a list to compare the nodes with their
  respective LocusLink IDs.  
}
\usage{
data(apopGraph)
}
\details{
  The \code{apopGraph} data set contains three objects:
  
  The first is \code{apopGraph}, which is an object of class
  \code{graph-NEL} and represents the hsa04210 graph from \code{KEGG}.

  The second is \code{apopAttrs}, which is a data.frame with two columns,
  and a row for every node in \code{apopGraph}.  The first column lists
  what color the node is represented with on the \code{KEGG} site.  The
  second column lists the type of the node - either \code{genesym} or
  \code{text}.  Most nodes are of type \code{genesym} as they represent
  genes, but some of the nodes in the \code{KEGG} graph were not genes
  and thus those nodes are of type \code{text}.

  The third, \code{apopLocusLink} is a named list where the names
  correspond to the node names in \code{apopGraph}.  The values of the
  list are the LocusLink IDs that correspond to that node in the KEGG graph.
}
\source{
\url{http://www.genome.ad.jp/kegg/pathway/hsa/hsa04210.html}
}
\examples{
data(apopGraph)
if (require("Rgraphviz") & interactive())
  plot(apopGraph)
}
\keyword{datasets}

\eof
\name{aveNumEdges}
\alias{aveNumEdges}
\title{ Calculate the average number of edges in a graph }
\description{
  aveNumEdges divides the number of edges in the graph
  by the number of nodes to give the average number of edges. 
}
\usage{
aveNumEdges(objgraph)
}
\arguments{
  \item{objgraph}{ the graph object }
}
\value{
  A double representing the average number of edges will be returned.
}

\author{ Elizabeth Whalen }

\seealso{ \code{\link{numEdges}}, \code{\link{mostEdges}}, 
   \code{\link{numNoEdges}} 
}

\examples{
 set.seed(124)
 g1 <- randomGraph(1:10, letters[7:12], p=.6)
 aveNumEdges(g1)

}
\keyword{ manip }




\eof
\name{biocRepos}
\alias{biocRepos}
\docType{data}
\title{A graph representing the Bioconductor package repository}
\description{
  This graph is a rendition of the Bioconductor package repository and
  represents the dependency graph of that repository.  An edge between
  two package denotes a dependency on the 'to' package by the 'from'
  package.  
}
\usage{data(biocRepos)}
\source{
  This graph was generated by the function
  \code{\link{buildRepDepGraph}}.
  
}
\seealso{\code{\link{buildRepDepGraph}}}
\examples{
   data(biocRepos)
   ## An example of usage will be here soon
}
\keyword{datasets}

\eof
\name{boundary}
\alias{boundary}

\title{ Returns the Boundary between a Graph and a SubGraph }
\description{
 The boundary of a subgraph is the set of nodes in the original graph
 that have edges to nodes in the subgraph. The function \code{boundary}
 computes the boundary and returns it as a list whose length is the same
 length as the number of nodes in the subgraph.
}
\usage{
boundary(subgraph, graph)
}

\arguments{
  \item{graph}{ the original graph from which the boundary will 
                    be created }
  \item{subgraph}{ can either be the vector of the node labels or the
    subgraph itself.}
}
\details{
  The \emph{boundary} of a \emph{subgraph} is the set of nodes in the
  graph which have an edge that connects them to the specified subgraph
  but which are themselves not elements of the subgraph.

  For convenience users can specify the subgraph as either a graph or a
  vector of node labels.
}
\value{
  This function returns a named list of length equal to the number of
  nodes in \code{subgraph}. The elements of the list
  correspond to the nodes in the \code{subgraph}. The elements are lists
  of the nodes in \code{graph} which share an edge with the
  respective node in \code{subgraph}.

}

\author{ Elizabeth Whalen and R. Gentleman }

\seealso{ \code{\link{subGraph}}, \code{\link{graph-class}}
 }

\examples{
  set.seed(123)
  g1 <- randomGraph(letters[1:10], 1:4, p=.3)
  ##both should be "a"
  boundary(c("g", "i"), g1)
}
\keyword{ manip }


\eof
\name{buildRepDepGraph}
\alias{buildRepDepGraph}
\alias{pkgInstOrder}
\title{Functionality to manage repository dependency graphs}
\description{
  These functions can be used to represent and manipulate dependency
  graphs for a specified package repository.
}
\usage{
buildRepDepGraph(repository, depLevel = c("Depends", "Suggests"))
pkgInstOrder(pkg, repGraph)
}
\arguments{
  \item{repository}{A URL to a \code{CRAN} style repository}
  \item{depLevel}{One of \code{Depends} or \code{Suggests}, detailing
    the level of dependencies to search.  The \code{Suggests} value
    includes everything in \code{Depends}.}
  \item{pkg}{The package to get the installation order for}
  \item{repGraph}{A \code{graph} object representing a repository, as
    from \code{buildRepDepGraph}}
}
\value{
  For \code{buildRepDepGraph}, a graph representing the dependency
  structure of the specified repository, where an edge from node
  \code{A} to node \code{B} represents a dependency on \code{B} by \code{A}.

  For \code{pkgInstOrder}, a vector is returned, listing the appropriate
  order one would take to install all of the necessary packages to
  install the specified package.  That is, it makes sure that at every
  step, any package being installed does not depend on one that has not
  yet been installed.  This order can then be used with functions such
  as \code{install.packages}.
}
\author{Jeff Gentry}
\examples{
    repos <- getOption("repositories")()["BIOC"] ## Get BIOC repos
    z <- buildRepDepGraph(repos)
}
\keyword{graphs}

\eof
\name{calcProb}
\alias{calcProb}

\title{ Calculate the hypergeometric probability of the subgraph's number of 
edges.
}
\description{
  \code{calcProb} calculates the probability of having the number of edges
  found in the subgraph given that it was made from \code{origgraph}.
  The hypergeometric distribution is used to calculate the 
  probability (using the pdf).
}
\usage{
calcProb(subgraph, origgraph)
}

\arguments{
  \item{subgraph}{ subgraph made from the original graph }
  \item{origgraph}{ original graph object from which the subgraph was made }
}

\value{
  The probability of the subgraph's number of edges is returned.
}

\author{ Elizabeth Whalen }

\seealso{ \code{\link{calcSumProb}} 
}

\examples{
 #none right now
}
\keyword{ manip }

\eof
\name{calcSumProb}
\alias{calcSumProb}

\title{ Calculate the probability that a subgraph has an unusual number
  of edges.}
\description{
  For any graph a set of nodes can be used to obtain an induced subgraph
  (see \code{\link{subGraph}}). An interesting question is whether that
  subgraph has an unusually large number of edges. This function
  computes the probability that a \emph{random} subgraph with the same
  number of nodes has more edges than the number observed in the
  presented subgraph. The appropriate probability distribution is
  the hypergeometric.
}
\usage{
calcSumProb(sg, g)
}

\arguments{
  \item{sg}{ subgraph made from the original graph }
  \item{g}{ original graph object from which the subgraph was made }
}

\value{
  The probability of having greater than or equal to the subgraph's 
  number of edges is returned.
}
\details{
The computation is based on the following argument. In the original
graph there are \eqn{n} nodes and hence \eqn{N=n*(n-1)/2} edges in the
complete graph. If we consider these \eqn{N} nodes to be of two types,
corresponding to those that are either in our graph, \code{g}, or not in
it. Then we think of the subgraph which has say \eqn{m} nodes and
\eqn{M=m*(m-1)/2} possible edges as representing \eqn{M} draws from an
urn containing \eqn{N} balls of which some are white (those in \code{g})
and some are black. We count the number of edges in the subgraph and use
a Hypergeomtric distribution to ask whether our subgraph is particularly
dense. 
  }
\author{ Elizabeth Whalen }

\seealso{ \code{\link{calcProb}} }

\examples{
  set.seed(123)
  V <- letters[14:22]
  g1 <- randomEGraph(V, .2)

  sg1 <- subGraph(letters[c(15,17,20,21,22)], g1)
  calcSumProb(sg1, g1)
}
\keyword{ manip }

\eof
\name{clearNode}
\alias{clearNode}
\title{ clearNode }
\description{
  This function removes all edges to or from the specified node in the graph.
}
\usage{
clearNode(node, object)
}
\arguments{
  \item{node}{a node }
  \item{object}{a \code{graph} }
}
\details{
  All edges to and from \code{node} are removed. \code{node} can be a
  vector. 
}
\value{
  A new instance of the graph with all edges to and from the specified
  node(s) removed.
}
\author{R. Gentleman}

\seealso{\code{\link{removeNode}}, \code{\link{removeEdge}}}

  \examples{
 V <- LETTERS[1:4]
 edL3 <- vector("list", length=4)
 for(i in 1:4)
   edL3[[i]] <- list(edges=(i\%\%4)+1, weights=i)
 names(edL3) <- V
 gR3 <- new("graphNEL", nodes=V, edgeL=edL3, "directed")
 g4 <- clearNode("A", gR3)

}
\keyword{manip }

\eof
\name{clusterGraph-class}
\docType{class}
\alias{clusterGraph-class}
\alias{clusterGraph-class}
\alias{adj,clusterGraph-method}
\alias{connComp,clusterGraph-method}
\alias{edges,clusterGraph,missing-method}
\alias{edges,clusterGraph,character-method}
\alias{edgeWeights,clusterGraph-method}
\alias{nodes,clusterGraph-method}
\alias{numNodes,clusterGraph-method}
\alias{show,clusterGraph-method}
\title{Class "clusterGraph" }
\description{  A cluster graph is a special sort of graph for clustered
  data. Each cluster forms a completely connected subgraph. Three are no
  edges between clusters.}
\section{Objects from the Class}{
Objects can be created by calls of the form \code{new("clusterGraph", ...)}.
}
\section{Slots}{
  \describe{
    \item{\code{clusters}:}{Object of class \code{"list"} a list of the
      labels of the elements, one element of the list for each cluster. }
  }
}
\section{Extends}{
Class \code{"graph"}, directly.
}
\section{Methods}{
  \describe{
    \item{connComp}{\code{signature(object = "clusterGraph")}: find the
      connected components; simply the clusters in this case. }
    \item{acc}{\code{signature(object = "clusterGraph")}: find the
      accessible nodes from the supplied node. }
    \item{adj}{\code{signature(object = "clusterGraph")}: find the
      adjacent nodes to the supplied node. }
    \item{nodes}{\code{signature(object = "clusterGraph")}: return the
      nodes. }
    \item{numNodes}{\code{signature(object = "clusterGraph")}: return
      the number of nodes. }
  }
}
\author{R. Gentleman}

\seealso{ \code{\link{graph-class}}, \code{\link{distGraph-class}} }

\examples{
  cG1 <- new("clusterGraph", clusters=list(a=c(1,2,3), b=c(4,5,6)))
  cG1
  acc(cG1, c("1", "2"))
}
\keyword{classes}

\eof
\name{combineNodes}
\alias{combineNodes}
\title{ combineNodes  }
\description{
  A function to combine, or collapse, a specified set of nodes in a graph.
}
\usage{
combineNodes(nodes, graph, newName)
}
\arguments{
  \item{nodes}{A set of nodes that are to be collapsed. }
  \item{graph}{The graph containing the nodes }
  \item{newName}{The name for the new, collapsed node. }
}
\details{
  The nodes specified are reduced to a single new node with label given
  by \code{newName}. The in and out edges of the set of nodes are all
  made into in and out edges for the new node.
}
\value{
  An new instance of a graph of the same class as \code{graph} is
  returned. This new graph has the specified nodes reduced to a single
  node. 
}
\author{R. Gentleman}

\seealso{\code{\link{inEdges}}, \code{\link{addNode}}}

\examples{
  V <- LETTERS[1:4]
  edL1 <- vector("list", length=4)
  names(edL1) <- V
  for(i in 1:4)
    edL1[[i]] <- list(edges=c(2,1,4,3)[i], weights=sqrt(i))
  gR <- new("graphNEL", nodes=V, edgeL=edL1)
  gR <- addNode("M", gR)
  gR <- addEdge("M", "A", gR, 1)
  gR <- addEdge("B", "D", gR, 1)
  gX <- combineNodes(c("B","D"), gR, "X")
}
\keyword{ manip}

\eof
\name{distGraph-class}
\docType{class}
\alias{distGraph-class}
\alias{threshold}
\alias{Dist}
\alias{[.dist}
\alias{distGraph-class}
\alias{adj,distGraph-method}
\alias{Dist,distGraph-method}
\alias{edges,distGraph,missing-method}
\alias{edges,distGraph,character-method}
\alias{nodes,distGraph-method}
\alias{numNodes,distGraph-method}
\alias{show,distGraph-method}
\alias{threshold,distGraph-method}
\alias{edgeWeights,distGraph-method}
\title{Class "distGraph"}
\description{ A class definition for graphs that are based on distances.}
\section{Objects from the Class}{
Objects can be created by calls of the form \code{new("distGraph", ...)}.
}
\section{Slots}{
  \describe{
    \item{\code{Dist}:}{Object of class \code{"dist"} that forms the
    basis for the edge weights used in the \code{distGraph}.}
  }
}
\section{Extends}{
Class \code{"graph"}, directly.
}
\section{Methods}{
  \describe{
    \item{show}{\code{signature(object = "distGraph")}: a print method}
    \item{Dist}{\code{signature(object = "distGraph")}: return the dist
      object. }
    \item{adj}{\code{signature(object = "distGraph")}: find the nodes
      adjacent to the supplied node. }
    \item{nodes}{\code{signature(object = "distGraph")}: return the
      nodes in the graph. }
    \item{numNodes}{\code{signature(object = "distGraph")}: return the
      number of nodes. }
    \item{threshold}{\code{signature(object = "distGraph")}: remove all
      distances that are larger than the supplied threshold. }
  }
}
\references{Shamir's paper and Butte et al}
\author{R. Gentleman }

\seealso{\code{\link{graph-class}}, \code{\link{clusterGraph-class}}}

\examples{
   set.seed(123)
   x <- rnorm(26)
   names(x) <- letters
   library(stats)
   d1 <- dist(x)
   g1 <- new("distGraph", Dist=d1)
}
\keyword{classes}

\eof
\name{duplicatedEdges}
\alias{duplicatedEdges}
\title{duplicatedEdges  }
\description{
  A multigraph is a graph where edges between nodes can be represented
  several times. For some algorithms this causes
  problems. \code{duplicatedEdges} tests an instance of the
  \code{graphNEL} class to see if it has duplicated edges and returns
  \code{TRUE} if it does and \code{FALSE} otherwise.
}
\usage{
duplicatedEdges(graph)
}
\arguments{
  \item{graph}{An instance of the class \code{graphNEL} }
}
\details{
  It would be nice to handle other types of graphs.
}
\value{
  A logical, either \code{TRUE} if the graph has duplicated edges or
  \code{FALSE} it not.
}

\author{R. Gentleman }

\seealso{ \code{\link{connComp}}, \code{\link{ugraph}} }

\examples{
##---- Should be DIRECTLY executable !! ----
##-- ==>  Define data, use random,
}
\keyword{manip }

\eof
\name{edgeMatrix}
\alias{edgeMatrix}
\alias{eWV}
\alias{pathWeights}
\alias{edgeMatrix,graphNEL-method}
\alias{edgeMatrix,clusterGraph-method}
\alias{edgeMatrix,distGraph-method}
\title{Compute an Edge Matrix or weight vector for a Graph  }
\description{
 For our purposes an \emph{edge matrix} is a matrix with two rows and as
 many columns as there are edges. The entries in the first row are the
 index of the node the edge is \emph{from}, those in the second row indicate
 the node the edge is \emph{to}.

 If the graph is \dQuote{undirected} then the \code{duplicates} option
 can be used to indicate whether reciprocal edges are wanted. The
 default is to leave them out. In this case the notions of \emph{from}
 and \emph{to} are not relevant.
}
\usage{
edgeMatrix(object, duplicates=FALSE)
eWV(g, eM, sep = ifelse(edgemode(g) == "directed", "->",
                 "--"), useNNames=FALSE)
pathWeights(g, p, eM=NULL)
}
\arguments{
  \item{object}{An object that inherits from \code{graph}. }
  \item{g}{An object that inherits from \code{graph}. }
  \item{duplicates}{Whether or not duplicate edges should be produced
    for \dQuote{undirected} graphs. }
  \item{eM}{An edge matrix}
  \item{sep}{a character string to concatenate node labels in
    the edge label}
  \item{useNNames}{a logical; if TRUE, node names are used in
    the edge label; if FALSE, node indices are used}
  \item{p}{a vector of node names constituting a path in graph \code{g}}
  \item{\dots}{arguments passed to \code{edgeMatrix}.}
}
\details{
  Implementations for \code{graphNEL}, \code{clusterGraph} and
  \code{distGraph} are available. 
}
\value{
  \code{edgeMatrix} returns a matrix with two rows, \emph{from} and \emph{to}, and as many columns
  as there are edges. Entries indicate the index in the node vector that
  corresponds to the appropriate end of the edge.

  \code{eWV} uses the edge matrix to create an annotated vector of 
  edge weights.

  \code{pathWeights} returns an annotated vector of edge weights
  for a specified path in a graph.
 }

\note{A path through an undirected graph may have several
representations as a named vector of edges.  Thus in the
example, when the weights for path b-a-i are requested,
the result is the pair of weights for edges a--b and a--i,
as these are the edge labels computed for graph g1.}

\author{R. Gentleman }

\seealso{ \code{\link{edges}} }

\examples{
  set.seed(123)
  g1 <- randomGraph(letters[1:10], 1:4, p=.3)
  edgeMatrix(g1)
  g2 <- new("clusterGraph", clusters=list(a=c(1,2,3), b=c(4,5,6)))
  em2 <- edgeMatrix(g2)
  eWV(g1, edgeMatrix(g1))
  eWV(g1, edgeMatrix(g1), useNNames=TRUE)
  pathWeights(g1, c("b", "a", "i"))
}
\keyword{manip }

\eof
\name{edgeWeights}
\alias{edgeWeights}
\title{Retrieve the edge weights for specified indices. }
\description{
  Obtains the weights for all edges from the nodes specified by
  \code{index}. 
}
\usage{
edgeWeights(object, index)
}
\arguments{
  \item{object}{A graph, any object that inherits from the \code{graph}
    class.} 
  \item{index}{If supplied either a numeric vector or a vector of node names. }
}
\details{
  If \code{index} is suppled then edge weights from these nodes to all
  adjacent nodes are found and returned. If \code{index} is not supplied
  then the edge weights for all nodes are returned.
}
\value{
  A list of named edge weight vectors. The names are index offsets into
  the set of nodes. 
}
\author{R. Gentleman }

\seealso{\code{\link{nodes}}, \code{\link{edges}}}

\examples{
  V <- LETTERS[1:4]
  edL2 <- vector("list", length=4)
  names(edL2) <- V
  for(i in 1:4)
    edL2[[i]] <- list(edges=c(2,1,2,1)[i], weights=sqrt(i))
  gR2 <- new("graphNEL", nodes=V, edgeL=edL2)
  edgeWeights(gR2, "C")
  edgeWeights(gR2)

}
\keyword{manip}

\eof
\name{fromGXL-methods}
\docType{methods}
\title{  Methods for GXL manipulations in package graph }
\alias{fromGXL-methods}
\alias{toGXL-methods}
\alias{GXL}
\alias{dumpGXL-methods}
\alias{dumpGXL}
\alias{fromGXL}     
\alias{gxlTreeNEL}
\alias{NELhandler}  
\alias{toGXL}      
\alias{validateGXL}
\alias{connection-class}
\alias{file-class}
\alias{dumpGXL,connection-method}
\alias{fromGXL,connection-method}
\alias{validateGXL,connection-method}
\description{GXL \url{www.gupro.de/GXL} is "an XML sublanguage designed
to be a standard exchange format for graphs".  This document describes
tools in the graph package for importing GXL data to R and for
writing graph data out as GXL.}
\value{
\item{fromGXL}{currently returns a graphNEL when possible.  This
function is based on xmlEventParse with handlers defined
in the function NELhandler.  The dump()
element of this handler should emit information on
all children of nodes and edges; the asGraphNEL()
element will return a graphNEL object with weights
if child <attr> with name attribute "weights" is
present for each edge element.}
\item{toGXL}{for an
input of class "graphNEL", returns an object of class c("XMLInternalDOM", "XMLOutputStream");
see the example for how to convert this to a text stream encoding XML}
\item{dumpGXL}{returns an R list with all the node, edge, and named attribute
information specified in the GXL stream}  
\item{validateGXL}{returns
silently (invisibly returns the parsed tree) for a DTD-compliant stream, or is otherwise very noisy}
}
\section{Methods}{
\describe{
\item{fromGXL}{ con = connection: returns a graphNEL based on a parsing of
the GXL stream on the connection }
\item{dumpGXL}{ con = connection: returns an R list based on a parsing of
the GXL stream on the connection }
\item{validateGXL}{ con = connection: checks the GXL stream against its DTD}
\item{toGXL}{ object = graphNEL: creates an XMLInternalDOM representing the graph in GXL}
}
}
\author{Vince Carey <stvjc@channing.harvard.edu>}
\note{At present, toGXL does not return a validating GXL stream
because XML package does not properly handle the dtd and namespaces
arguments to xmlTree.  This is being repaired.  To fix
the stream, add
\code{
<!DOCTYPE gxl SYSTEM "http://www.gupro.de/GXL/gxl-1.0.1.dtd">
}
as second record in the output.

Some structures in a graphNEL and some tags in GXL may not be handled
at this time.

Note added 18 Jul 2003: It has been observed that graphviz dot2gxl generates
GXL with graph attribute edgeids=FALSE.  fromGXL will fail to process
the edges correctly.  You can manually add arbitrary edgeids to the dot2gxl
output edges and fromGXL will succeed.  Perhaps fromGXL can be made less
dependent on edge id attributes.

Note added 15 August 2003: dot2gxl is supposed to introduce
edgeids now.  

Note added 15 August 2003: all GXL to be handled should possess
an edgemode attribute.
}
\examples{
sf <- file(system.file("GXL/simpleExample.gxl", package="graph"))
show(fromGXL(sf))
print(dumpGXL(sf))
#validateGXL(sf)
#bad <- file(system.file("GXL/c2.gxl", package="graph"))
# here's how you can check if the GXL is well-formed
#
# try( validateGXL(bad) )
#
gR <- new("graphNEL", nodes=letters[1:4], edgeL=list(
 a=list(edges=4), b=list(edges=3), c=list(edges=c(2,1)), d=list(edges=1)),
 edgemode="directed")
#
# following requires that you are using XML bound with recent libxml2
#
an <- as.numeric
if (an(libxmlVersion()$major)>=2 && an(libxmlVersion()$minor)>=4)
  cat(saveXML(toGXL(gR)$value()))
wtd <- file(system.file("GXL/kmstEx.gxl", package="graph"))
wtdg <- fromGXL(wtd)
print(edgeWeights(wtdg))
}
\keyword{methods}
\keyword{ models }

\eof
\name{gEdge-class}
\docType{class}
\alias{gEdge-class}
\alias{edgeID}
\alias{bNode}
\alias{eNode}
\alias{edgeID,gEdge-method}
\alias{bNode,gEdge-method}
\alias{eNode,gEdge-method}
\title{Class "gEdge" }
\description{A very generic class for describing the edges of a graph.}
\section{Objects from the Class}{
Objects can be created by calls of the form \code{new("gEdge", ...)}.
}
\section{Slots}{
  \describe{
    \item{\code{edgeID}:}{Object of class \code{"integer"} a unique integer 
           identifier for the edge. }
    \item{\code{edgeType}:}{Object of class \code{"character"} the type of 
         edge. }
    \item{\code{weight}:}{Object of class \code{"numeric"} an edge weight. }
    \item{\code{bNode}:}{Object of class \code{"integer"} a beginning
      node for the edge. }
    \item{\code{eNode}:}{Object of class \code{"integer"} an ending node for
          the edge. }
  }
}

\section{Methods}{
No methods defined with class "gEdge" in the signature.
}
\author{R. Gentleman}
\note{Based on GXL's definition of edges.}
\seealso{ \code{\link{gNode-class}}}

\examples{
##---- Should be DIRECTLY executable !! ----
}
\keyword{classes}

\eof
\name{gNode-class}
\docType{class}
\alias{gNode-class}
\alias{edgeOrder}
\alias{fromEdges}
\alias{toEdges}
\alias{fromEdges<-}
\alias{label}
\alias{nodeID}
\alias{nodeType}
\alias{toEdges<-}
\alias{edgeOrder,gNode-method}
\alias{fromEdges,gNode-method}
\alias{toEdges,gNode-method}
\alias{fromEdges<-,gNode-method}
\alias{label,gNode-method}
\alias{nodeID,gNode-method}
\alias{nodeType,gNode-method}
\alias{toEdges<-,gNode-method}
\title{Class "gNode"}
\description{This is an implemention of a very general graph node class.
It is patterned after the descriptions in GXL. }
\section{Objects from the Class}{
Objects can be created by calls of the form \code{new("gNode", ...)}.
These are nodes of a graph and can have any number of attributes.
}
\section{Slots}{
  \describe{
    \item{\code{nodeID}:}{Object of class \code{"integer"} a unique
      integer identifier.} 
    \item{\code{nodeType}:}{Object of class \code{"character"} the type
      of node. Could be a factor. }
    \item{\code{toEdges}:}{Object of class \code{"integer"} A vector of
      the edge identifiers corresponding to the edges pointing to that node. }
    \item{\code{fromEdges}:}{Object of class \code{"integer"} A vector
      of the edge identifiers corresponding to the edges starting at
      that node. }
    \item{\code{edgeOrder}:}{Object of class \code{"integer"} A vector
      of edge identifiers indicating the order in which they would be
      traversed (may also be used for rendering). }
    \item{\code{label}:}{Object of class \code{"character"} A label of
      the node. Should be unique. }
  }
}

\section{Methods}{
  \describe{
    \item{edgeOrder}{\code{signature(object = "gNode")}: A method that
      returns the \code{edgeOrder} for the named node.}
    \item{fromEdges}{\code{signature(object = "gNode")}: A method that
      returns the \code{fromEdges}. }
    \item{fromEdges<-}{\code{signature(object = "gNode")}: A method that
    assigns the \code{fromEdges}}
    \item{label}{\code{signature(object = "gNode")}:A method that
      returns the node label. }
    \item{nodeID}{\code{signature(object = "gNode")}: A method that
      returns the node identifier. }
    \item{nodeType}{\code{signature(object = "gNode")}: A method that
      returns the node type. }
    \item{toEdges}{\code{signature(object = "gNode")}: A method that
      returns the \code{toEdges}. }
    \item{toEdges<-}{\code{signature(object = "gNode")}: A method that
      assigns the \code{toEdges}. }
  }
}
\author{R. Gentleman }

\seealso{\code{\link{distGraph-class}},
 \code{\link{graphNEL-class}}}

\examples{
  ##none yet
}
\keyword{classes}

\eof
\name{generalGraph-class}
\docType{class}
\alias{generalGraph-class}
\alias{coerce,generalGraph-method}
\alias{initialize,generalGraph-method}
\title{Class "generalGraph" ~~~ }
\description{  ~~ A concise (1-5 lines) description of what the class is.  ~~}
\section{Objects from the Class}{
Objects can be created by calls of the form \code{new("generalGraph", nodes, edges)}.
}
\section{Slots}{
  \describe{
    \item{\code{nodes}:}{Object of class \code{"hashtable"} }
    \item{\code{edges}:}{Object of class \code{"hashtable"} }
    \item{\code{edgemode}:}{Object of class \code{"character", from class "graph"} }
  }
}
\section{Extends}{
Class \code{"graph"}, directly.
}
\section{Methods}{
  \describe{
    \item{coerce}{\code{signature(from = "graphNEL", to = "generalGraph")}: ... }
    \item{initialize}{\code{signature(.Object = "generalGraph")}: ... }
  }
}
\author{R. Gentleman }


\seealso{  }

\examples{
##---- Should be DIRECTLY executable !! ----
}
\keyword{classes}

\eof
\name{graph-class}
\docType{class}
\alias{graph-class}
\alias{complement}
\alias{connComp}
\alias{degree}
\alias{degree}
\alias{edgemode}
\alias{edgemode<-}
\alias{intersection}
\alias{isConnected}
\alias{join}
\alias{numNodes}
\alias{union}
\alias{complement,graph-method}
\alias{connComp,graph-method}
\alias{degree,graph,missing-method}
\alias{degree,graph,ANY-method}
\alias{dfs,graph-method}
\alias{edgemode,graph-method}
\alias{edgemode<-,graph,character-method}
\alias{intersection,graph,graph-method}
\alias{isConnected,graph-method}
\alias{join,graph,graph-method}
\alias{numNodes,graph-method}
\alias{union,graph,graph-method}
\title{Class "graph" }
\description{A virtual class that all graph classes should extend. }
\section{Objects from the Class}
{A virtual Class: No objects may be created from it.}
\section{Slots}{
  \describe{
    \item{\code{edgemode}:}{Indicates whether edges are
      \code{"directed"} or \code{"undirected"}}
  }
}

\section{Methods}{
  \describe{
    \item{acc}{\code{signature(object = "graph")}: find all nodes
      accessible from the specified node. }

        \item{complement}{\code{signature(x = "graph")}: compute the
      complement of the supplied graph. The complement is defined with
      respect to the complete graph on the nodes in \code{x}.
      Currently this returns an object of class \code{graphNEL}.}

    \item{connComp}{\code{signature(object = "graph")}: find the
      connected components of a graph.}


    \item{degree}{\code{signature(object = "graph", Nodes = "missing")}:
    find the \code{degree} of a node (number of coincident edges).}

  \item{degree}{\code{signature(object = "graph", Nodes = "ANY")}: as
      above.  }

    \item{dfs}{\code{signature(object = "graph")}: execute a depth first
    search on a graph starting with the specified node.}

      \item{edges}{\code{signature(object="graph", which="character")}:
      return the edges indicated by \code{which}. \code{which} can be
      missing in which case all edges are returned or it can be a
      character vector with the node labels indicating the nodes whose
      edge lists are wanted.}

    \item{edgemode}{\code{signature(object="graph")}: return the
      \code{edgemode} for the graph. Currently this can be either
      \code{directed} or \code{undirected}.}
    \item{edgemode<-}{\code{signature(object="graph",
	value="character")}: set the \code{edgemode}
      for the graph. Currently this can be either
      \code{directed} or \code{undirected}.}

  \item{intersection}{\code{signature(x = "graph", y = "graph")}: compute the
      intersection of the two supplied graphs. They must have identical
      nodes. Currently this returns an object of class
      \code{graphNEL}. With edge weights of 1 for any matching edge.}

    \item{isConnected}{\code{signature(object = "graph")}: A boolean
      that details if a graph is fully connected or not.}

    \item{join}{\code{signature(x = "graph", y = "graph")}: returns the
      joining of two graphs.  Nodes which are shared by both graphs will
      have their edges merged.}

    \item{nodes<-}{A generic function that allows different
      implementations of the \code{graph} class to reset the node labels}

    \item{numNodes}{\code{signature(object = "graph")}: compute the
      number of nodes in a graph. }

    \item{plot}{Please see the help page for the \code{plot.graph}
      method in the \code{Rgraphviz} package}

    \item{union}{\code{signature(x = "graph", y = "graph")}: compute the
      union of the two supplied graphs. They must have identical
      nodes. Currently this returns an object of class \code{graphNEL}.}

    
  }
}

\details{
  \code{degree} returns either a named vector (names correspond to the
  nodes in the graph) containing the degree for undirected graphs or a
  list with two components, \code{inDegree} and \code{outDegree} for
  directed graphs.

  \code{connComp} returns a list of the connected components. Each
  element of this list contains the labels of all nodes in that
  component.

  For a \emph{directed graph} or \emph{digraph} the \dfn{underlying
    graph} is the graph that results from removing all direction from
  the edges. This can be achieved using the function \code{\link{ugraph}}.

  A \dfn{weakly connected} component of a \emph{digraph} is one that is
  a connected component of the underlying graph. This is the default for
  \code{connComp}. A \emph{digraph} is  \dfn{strongly connected} if
  every two vertices are mutually reachable. A \dfn{strongly connected}
  component of a \emph{digraph}, \strong{D}, is a maximal \emph{strongly
  connected} subdigraph of \strong{D}. See the \pkg{RBGL} package for an
  implementation of Trajan's algorithm to find \emph{strongly
  connected} components (\code{\link{strongComp}}).

  In the \pkg{graph} implementation of \code{connComp} \emph{weak
    connectivity} is used. If the argument to \code{connComp} is a
  directed graph then \code{\link{ugraph}} is called to create the
  underlying undirected graph and that is used to compute connected
  components. Users who want different behavior are encouraged to use
  \pkg{RBGL}.
}

\references{ Graph Theory and its Applications, J. Gross and J. Yellen. }
\author{R. Gentleman and E. Whalen. }

\seealso{\code{\link{graphNEL-class}}, \code{\link{distGraph-class}} }

\examples{
  set.seed(123)
  g1 <- randomGraph(letters[1:10], 1:4, p=.3)
  edges(g1)
  edges(g1, "a")

}
\keyword{classes}

\eof
\name{graph2SparseM}
\alias{graph2SparseM}
\alias{sparseM2Graph}
\title{Coercion methods between graphs and sparse matrices  }
\description{
  These functions provide coercions between objects that inherit from
  the \code{graph} class to sparse matrices from the \code{SparseM}
  package. 
}
\usage{
graph2SparseM(g, useweights=FALSE)
sparseM2Graph(sM, nodeNames) 

}
\arguments{
  \item{g}{An instance of the \code{graph} class. }
  \item{useweights}{A logical value indicating whether to use the edge
    weights in the graph as values in the sparse matrix.}
  \item{sM}{A sparse matrix.}
  \item{nodeNames}{A \code{character} vector of the node names.}
}
\details{
  A very simple coercion from one representation to another.

  Currently it is presumed that the matrix is square. For other graph
  formats, such as bipartite graphs, some improvements will be needed;
  patches are welcome.
  
}
\value{
  \code{graph2SparseM} takes as input an instance of the \code{graph}
  class and returns a sparse matrix while \code{sparseM2Graph} takes a
  sparse matrix as input and returns an in instance of the graph class.
}

\author{R. Gentleman }

\seealso{\code{\link{graph-class}} And for other conversions,
\code{\link{aM2bpG}} and \code{\link{ftM2adjM}} }

\examples{
  set.seed(123)
  g1 <- randomGraph(letters[1:10], 1:4, p=.3)
  s1 <- graph2SparseM(g1)

  sparseM2Graph(s1, letters[1:10])

}
\keyword{manip}

\eof
\name{graphExamples}
\alias{graphExamples}
\docType{data}
\title{A List Of Example Graphs}
\description{
This data set contains a list of example \code{graphNEL} objects, which can then
be used for plotting.
}
\usage{data(graphExamples)}
\source{
  Various sources, primarily from \code{\link{randomGraph}} and
  \code{\link{randomEGraph}} 
}
\examples{
data(graphExamples)
a <- graphExamples[[1]]
a
}
\keyword{datasets}

\eof
\name{graphID-class}
\docType{class}
\alias{graphID-class}
\title{Unique Graph Identifications}
\description{This class allows nodes and edges to have unique
  identifiers}
\author{Robert Gentleman}
\seealso{\code{\link[Ruuid]{Ruuid-class}} in package \code{Ruuid}.}

%\examples{
%##---- Should be DIRECTLY executable !! ----
%}
\keyword{classes}

\eof
\name{graphNEL-class}
\docType{class}
\alias{graphNEL-class}
\alias{coerce}
\alias{edgeL}
\alias{edges}
\alias{edges}
\alias{initialize}
\alias{nodes<-}
\alias{nodes}
\alias{show}
\alias{addEdge,character,character,graphNEL,numeric-method}
\alias{addNode,character,graphNEL-method}
\alias{adj,graphNEL-method}
\alias{clearNode,character,graphNEL-method}
\alias{coerce,graphNEL,generalGraph-method}
\alias{combineNodes,character,graphNEL,character-method}
\alias{edgeL,graphNEL-method}
\alias{edges,graphNEL,missing-method}
\alias{edges,graphNEL,character-method}
\alias{edgeWeights,graphNEL-method}
\alias{inEdges,missing,graphNEL-method}
\alias{inEdges,character,graphNEL-method}
\alias{initialize,graphNEL-method}
\alias{nodes<-,graphNEL,character-method}
\alias{nodes,graphNEL-method}
\alias{numNodes,graphNEL-method}
\alias{removeEdge,character,character,graphNEL-method}
\alias{removeNode,character,graphNEL-method}
\alias{show,graphNEL-method}
\alias{toGXL,graphNEL-method}
\title{Class "graphNEL"}
\description{
  This is a class of graphs that are represented in terms of nodes and
  an edge list. This is a suitable representation for a graph with a
  large number of nodes and relatively few edges.}
\section{Objects from the Class}{
  Objects can be created by calls of the form
  \code{new("graphNEL",...)}.
}
\section{Slots}{
  \describe{
    \item{\code{nodes}:}{Object of class \code{"vector"}.}
    \item{\code{edgeL}:}{Object of class \code{"list"}. The \code{edgeL}
      must be the same length as \code{nodes}. The elements of this
      vector correspond to the same element in \code{nodes}. The
      elements are themselves lists. If the node has any edges then this
      list will have an element named \code{edges}. If edge weights are
      used then there must be an element named \code{weights} and it
      must be the same length as the \code{edges} element.
    }
  }
}
\section{Extends}{
Class \code{"graph"}, directly.
}
\section{Methods}{
  \describe{
    \item{show}{\code{signature(object = "graphNEL")}:A print method for
    the graph.}
    \item{adj}{\code{signature(object = "graphNEL")}: A method for
      finding nodes adjacent to the suplied node.}
    \item{edgeL}{\code{signature(graph = "graphNEL")}: A mehtod for
      obtaining the edge list.}
    \item{edgeWeights}{\code{signature(object = "graphNEL")}: A method
      for obtaining the edge weights. }
    \item{edges}{\code{signature(object = "graphNEL")}: A method for
      obtaining the edges.}
    \item{nodes}{\code{signature(object = "graphNEL")}: A method for
      obtaining the nodes. }
    \item{numNodes}{\code{signature(object = "graphNEL")}:A method for
      determining how many nodes are in the graph. }
    \item{subGraph}{\code{signature(subgraph="character", graph =
	"graphNEL")}:A method for
      obtaining the induced subgraph based on the set of supplied nodes
      and the supplied graph.}
    \item{plot}{Please see the help page for \code{plot.graphNEL} in the
      \code{Rgraphviz} package}
    \item{graph2graphviz}{\code{signature(object = "graphNEL")}: A method
      that will convert a \code{graphNEL} object into a matrix suitable
      for interaction with \code{Rgraphviz}.  Not intended to be called
      directly.  This function will insure that no NA's (or other
      undesired values) are in the graph, or created by coersion.}
    \item{nodes<-}{\code{signature(object="graphNEL",
	value="character")}: A method for replacing the nodes in a graph
      object. It checks to be sure the values are the right length and
      unique. }
  }
}

\details{
   The \code{graphNEL} class provides a very general structure for
   representing graphs. It will be reasonably efficient for lists with
   relatively more nodes than edges. The class is currently able to
   handle multi-edges and edges of any type.

   The \code{edgeL} is a named \code{list} of the same length as the
   node vector. The names are the names of the nodes. Each element of
   \code{edgeL} is itself a list. Each element of this (sub)list is a
   vector (all must be the same length) and each element represents an
   edge to another node. The sublist named \code{edges} holds index
   values into the node vector. And each such entry represents an edge
   from the node which has the same name as the component of
   \code{edgeL} to the node with index provided. Another component that
   is often used is named \code{weights}. It represents edge weights.
   The user can specify any other edge attributes (such as types
   etc). They are responsible for any special handling that
   these might require.

   Multi-edges are represented by multiple entries. For an
   \code{undirected} instance all edges are reciprocated (there is an
   edge from A to B and from B to A).

   Note that the reason for using indices to represent the \code{to} end
   of a node is so that we can easily support permutation of the node
   labels as a way to generate randomizations of the graph.

  }
\author{R. Gentleman}

\seealso{\code{\link{distGraph-class}},
  \code{\link{clusterGraph-class}} } 

\examples{
   set.seed(123)
   V <- LETTERS[1:4]
   edL <- vector("list", length=4)
   names(edL) <- V
   toE <- LETTERS[4:1]
   for(i in 1:4) 
      edL[[i]] <- list(edges=5-i, weights=runif(1))
   gR <- new("graphNEL", nodes=V, edgeL=edL)
   edges(gR)
   edgeWeights(gR)

}
\keyword{classes}

\eof
\name{hashtable-class}
\docType{class}
\alias{hashtable-class}
\alias{hash}
\alias{contents}
\alias{hash,ANY,ANY,hashtable-method}
\alias{contents,hashtable,ANY-method}
\alias{contents,hashtable-method}
\alias{initialize,hashtable-method}
\title{Class "hashtable"}
\description{ A preliminary implementation of a hash table class}
\section{Objects from the Class}{
Objects can be created by calls of the form \code{new("hashtable", ...)}.
}
\section{Slots}{
  \describe{
    \item{\code{hashtable}:}{Object of class \code{"environment"}. }
  }
}

\section{Methods}{
  \describe{
    \item{contents}{\code{signature(object = "hashtable")}: ... }
    \item{hash}{\code{signature(key = "ANY", value = "ANY", htable =
	"hashtable")}: ... } 
    \item{initialize}{\code{signature(.Object = "hashtable")}: ... }
  }
}
\author{R.Gentleman }


\seealso{  }

\examples{
##---- Should be DIRECTLY executable !! ----
}
\keyword{classes}

\eof
\name{inEdges}
\alias{inEdges}
\alias{inEdges,graphNEL,missing-method}
\title{ inEdges }
\description{
  Computes and returns a list of all the edges in to the specified set
  of nodes.
}
\usage{
inEdges(node, object)
}
\arguments{
  \item{node}{a set of nodes }
  \item{object}{a \code{graph} object }
}
\details{
  All in edges for the set of nodes are computed. For an undirected graph
  this is also all the out edges.
}
\value{
  A list with one element for each node, the values are the vector of
  node labels for those nodes with edges to the specified node.
}
\author{R. Gentleman}

\seealso{\code{\link{removeNode}}, \code{\link{clearNode}}}

\examples{

  V <- LETTERS[1:4]
  edL3 <- vector("list", length=4)
  for(i in 1:4)
    edL3[[i]] <- list(edges=(i\%\%4)+1, weights=i)
  names(edL3) <- V
  gR3 <- new("graphNEL", nodes=V, edgeL=edL3, "directed")
  inEdges(c("A", "B"), gR3)

}
\keyword{ manip}

\eof
\name{listEdges}
\alias{listEdges}
\title{List the Edges of a Graph  }
\description{
  A list where each element contains all edges between two nodes,
  regardless of orientation. The list has names which are node pairs,
  in lexicographic order, and elements all edges between those nodes.
}
\usage{
listEdges(object, dropNULL=TRUE)
}
\arguments{
  \item{object}{An instance of the \code{\link{graphNEL-class}} class. }
  \item{dropNULL}{Should those node pairs with no edges be dropped from
    the returned list. }
}
\details{
  The function is currently only implemented for graphs of the
  \code{\link{graphNEL-class}}. The edges in the returned list are
  instances of the \code{\link{simpleEdge-class}}.
}
\value{
  A named list of \code{\link{simpleEdge-class}} objects. 
}
\author{R. Gentleman}

\seealso{\code{\link{simpleEdge-class}}, \code{\link{edges}}}

\examples{
        set.seed(123)
        V <- LETTERS[1:4]
        edL <- vector("list", length=4)
        names(edL) <- V
        toE <- LETTERS[4:1]
        for(i in 1:4) 
           edL[[i]] <- list(edges=5-i, weights=runif(1))
        gR <- new("graphNEL", nodes=V, edgeL=edL)
        listEdges(gR)
}
\keyword{manip}

\eof
\name{Matrics to Graph coersions}
\alias{aM2bpG}
\alias{ftM2adjM}
\alias{coerce,matrix,graphNEL-method}

\title{Coercing matrices to graphs}
\description{
A collection of functions and methods to convert various forms of
matrices into graph objects.
}
\usage{
aM2bpG(aM)
ftM2adjM(ft, W=NULL, V=NULL, edgemode="directed")
}
\arguments{
  \item{ft}{A matrix with two columns containing the from/to
    representation of graph edges.}
  \item{W}{An optional vector of edge weights.}
  \item{V}{An optional vector of node names.}
  \item{aM}{An affiliation matrix for a bipartite graph.}
  \item{edgemode}{Specifies if the resulting graph is to be directed or undirected}
}
\details{

In the fuction \code{ftM2adjM}, a \code{from/to} matrix is converted
into an \code{adjacency} matrix (which can then be coerced directly into
a \code{\link{graphNEL-class}} with \code{as}.  Ths first column of the
\code{ft} represents the \code{from} nodes and the second column
represents \code{to} nodes.  This representation does not allow for
unconnected nodes except with the \code{V} argument (see below).  The
\code{edgemode} parameter can be used to specify if the desired output
is a directed or undirected graph. 
  
Also in \code{ftM2adjM}, \code{W} is an optional vector of edge weights.
The order of the edge weights in the vector should correspond to the
order of the edges recorded in \code{L}.  If it is not specified, edge
weights of 1 are assigned by default.  The \code{V} argument is an
optional vector of node names.  All nodes in \code{ft} must be contained
in \code{V}, but not all nodes in \code{V} must be contained in
\code{ft}.  If \code{V} is not specified, it is set to all nodes
represented in \code{ft} or \code{M}.  Specifying \code{V} is most useful
for creating a graph that includes nodes with degree 0. 

\code{aM} is an affiliation matrix as frequently used in social networks
analysis.  The rows of \code{aM} represent actors, and the columns
represent events.  An entry of "1" in the ith row and jth column
represents affiliation of the ith actor with the jth event.  Weighted
entries may also be used.  \code{aM2bpG} returns a \code{graphNEL} object with
nodes consisting of the set of actors and events, and directed (possibly
weighted) edges from the actors to their corresponding events.  If
plotted using \code{Rgraphviz} and the \code{dot} layout, the bipartite structure of
the graph returned by \code{aM2bpG} should be evident. 

An \code{adjacency} matrix can be directly coerced into a
\code{graphNEL} using the \code{as} method.  If the matrix is a
symmetric matrix, then the resulting graph will be \code{undirected},
otherwise it will be \code{directed}.

}
\value{
  For \code{aM2bpG}, an object of class \code{graphNEL}.  For
  \code{ftM2adjM}, an adjacency matrix representation of the coerced graph.
}

\author{Denise Scholtens}

\examples{
From <- c("A","A","C","C")
To <- c("B","C","A","D")
L <- cbind(From,To)

W <- 1:4
M1 <- ftM2adjM(L,W, edgemode="undirected")
M2 <- ftM2adjM(L)

G1 <- as(M1, "graphNEL")
G2 <- as(M2, "graphNEL")
}

\keyword{graphs }

\eof
\name{mostEdges}
\alias{mostEdges}

\title{ Find the node in a graph with the greatest number of edges }
\description{
  \code{mostEdges} finds the node that has the most edges in the graph.
 This is the node with the highest degree.
}
\usage{
mostEdges(objGraph)
}

\arguments{
  \item{objGraph}{ the graph object }
}
\value{
  \item{index}{the index of the node with the most edges}
  \item{id}{the node value with the most edges; may be affy id, locus
                link id, or genename depending on the node type}
  \item{maxLen}{the number of edges for that node}

}

\author{ Elizabeth Whalen }

\seealso{ \code{\link{numEdges}}, \code{\link{aveNumEdges}},
  \code{\link{numNoEdges}}
}

\examples{
  set.seed(123)
  g1 <- randomGraph(11:30, letters[20:26], p=.4)
  mostEdges(g1)
}
\keyword{ manip }


\eof
\name{numEdges}
\alias{numEdges}
\title{ Calculate the number of edges in a graph }
\description{
  \code{numEdges} counts the number of edges in the graph object.  
}
\usage{
numEdges(graph)
}

\arguments{
  \item{graph}{ the graph }
}

\value{
  An integer representing the number of edges will be returned.
}

\author{ Elizabeth Whalen }

\seealso{ \code{\link{aveNumEdges}}, \code{\link{mostEdges}}, 
   \code{\link{numNoEdges}} 
}

\examples{
 set.seed(333)
 g1 <- randomEGraph(letters[13:26], .2)
  numEdges(g1)
}
\keyword{ manip }


\eof
\name{numNoEdges}
\alias{numNoEdges}
\title{ Calculate the number of nodes that have an edge list of NULL }
\description{
  \code{numNoEdges} calculates the number of nodes that have an edge list
 of NULL (i.e. no edges). 
}
\usage{
numNoEdges(objGraph)
}

\arguments{
  \item{objGraph}{ the graph object }
}

\value{
  An integer representing the number of NULL edge lists in the graph.
}

\author{ Elizabeth Whalen }

\seealso{ \code{\link{numEdges}}, \code{\link{aveNumEdges}}, 
   \code{\link{mostEdges}} 
}

\examples{
  set.seed(999)
  g1 <- randomEGraph(letters, .01)
  numNoEdges(g1)
}
\keyword{ manip }

\eof
\name{randomEGraph}
\alias{randomEGraph}
\title{ Random Edge Graph  }
\description{
  A function to create random graphs according to a random edge model.
  The user supplies the set of nodes for the graph as \code{V} and
  either a probability, \code{p}, that is used for each edge or the
  number of edges, \code{edges} they want to have in the resulting graph.
}
\usage{
randomEGraph(V, p, edges)
}
\arguments{
  \item{V}{The nodes for the graph. }
  \item{p}{ The probability of an edge being selected. }
  \item{edges}{ The number of edges wanted. }
}
\details{
  The user must specify the set of nodes and either a probability for
  edge selection or the number of edges wanted, but not both.
   Let \code{nV} denote the
  number of nodes. There are \code{choose(nV, 2)} edges in the complete
  graph. If \code{p} is specified then a biased coin (probability of
  heads being \code{p}) is tossed for each edge and if it is heads that
  edge is selected. If \code{edges} is specified then that many edges
  are sampled without replacement from the set of possible edges.
}
\value{
  An object of class \code{\link{graphNEL-class}} that contains the nodes and
  edges. 
}
\author{R. Gentleman }

\seealso{\code{\link{randomGraph}}}

\examples{
  set.seed(123)
  V <- letters[14:22]
  g1 <- randomEGraph(V, .2)

  g2 <- randomEGraph(V, edges=30)
}
\keyword{ manip }

\eof
\name{randomGraph}
\alias{randomGraph}
\title{ Random Graph }
\description{
 This function generates a random graph according to a model that
 involves a latent variable. The construction is to randomly assign
 members of the set \code{M} to the nodes, \code{V}. An edge is assigned
 between two elements of \code{V} when they both have the same element
 of \code{M} assigned to them. An object of class \code{graphNEL} is
 returned. 
}
\usage{
randomGraph(V, M, p, weights=TRUE)
}
\arguments{
  \item{V}{The nodes of the graph. }
  \item{M}{A set of values used to generate the graph. }
  \item{p}{A value between 0 and 1 that indicates the probability of
    selecting an element of \code{M}}
  \item{weights}{A logical indicating whether to use the number of
    shared elements of \code{M} as weights. }
}
\details{
  The model is quite simple. To generate a graph, \code{G}, the user
  supplies the list of nodes, \code{V} and a set of values \code{M}
  which will be used to create the graph. For each node in \code{V} a
  logical vector with length equal to the length of \code{M} is
  generated. The probability of a \code{TRUE} at any position is
  determined by \code{p}. Once valus from \code{M} have been assigned to
  each node in \code{V} the result is processed into a graph. This is
  done by creating an edge between any two elements of \code{V} that
  share an element of \code{M} (as chosen by the selection process).

  The sizes of \code{V} and \code{M} and the values of \code{p}
  determine how dense the graph will be.
}
\value{
An object of class \code{\link{graphNEL-class}} is returned.
}
\author{R. Gentleman }

\seealso{\code{\link{randomEGraph}}}

\examples{
   set.seed(123)
   V <- letters[1:10]
   M <- 1:4
   g1 <- randomGraph(V, M, .2)
}
\keyword{ manip}

\eof
\name{randomNodeGraph}
\alias{randomNodeGraph}
\title{Generate a Graph with specified degree distribution.  }
\description{
  \code{randomNodeGraph} generates a random graph with the specified
  degree distribution. Self-loops are allowed. The resultant graph is
  directed (but can always be coerced to be undirected).
}
\usage{
randomNodeGraph(nodeDegree)
}
\arguments{
  \item{nodeDegree}{A named integer vector specifying the node degrees. }
}
\details{
  The input vector must be named, the names are taken to be the names of
  the nodes. The sum must be even (there is a theorem that says we
  require that to construct a graph). Self-loops are allowed, although
  patches to the code that make this a switchable parameter would be
  welcome. 
}
\value{
  An instance of the \code{graphNEL} class. The graph is directed.
  
}
\references{Random Graphs as Models of Networks, M. E. J. Newman. }
\author{R. Gentleman}

\seealso{\code{\link{randomGraph}},\code{\link{randomEGraph}}}

\examples{
set.seed(123)
c1 <- c(1,1,2,4)
names(c1) <- letters[1:4]

g1 <- randomNodeGraph(c1)
}
\keyword{manip }

\eof
\name{removeEdge}
\alias{removeEdge}
\title{ removeEdge }
\description{
  A function to remove the specified edges from a graph.
}
\usage{
removeEdge(from, to, graph)
}
\arguments{
  \item{from}{from edge labels }
  \item{to}{ to edge labels}
  \item{graph}{a \code{graph} object }
}
\details{
  The edges in the graph from nodes in the \code{from} set to the
  \code{to} set are removed and a new graph object returned.
  Both \code{from} and \code{to} can be vectors, not necessarily of the
  same length (in which case standard rules for replicating the shorter
  one apply).

  If the graph is \code{undirected} then the undirected edge is removed.
  
}
\value{
  A new instance of a graph with the same class as \code{graph} is
  returned. Any of the listed edges in \code{graph} have been removed.
}
\author{R. Gentleman}

\seealso{\code{\link{addNode}},\code{\link{addEdge}},\code{\link{removeNode}},}

\examples{
  V <- LETTERS[1:4]
  edL1 <- vector("list", length=4)
  names(edL1) <- V
  for(i in 1:4)
    edL1[[i]] <- list(edges=c(2,1,4,3)[i], weights=sqrt(i))
  gR <- new("graphNEL", nodes=V, edgeL=edL1)

  gX <- removeEdge("A", "B", gR)
}
\keyword{manip}

\eof
\name{removeNode}
\alias{removeNode}
\title{ removeNode }
\description{
  A function to remove a node from a graph. All edges to and from the
  node are also removed.
}
\usage{
removeNode(node, object)
}
\arguments{
  \item{node}{The label of the node to be removed. }
  \item{object}{The graph to remove the node from. }
}
\details{
  The specified node is removed from the graph as are all edges to and
  from that node. A new instance of the same class as \code{object} with
  the specified node(s) is returned.

  Note, node can be a vector of labels, in which case all nodes are
  removed.

  This is similar to \code{\link{subGraph}}.
}
\value{
  A new instance of a graph of the same class as \code{object} but with
  all specified nodes removed.
  
}
\author{R. Gentleman}
\seealso{\code{\link{removeEdge}}, \code{\link{addEdge}},
  \code{\link{addNode}},\code{\link{subGraph}} }

\examples{
  V <- LETTERS[1:4]
  edL2 <- vector("list", length=4)
  names(edL2) <- V
  for(i in 1:4)
    edL2[[i]] <- list(edges=c(2,1,2,1)[i], weights=sqrt(i))
  gR2 <- new("graphNEL", nodes=V, edgeL=edL2)
  gX <- removeNode("C", gR2)

}
\keyword{manip}

\eof
\name{simpleEdge-class}
\docType{class}
\alias{simpleEdge-class}

\title{Class "simpleEdge".}
\description{A simple class for representing edges in graphs.}
\section{Objects from the Class}{
Objects can be created by calls of the form \code{new("simpleEdge", ...)}.
}
\section{Slots}{
  \describe{
    \item{\code{edgeType}:}{Object of class \code{"character"}; the type
    of the edge.}
    \item{\code{weight}:}{Object of class \code{"numeric"}; the edge weight.}
    \item{\code{directed}:}{Object of class \code{"logical"}; is the
      edge directed. }
    \item{\code{bNode}:}{Object of class \code{"character"}; the
      beginning node. }
    \item{\code{eNode}:}{Object of class \code{"character"}; the ending node. }
  }
}
\section{Methods}{
No methods defined with class "simpleEdge" in the signature.
}
\author{R. Gentleman}
\note{All slots are length one vectors (this is not currently checked
  for). If the edge is not directed there is no real meaning to the
  concepts of beginning node or ending node and these should not be
  interpreted as such. }

\seealso{
 See also\code{\link{gEdge-class}}}
\examples{
  new("simpleEdge", bNode="A", eNode="D")
}
\keyword{classes}

\eof
\name{startids}
\alias{startids}
\alias{newID}
\alias{nullgraphID}
\alias{idenv}
\title{Functions to associate identifiers with graph components.}
\description{
  We are experimenting with different methods for dealing with
  components of graphs (such as subgraphs). One way to do this is to
  assign a unique identifier to each component of a graph. Subgraphs
  then need only retain the identity of the graph and the ids for the
  selected components.

  The intention is to have these be internal to the graph package and
  they will likely not be exported. They are primarily for housekeeping
  purposes.

  There are currently two ways of assigning IDs. One relies on the
  \code{Ruuid} package the other on integers. In the long run only
  \code{Ruuid} will be supported.
}
\usage{
startids(x)
newID(cur)
}
\arguments{
  \item{x}{an integer to start numbering components from.}
  \item{cur}{an integer, the current index}

}
\details{
  \code{startids} sets the starting value for the sequence. It has
}
\value{
  These are used basically just for side effects and internally. Users
  should not use them. If you want uuids please use \code{Ruuid}
  directly. 
}

\author{R. Gentleman }
\examples{

}
\keyword{ manip }


\eof
\name{subGraph}
\alias{subGraph}
\alias{subGraph,character,graphNEL-method}
\alias{subGraph,character,clusterGraph-method}
\alias{subGraph,character,distGraph-method}
\title{Create a Subgraph  }
\description{
  Given a set of nodes and a graph this function creates
  and returns subgraph with only the supplied nodes and any edges
  between them.
}
\usage{
subGraph(snodes, graph)
}
\arguments{
  \item{snodes}{A \code{character} vector of node labels. }
  \item{graph}{A graph object, it must inherit from the \code{graph} class. }
}
\details{
  The returned subgraph is a copy of the graph. Implementations for
  Implementations for \code{graphNEL}, \code{distGraph} and
  \code{clusterGraph}. 
}
\value{
  A graph of the same class as the \code{graph} argument but with only
  the supplied nodes.
}
\author{R. Gentleman }

\seealso{ \code{\link{nodes}},\code{\link{edges}} }

\examples{
        set.seed(123)
        x <- rnorm(26)
        names(x) <- letters
        library(stats)
        d1 <- dist(x)
        g1 <- new("distGraph", Dist=d1)
        subGraph(letters[1:5], g1)

}
\keyword{manip}

\eof
\name{ugraph}
\alias{ugraph}
\title{Underlying Graph}
\description{
 For a \emph{directed} graph the underlying graph is the graph that is
 constructed where all edge orientation is ignored. This function
 carries out such a transformation on \code{graphNEL} instances.
}
\usage{
ugraph(graph)
}
\arguments{
  \item{graph}{An instance of \code{graphNEL}. }
}

\details{
   If \code{graph} is already \emph{undirected} then it is simply
   returned.

   If \code{graph} is a multi-graph (has multiple edges) an error is
   thrown as it is unclear how to compute the underlying graph in that
   context.
 }
 

\value{
  An instance of \code{graphNEL} with the same nodes as the input but
  which is \code{undirected}.
}
\references{Graph Theory and its Applications, J. Gross and J. Yellen. }
\author{R. Gentleman }

\seealso{\code{\link{connComp}}}

\examples{
V <- letters[1:4]
edL2 <- vector("list", length=4)
names(edL2) <- V
for(i in 1:4)
  edL2[[i]] <- list(edges=c(2,1,2,1)[i], weights=sqrt(i))
gR2 <- new("graphNEL", nodes=V, edgeL=edL2, edgemode="directed")

ugraph(gR2)

}
\keyword{manip}

\eof
\name{validGraph}
\alias{validGraph}
\title{ Test whether graph object is valid }
\description{
  validGraph is a validating function for a graph object.
}
\usage{
validGraph(object, quietly=FALSE)
}

\arguments{
  \item{object}{ a graph object to be tested }
  \item{quietly}{ \code{TRUE} or \code{FALSE} indicating whether 
  output should be printed.}
}
}

\value{
 If the graph object is valid, \code{TRUE} is returned otherwise
 \code{FALSE} is returned. If \code{object} is not a valid graph and
 \code{quietly} is set to \code{FALSE} then descriptions of the problems
 are printed.
}

\author{ Elizabeth Whalen }

\seealso{\code{\link{graph-class}} }

\examples{
  
  testGraph<-new("graphNEL")
  testGraph@nodes<-c("node1","node2","node3")
  testGraph@edgeL<-list(node1=c("node2"),node2=c("node1","node3"),
                        node3=c("node2"))
  validGraph(testGraph)

}
\keyword{ manip}

\eof
\name{write.tlp}
\alias{write.tlp}
\title{ Write a graph object in a file in the Tulip format}
\description{
  Write a graph object in a file in the Tulip format.
}
\usage{
write.tlp(graph, filename)
}
%- maybe also 'usage' for other objects documented here.
\arguments{
  \item{graph}{a \code{graph} object}
  \item{filename}{ Name of the output file}
}
\details{
  The Tulip format is used by the program Tulip.
}
\references{http://www.tulip-software.org/}
\author{ Laurent Gautier <laurent@cbs.dtu.dk>}
}
\keyword{ manip }


\eof
