\name{aCGH}
\alias{aCGH}
\alias{create.aCGH}
\alias{log2.ratios}
\alias{clones.info}
\alias{is.aCGH}
\alias{dim.aCGH}
\alias{num.clones}
\alias{nrow.aCGH}
\alias{num.samples}
\alias{num.chromosomes}
\alias{ncol.aCGH}
\alias{clone.names}
\alias{row.names.aCGH}
\alias{rownames.aCGH}
\alias{clone.names<-}
\alias{row.names<-.aCGH}
\alias{rownames<-.aCGH}
\alias{col.names.aCGH}
\alias{col.names<-.aCGH}
\alias{colnames.aCGH}
\alias{colnames<-.aCGH}
\alias{sample.names}
\alias{sample.names<-}
\alias{log2.ratios.imputed}
\alias{log2.ratios.imputed<-}
\alias{hmm}
\alias{hmm<-}
\alias{sd.samples}
\alias{sd.samples<-}
\alias{genomic.events}
\alias{genomic.events<-}
\alias{phenotype}
\alias{phenotype<-}
\alias{[.aCGH}
\alias{print.aCGH}
\alias{summary.aCGH}
\alias{plot.aCGH}
\alias{minna}
\alias{maxna}
\alias{corna}
\alias{floor.func}
\alias{length.num.func}
\alias{prop.num.func}
\alias{subset.hmm}

\title{Class aCGH}
\description{Objects of this class represent batch of arrays of
  Comparative Genomic Hybridization data. In addition to that, there are
  slots for representing phenotype and various genomic events associated
  with aCGH experiments, such as transitions, amplifications,
  aberrations, and whole chromosomal gains and losses.
  Currently objects of class aCGH are represented as S3 classes which
  are named list of lists with functions for accessing elements of that
  list. In the future, it's anticipated that aCGH objects will be
  implemented using S4 classes and methods.
}
%- maybe also 'usage' for other objects documented here.
\details{
  One way of creating objects of class aCGH is to provide the two
  mandatory arguments to \code{create.aCGH} function: \code{log2.ratios}
  and \code{clones.info}. Alternatively aCGH object can be created using
  \code{\link{aCGH.read.Sprocs}} that reads Sproc data files and
  creates object of type aCGH.
}
\note{
  \code{clones.info} slot has to contain a list with at least
  4 columns: Clone (clone name), Target (unique ID, e.g. Well ID), Chrom
  (chromosome number, X chromosome = 23 in human and 20 in mouse, Y
  chromosome = 24 in human and 21 in mouse) and kb (kb position on the
  chromosome).
}
\value{
  
  \item{log2.ratios}{
    Data frame containing the log2 ratios of copy number changes; rows
    correspond to the clones and the columns to the samples (Mandatory).
  }
  \item{clones.info}{
    Data frame containing information about the clones used for
    comparative genomic hybridization. The number of rows of
    \code{clones.info} has to match the number of rows in
    \code{log2.ratios} (Mandatory).
  }
  \item{phenotype}{
    Data frame containing phenotypic information about samples used in
    the experiment generating the data. The number of rows of
    \code{phenotype} has to match the number of columns in
    \code{log2.ratios} (Optional).
  }
  \item{log2.ratios.imputed}{
    Data frame containing the imputed log2 ratios. Calculate this using
    \code{impute.lowess} function; look at the examples below (Optional).
  }
  \item{hmm}{
    The structure of the hmm element is described in
    \code{\link{hmm}}. Calculate this using
    \code{\link{find.hmm.states}} function; look at the examples below
    (Optional).
  }
  \item{sd.samples}{
    The structure of the sd.samples element is described in
    \code{\link{computeSD.Samples}}. Calculate this using
    \code{\link{computeSD.Samples}} function; look at the examples
    below (Optional). It is prerequisite that the hmm states are
    estimated first.
  }
  \item{genomic.events}{
    The structure of the genomic.events element is described in
    \code{\link{find.genomic.events}}. Calculate this using
    \code{\link{find.genomic.events}} function; look also at the
    examples below. It is prerequisite that the hmm states and
    sd.samples are computed first. The \code{genomic.events} is used
    widely in variety of plotting functions such as
    \code{\link{plotHmmStates}}, \code{\link{plotFreqStat}}, and
    \code{\link{plotSummaryProfile}}.
  }
  \item{dim.aCGH}{
    returns the dimensions of the aCGH object: number of clones by
    number of samples.
  }
  \item{num.clones}{
    number of clones/number of rows of the log2.ratios data.frame.
  }
  \item{nrow.aCGH}{
    same as \code{num.clones}.
  }
  \item{is.aCGH}{
    tests if its argument is an object of class \code{aCGH}.
  }
  \item{num.samples}{
    number of samples/number of columns of the log2.ratios data.frame.
  }
  \item{nrow.aCGH}{
    same as \code{num.samples}.
  }
  \item{num.chromosomes}{
    number of chromosomes processed and stored in the aCGH object.
  }
  \item{clone.names}{
    returns the names of the clones stored in the clones.info slot of
    the aCGH object.
  }
  \item{row.names.aCGH}{
    same as \code{clone.names}.
  }
  \item{sample.names}{
    returns the names of the samples used to create the aCGH
    object. If the object is created using
    \code{\link{aCGH.read.Sprocs}}, these are the file names of the
    individual arrays.
  }
  \item{col.names.aCGH}{
    same as \code{sample.names}.
  }
  \item{[.aCGH}{
    subsetting function. Works the same way as \code{\link{[.data.frame}}.
  }
  Most of the functions/slots listed above have assignment operators
  '<-' associated with them.
  
}
\seealso{
  \code{\link{aCGH.read.Sprocs}}
  \code{\link{find.hmm.states}}
  \code{\link{computeSD.Samples}}
  \code{\link{find.genomic.events}}
  \code{\link{plotGenome}}
  \code{\link{plotHmmStates}}
  \code{\link{plotFreqStat}}
  \code{\link{plotSummaryProfile}}
}
\examples{

## Creating aCGH object from log2.ratios and clone info files
## For alternative way look at aCGH.read.Sprocs help

datadir <- system.file("data", package = "aCGH")
clones.info <-
      read.table(file = file.path(datadir, "clones.info.ex.csv"),
                 header = TRUE, sep = "\t")
log2.ratios <-
      read.table(file = file.path(datadir, "log2.ratios.ex.csv"),
                 header = TRUE, sep = "\t")
pheno.type <-
      read.table(file = file.path(datadir, "pheno.type.ex.csv"),
                 header = TRUE, sep = "\t")
ex.acgh <- create.aCGH(log2.ratios, clones.info, pheno.type)

## Printing, summary and basic plotting for objects of class aCGH

data(colorectal)
colorectal
summary(colorectal)
sample.names(colorectal)
phenotype(colorectal)
plot(colorectal)

## Subsetting aCGH object

colorectal[1:1000, 1:30]

## Imputing the log2 ratios 

log2.ratios.imputed(ex.acgh) <- impute.lowess(ex.acgh)

## Determining hmm states of the clones
## WARNING: Calculating the states takes some time

hmm(ex.acgh) <- find.hmm.states(ex.acgh)

## Calculating the standard deviations for each array

sd.samples(ex.acgh) <- computeSD.Samples(ex.acgh)

## Finding the genomic events associated with each sample

genomic.events(ex.acgh) <- find.genomic.events(ex.acgh)

## Plotting and printing the hmm states

plotHmmStates(ex.acgh, 1)
postscript("/tmp/hmm.states.temp.ps")
plotHmmStates(ex.acgh, 1)
dev.off()

## Plotting summary of the sample profiles

plotSummaryProfile(colorectal)

}
\author{
  Peter Dimitrov
}
\keyword{classes}% at least one, from doc/KEYWORDS

\eof
\name{aCGH.read.Sprocs}
\alias{aCGH.read.Sprocs}
\alias{dotify.names}
\alias{read.Sproc.files}
\alias{extract.clones.info}
\title{Create object of class "aCGH" from Sproc files}
\description{
  This function reads in two-channel Array Complimentary Genomic
  Habrydization Sproc files, flags them for bad quality and missing
  data, and creates object of class \code{aCGH}.
}
\usage{
aCGH.read.Sprocs(fnames, latest.mapping.file = NULL, maxsd = 0.2,
                 minreplic = 2, chrom.remove.threshold = 24,
                 prop.missing = 0.25, sample.names = fnames,
                 sample.quality.threshold = 0.4,
                 cols = c("Log2Rat", "Log2StdDev", "NReplic", "Bad.P")
                 )
}
%- maybe also 'usage' for other objects documented here.
\arguments{
  \item{fnames}{a vector of character strings containing the file names
    of each Sproc data file.}
  \item{latest.mapping.file}{The name of an optional file that contains
    newer clone mapping different from the clone mapping used at the
    time when the arrays were created.}
  \item{maxsd}{maximum of standard deviation of log2 ratios used in
    pre-filtering.}
  \item{minreplic}{minimum number of replicates per clone for a single
    chip used to calculate the log2 ratios.}
  \item{chrom.remove.threshold}{Chromosomes are ordered and numbered as
    usual, except for X and Y chromosome, which in for Homo sapiens
    genome have numbers 23 and 24 repsectivelly, in for Mus musculus 20
    and 21, etc.}
  \item{prop.missing}{Clones are screened out and if the proportion
    missing in the samples is \code{prop.missing} they are removed.}
  \item{sample.names}{Sample names. If they are missing, the file names
    are used after stripping the characters after the last dot in the
    filename if one exists; for example 'myfile.txt' becomes myfile.}
  \item{sample.quality.threshold}{Mark those samples that have their
    proportion of missing values \code{sample.quality.threshold}.}
  \item{cols}{
    character vector of length 4 containing the following Sproc file
    column names: log2 ratios, std. deviations of the log2
    ratios, number of replicates for each clone and flags for bad
    clones. Defaults to c("Log2Rat", "Log2StdDev", "NReplic",
    "Bad.P"). Note that all the whitespace characters in the column
    names will be replaced with dots.
  }
  
}
\value{
  Object of class \code{aCGH}.
}
\author{Jane Fridlyand, Peter Dimitrov}
\seealso{
  \code{aCGH}
}
\examples{

datadir <- system.file("examples", package = "aCGH")
latest.mapping.file <-
      file.path(datadir, "human.clones.info.Jul03.csv")
ex.acgh <-
    aCGH.read.Sprocs(system(paste("ls -1", file.path(datadir,
                    "*.txt")), intern = TRUE), latest.mapping.file,
                    chrom.remove.threshold = 23)
ex.acgh

## Testing if creating the object went right. Should all be true

all(log2.ratios(ex.acgh)[ 1, ] == c(-0.077698 , 0.007389))
clone.name <- "RP11-88I12"
all(log2.ratios(ex.acgh)[ clone.name, ] == c(0.096387 , 0.050999))

}
\keyword{file}% at least one, from doc/KEYWORDS

\eof
\name{colorectal}
\alias{colorectal}
\alias{clones.info.ex}
\alias{log2.ratios.ex}
\alias{pheno.type.ex}
\docType{data}
\title{Colorectal array CGH dataset}
\description{
  The colorectal dataset is an object of class \code{\link{aCGH}}. It
  represents a collection of 124 array CGH profiles of primary
  colorectal tumors and their derived attributes. Each sample was
  measured on the BAC clone DNA microarray with approximate resolution
  of 1.4 Mb per clone. There were approximately 2400 clones spotted on
  the array and each clone was printed in triplicates located
  immediately next to each other. Each array consisted of the 16 (4 by
  4) subarrays. The clones were mapped on the July 03 UCSC freeze. There
  were a number of the discrete and continious phenotypes associated
  with the samples such as age, mutation status for various markers,
  stage, location and so on. All images were quantified and
  normalized by Dr. Taku Tokuyasu using custom image software SPOT and
  postprocessing custom software SPROC.
}
\usage{data(colorectal)}
% \format{
%   The format is the object of class aCGH.
% }
\source{
  These data were generated at Dr. Fred Waldman's lab at UCSF Cancer
  Center by K. Nakao and K. Mehta. The manuscript describing the data
  and the analysis are described in
  High-resolution analysis of DNA copy number alterations in colorectal
  cancer by array-based comparative genomic hybridization,
  \emph{Carcinogenesis}, 2004, Nakao et. al.
}
\references{
  Nakao et. al., High-resolution analysis of DNA copy number
  alterations in colorectal cancer by array-based comparative genomic
  hybridization, \emph{Carcinogenesis}, 2004
  Jain et. al, Fully automatic quantification of microarray image data,
  \emph{Genome Research}, 2003
}
\examples{

data(colorectal)
## WARNING: plotting the heatmap takes some time
plot(colorectal)
plotGenome(colorectal[,1:2])

}
\seealso{
  \code{\link{aCGH}}
  \code{\link{plotGenome}}
}
\keyword{datasets}

\eof
\name{computeSD.func}
\alias{computeSD.func}
\alias{computeSD.Samples}
\alias{}
%- Also NEED an `\alias' for EACH other topic documented here.
\title{Function to estimate experimental variability of a sample}
\description{
  This functions estimate experimental variability of a given
  sample. This value can be used to rank samples in terms of the quality
  as well as to derive thresholds for declaring gained and lost clones.
}
\usage{
computeSD.Samples(aCGH.obj, minDiff = .25, maxChrom = 22,
                  maxmadUse = .3, maxmedUse = .5, maxState = 3,
                  minClone = 20, modelUse = 1)
computeSD.func(statesres = states.bic, maxmadUse = 0.2, maxmedUse = 0.2,
               maxState = 3, maxStateChange = 10, minClone = 20,
               maxChrom = 22)
}
%- maybe also `usage' for other objects documented here.
\arguments{
  \item{aCGH.obj}{Object of class \code{aCGH}}
  \item{minDiff}{The threshold specifying the minimum distance between
    medians of different states. The states that have difference of
    medians smaller than \code{minDiff} are merged. See
    \code{\link{mergeFunc}}.
  }
  \item{statesres}{The states.hmm object, generally is the output of
    \code{\link{mergeFunc}}.}
  \item{maxmadUse}{Maximum median absolute deviation allowed to
    controbute to the overall variability calculation.}
  \item{maxmedUse}{Maximum median value for a state allowed to
    contribute to the calculation.}
  \item{maxState}{Maximum number of the states on a given chromosome for
    the states from that chromosome to be allowed to enter noise
    variability calculation.}
  \item{maxStateChange}{Maximum number of changes from state to state on
    a given chromosome for that chromosome to enter  noise variability
    calculation.}
  \item{minClone}{Minimum number of clones in a state for clones in that
    sate to enter variability calculation.}
  \item{maxChrom}{Maxiumum chromosomal index (generally only autosomes
    are used for this calculation.}
  \item{modelUse}{Indicates the index of the model to use. See
    \code{\link{find.hmm.states}}}
}
\details{
  Median absolute deviation is estimated in all the states passing the
  criteria defined by the parameters of the function. Then median of all
  MADs on individual chromosomes as well as across all chromosomes is
  taken to estimate chromosomal  experimental variability and sample
  experimental variability.
}
\value{
  \item{madChrom}{Returns a matrix containing estimated variability for
    each chromosome for each sample.}
  \item{madGenome}{Returns a vector with estimate of experimental
    varibility for each sample.}
}
\seealso{
  \code{\link{aCGH}}
}
\references{Application of Hidden Markov Models to the analysis of the
  array CGH data, Fridlyand et.al., \emph{JMVA}, 2004
}
\author{Jane Fridlyand}
\keyword{models}% at least one, from doc/KEYWORDS

\eof
\name{find.genomic.events}
\alias{find.genomic.events}
\title{Finds the genomic events associated with each of the array CGH samples}
\description{
  Finds the genomic events associated with each of the array CGH
  samples. Events include whole chromosomal gains and losses,
  aberrations, transitions, amplifications and their respective counts
  and sizes. The hmm states has to be computed before using this
  function.
}
\usage{
find.genomic.events(aCGH.obj, maxChrom = 23, minDiff = 0.25,
                    modelUse = 1, factor = 5, maxClones = 1,
                    maxLen = 1000, absValSingle = 1, absValRegion = 1,
                    diffVal1 = 1, diffVal2 = 0.5, maxSize = 10000,
                    pChrom.min = 0.9, medChrom.min = 0.1)
}
\arguments{
  \item{aCGH.obj}{Object of class aCGH}
  \item{maxChrom}{Highest chromosomal number to find events}
  \item{minDiff}{Minimum difference between the states. See \code{\link{mergeFunc}}}
  \item{modelUse}{Model selection criterion to use, See \code{\link{find.hmm.states}}}
  \item{factor}{Determines outliers. See \code{\link{findOutliers.func}}}
  \item{maxClones}{Determines aberrations. See \code{\link{findAber.func}}}
  \item{maxLen}{Determines aberrations. See \code{\link{findAber.func}}}
  \item{absValSingle}{Determines amplifications. See \code{\link{findAmplif.func}}}
  \item{absValRegion}{Determines amplifications. See \code{\link{findAmplif.func}}}
  \item{diffVal1}{Determines amplifications. See \code{\link{findAmplif.func}}}
  \item{diffVal2}{Determines amplifications. See \code{\link{findAmplif.func}}}
  \item{maxSize}{Determines amplifications. See \code{\link{findAmplif.func}}}
  \item{pChrom.min}{Determines whole chromosomal gains and losses. Chromosome should contain no transitions, have its absolute median equal or greater than \code{medChrom.min} and at least \code{medChrom.min} has to be greater or less than 0. }
  \item{medChrom.min}{Determines whole chromosomal gains and losses. Chromosome should contain no transitions, have its absolute median equal or greater than \code{medChrom.min} and at least \code{medChrom.min} has to be greater or less than 0.}
}
\details{
  The default parameters generally work. Threshold for merging may be
  changed depending on the expected normal cell contamination and/or
  expected magnitude of the changes. AIC model generally works ,
  however, may need to be readjusted depending on how liberal or
  conservative one wants to be in finding genomic events. We recommend
  BIC criterion with delta = 1 for noisier data.
}

\value{
  \item{num.transitions}{
    matrix of dimensions maxChrom by number of samples. It contains number of transitions that were recorded on a given chromosome for a given sample.
  }
  \item{num.amplifications}{
    matrix of dimensions maxChrom by number of samples It contains number of amplifications that were recorded on a given chromosome for a given sample.
  }
  \item{num.aberrations}{
    matrix of dimensions maxChrom by number of samples. It contains number of focal aberrations that were recorded on a given chromosome for a given sample.
  }
  \item{num.outliers}{
    matrix of dimensions maxChrom by number of samples.  It contains number of outliers that were recorded on a given chromosome for a given sample.
  }
  \item{num.transitions.binary}{
    binary matrix of dimensions maxChrom by number of samples. Non-zero entry indicates whether 1 or more transitions were recorded on a given chromosome for a given sample.
  }
  \item{num.amplifications.binary}{
    binary matrix of dimensions maxChrom by number of samples. Non-zero entry indicates whether 1 or more amplifications were recorded on a given chromosome for a given sample.
  }
  \item{num.aberrations.binary}{
    binary matrix of dimensions maxChrom by number of samples. Non-zero entry indicates whether 1 or more focal aberrations were recorded on a given chromosome for a given sample.
  }
  \item{num.outliers.binary}{
    binary matrix of dimensions maxChrom by number of samples. Non-zero entry indicates whether 1 or more outliers were recorded on a given chromosome for a given sample.
  }
  \item{whole.chrom.gain.loss}{
    matrix of dimensions maxChrom by number of samples. Positive entry indicates that a given chromosome was gained in a given sample, negative entry indicates that a given chromosome was lost in a given sample, 0 entry is normal chromosome and NA marks chromosomes with one or more transition.
  }
  \item{size.amplicons}{
    matrix of dimensions maxChrom by number of samples. Reports size of a given chromosome that is amplified (kb units) in a given sample.
  }
  \item{num.amplicons}{
    matrix of dimensions maxChrom by number of samples. Reports number of disjoint amplicons on a given chromosome for a given sample. 
  }
  \item{outliers}{
    list containing 3 matrices of dimensions number of clones by number
    of samples. See \code{\link{findOutliers.func}}.
  }
  \item{aberrations}{
    list containing a matrix of dimensions number of clones by number
    of samples. See \code{\link{findAber.func}}.
  }
  \item{transitions}{
    list containing 2 matrices of dimensions number of clones by number
    of samples. See \code{\link{findTrans.func}}.
  }
  \item{amplifications}{
    list containing a matrix of dimensions number of clones by number
    of samples. See \code{\link{findAmplif.func}}
  }
}
\seealso{
  \code{\link{aCGH}}
  \code{\link{find.hmm.states}}
  \code{\link{mergeFunc}}
  \code{\link{findAber.func}}
  \code{\link{findTrans.func}}
  \code{\link{findAmplif.func}}
  \code{\link{findOutliers.func}}
}
\keyword{models}% at least one, from doc/KEYWORDS

\eof
\name{find.hmm.states}
\alias{find.hmm.states}
\alias{hmm.run.func}
\alias{mu1.func}
\alias{as.matrix.dist}
\alias{plotChrom.hmm.func}
\alias{plotCGH.hmm.func}
\alias{smoothData.func}
\alias{thresholdData.func}
\alias{plotChrom.samples.func}
\alias{plotChrom.grey.samples.func}
%- Also NEED an `\alias' for EACH other topic documented here.
\title{Determines states of the clones}
\description{
  This function runs unsupervised HMM algorithm and produces the
  essentual state information which is used for the subsequent structure
  determination.
}
\usage{
hmm.run.func(dat, datainfo = clones.info, vr = 0.01, maxiter = 100,
             aic = TRUE, bic = TRUE, delta = 1)
find.hmm.states(aCGH.obj, ...)
}
%- maybe also `usage' for other objects documented here.
\arguments{
  \item{aCGH.obj}{object of class aCGH}
  \item{dat}{dataframe with clones in the rows and samples in the
    columns}
  \item{datainfo}{dataframe containing the clones information that is
    used to map each clone of the array to a position on the genome. Has
    to contain columns with names Clone/Chrom/kb containing clone names,
    chromosomal assignment and kb positions respectively}
  \item{vr}{Initial experimental variance}
  \item{maxiter}{Maximum number of iterations}
  \item{aic}{TRUE or FALSE variable indicating whether or nor AIC
    criterion should be used for model selection (see DETAILS)}
  \item{bic}{TRUE or FALSE variable indicating whether or nor AIC
    criterion should be used for model selection (see DETAILS)}
  \item{delta}{numeric vector of penalty factors to use with BIC
    criterion (see DETAILS)}
  \item{...}{
    All the parameters that can be passed to hmm.run.func except dat and
    datainfo.
  }
}
\details{
  One or more model selection criterion is used to determine number of
  states on each chromosomes. If several are specified, then a separate
  matrix is produced for each criterion used. Delta is a fudge factor in
  BIC criterion: $\delta BIC(\gamma) = \log RSS(\gamma) +
  q_{\gamma}\delta\log n/n.$ Note that delta = 1 leads to conventional
  BIC.
  find.hmm.states(aCGH.obj, ...) uses aCGH object instead of log2 ratios
  matrix dat.
}
\value{
  Two lists of lists are returned. Each list contains information on the
  states with each of the specified model selection criteria. E.g., if
  AIC = T, BIC = T and delta = c(1,1.5), then each list will contain
  three lists corresponding to AIC, BIC(1) and BIC(1.5) as the 1st,2nd
  and 3rd lists repsectively. If AIC is used, it always comes first and
  BIC models come in the order of the delta vector.
  \item{states.hmm}{Each of the sublists contains 2+ 6*n columns where
    the first two columns contain chromosome and kb positions for each
    clone in the dataset supplied followed up by 6 columns for each
    sample where n = number of samples.

    column 1 = state

    column 2 = smoothed value for a clone

    column 3 = probability of being in a state

    column 4 = predicted value of a state

    column 5 = dispersion

    column 6 = observed value
  }
  \item{nstates.hmm}{Each of the sublists contains a matrix with each
    row corresponding to a chromosome and each column to a sample. The
    entries indicate how many different states were identified for a
    given sample on a given chromosome}
}
\references{Application of Hidden Markov Models to the analysis of the
  array CGH data, Fridlyand et.al., \emph{JMVA}, 2004 }
\author{Jane Fridlyand}
\section{WARNING}{When algortihm fails to fit an HMM for a given number
  of states on a chromosome, it prints a warning.}
\seealso{
  \code{\link{aCGH}}
}
\examples{

datadir <- system.file("examples", package = "aCGH")
latest.mapping.file <-
      file.path(datadir, "human.clones.info.Jul03.csv")
ex.acgh <-
    aCGH.read.Sprocs(system(paste("ls -1", file.path(datadir,
                    "*.txt")), intern = TRUE), latest.mapping.file,
                    chrom.remove.threshold = 23)
ex.acgh

hmm(ex.acgh) <- find.hmm.states(ex.acgh, aic = TRUE, delta = 1.5)
summary(ex.acgh)

}
\keyword{models}

\eof
\name{findAber.func}
\alias{findAber.func}
%- Also NEED an `\alias' for EACH other topic documented here.
\title{Function to  determines focal aberrations}
\description{
  The function identifies clones that are focal aberrations.
}
\usage{
findAber.func(maxClones = 1, maxLen = 1000, statesres = states.bic)
}
%- maybe also `usage' for other objects documented here.
\arguments{
  \item{maxClones}{Maximum number of clones assigned to the same state
    which can be considered to be focal aberrations}
  \item{maxLen}{Maximum lengeth of the region containing clones assigned
    to the state so that those clones can be considered to be focal
    aberrations}
  \item{statesres}{The states output of the \code{\link{hmm.run.func}}}
}
\details{
  The focal aberrations are the one or more clones assigned to the
  state different from the states of the surrounding clones.  They may
  indicate copy number polymorphisms or interesting high or low focal
  changes.
}
\value{
  \item{aber}{Binary matrix with a row for each clone and column for
    each sample. 1 indicates presence of a focal aberrations, 0 lack
    of such.}
}
\references{"Application of Hidden Markov Models to the analysis of the array CGH  data", Fridlyand et.al., JMVA, 2004
}
\author{Jane Fridlyand}
\keyword{models}% at least one, from doc/KEYWORDS

\eof
\name{findAmplif.func}
\alias{findAmplif.func}
%- Also NEED an `\alias' for EACH other topic documented here.
\title{Function to determine high level amplifications}
\description{
  This function identifies high level amplifications by considering the
  height, the width of an amplicon relative to the urrounding
  clones. Only narrow peaks much higher than its neigbors are considered
  as high level amplifications.
}
\usage{
findAmplif.func(absValSingle = 1, absValRegion = 1.5, diffVal1 = 1,
diffVal2 = 0.5, maxSize = 10000, translen.matr = res3$translen.matrix,
trans.matr = res3$trans.matr, aber = res2$aber, outliers = res1$outlier,
pred = res1$pred.out, pred.obs = res1$pred.obs.out, statesres =
states.bic)
}
%- maybe also `usage' for other objects documented here.
\arguments{
  \item{absValSingle}{A clone is declared to be an amplification if it
    is a focal aberration or an outlier and its value exceeds
    absValSingle}
  \item{absValRegion}{A clone is an amplification if if a clone belong
    to a region with width less than maxSize and  observed value for a
    clones is greater than absValRegion}
  \item{diffVal1}{Clone is an amplification if it is an aberration and
    greater by diffVal1 than max of the two surrounding stretches}
  \item{diffVal2}{Clone is an amplification if it is an outlier, its
    observed values is greater by diffVal2 than max of the two
    surrounding stretches}
  \item{maxSize}{The clones  may not be declared as amplifications if
    they belong to the states with spanning more than maxSize}
  \item{translen.matr}{State length matrix. The output of the
    \code{\link{findTrans.func}}}
  \item{trans.matr}{Transition matrix. The output of the
    \code{\link{findTrans.func}}}
  \item{aber}{Aberration matrix. The output of the
    \code{\link{findAber.func}}}
  \item{outliers}{Outliers matrix. The output of the
    \code{\link{findOutliers.func}}}
  \item{pred}{Predicted values matrix. The output of the
    \code{\link{findOutliers.func}}}
  \item{pred.obs}{Predicted values matrix with observed values assigned
    to the outliers. The output of the \code{\link{findOutliers.func}}}
  \item{statesres}{The states output of the \code{\link{hmm.run.func}}}
}
\details{
  Note that all the distances are in Megabases and all the heights are
  on log2ratio scale.
}
\value{
  \item{amplif.matrix}{Binary matrix with a row for each clone and
    column for each sample. "1" indicates amplification}
  ...
}
\seealso{
  \code{\link{aCGH}}
}
\references{Application of Hidden Markov Models to the analysis of the
  array CGH  data, Fridlyand et.al., \emph{JMVA}, 2004}
\author{Jane Fridlyand}
\keyword{models}% at least one, from doc/KEYWORDS

\eof
\name{findOutliers.func}
\alias{findOutliers.func}
%- Also NEED an `\alias' for EACH other topic documented here.
\title{Function to identify outlier clones}
\description{
  The function identified the clones that are outliers.
}
\usage{
findOutliers.func(thres = madGenome, factor = 4, statesres = states.bic)
}
%- maybe also `usage' for other objects documented here.
\arguments{
  \item{thres}{Estimate of experimental variability, generally,
    madGenome}
  \item{factor}{Factor indicating how many standard}
  \item{statesres}{The states output of the \code{\link{hmm.run.func}}}
}
\details{
  The outliers are the clones that are dissimilar enough from the clones
  assigned to the same state. Magnitude of the factor determines how
  many MADs away from a median a value needs to be to be declared an
  outlier. Outliers consitent over many samples may indicate
  technological artificat with that clone or possibly copy number
  polymorpism.
}
\value{
  \item{outlier}{Binary matrix with a row for each clone and column for
    each sample. "1" indicates outlier, 0 otherwise.}
  \item{pred.obs.out}{Matrix with a row for each clone and column for
    each sample. The entries are the median value for the state with
    outliers exceluded for all clones but outliers. The value if the
    observed value for the outliers.}
  \item{pred.out}{Matrix with a row for each clone and column for each
    sample. The entries are the median value for the state }
}
\seealso{
  \code{\link{aCGH}}
}
\references{Application of Hidden Markov Models to the analysis of the
  array CGH  data, Fridlyand et.al., \emph{JMVA}, 2004
}
\author{Jane Fridlyand }
\keyword{models}% at least one, from doc/KEYWORDS

\eof
\name{findTrans.func}
\alias{findTrans.func}
%- Also NEED an `\alias' for EACH other topic documented here.
\title{Funtion identifying the transitions}
\description{
  This function identifies the start and end of the states (regions with
  the constant estimated copy number).
}
\usage{
findTrans.func(outliers = res1$outliers, aber = res2$aber, statesres =
states.bic)
}
%- maybe also `usage' for other objects documented here.
\arguments{
  \item{outliers}{Binary matrix of the outliers (generally output of the
    \code{\link{findOutliers.func}})}
  \item{aber}{Binary matrix of the focal aberrations (generally output
    of the \code{\link{findAber.func}})}
  \item{statesres}{The states output of the \code{\link{hmm.run.func}}}
}
\details{
  The transitions end is placed at the last non-focal aberration clone
  of the contiguous region containing clones belonging to the same state
  and transitions start is placed at the first non-focal aberration
  clone of the contiguous region containing clones belonging to the same
  state.
}
\value{
  \item{trans.matrix}{Matrix with a row for each clone and column for
    each sample. The starts of the states are indicated by "1" , the end
    are by "2" and the focal aberrations are coded as "3"}
  \item{translen.matrix}{Matrix with a row for each clone and column for
    each sample. The entries are the length of the region to which a
    clone belongs. Zero length is assigned to the focal
    aberrations. This output may be buggy at the moment.}
}
\seealso{
  \code{\link{aCGH}}
}
\references{
  Application of Hidden Markov Models to the analysis of the array CGH
  data, Fridlyand et.al., \emph{JMVA}, 2004.
}
\author{Jane Fridlyand}
\keyword{models}% at least one, from doc/KEYWORDS

\eof
\name{heatmap}
\alias{heatmap}
%- Also NEED an '\alias' for EACH other topic documented here.
\title{Default method for plotting array CGH objects}
\description{
  Clusters samples and produces heatmapp of the observed log2ratios.
}
\usage{
heatmap(x = aCGH.obj, imp = TRUE, Rowv = NA, Colv = NULL, distfun =
        dist, hclustfun = hclust, add.expr, symm = FALSE,
        revC = identical(Colv, "Rowv"), scale = "none", na.rm = TRUE,
        margins = c(5, 5), ColSideColors, RowSideColors,
        cexRow = 0.2 + 1/log10(nr), cexCol = 0.2 + 1/log10(nc),
        labRow = NULL, labCol = NULL, main = NULL, xlab = NULL,
        ylab = NULL, verbose = getOption("verbose"), methodR = "ward",
        methodC = "ward", zlm = c(-0.5, 0.5), ...)
}
%- maybe also 'usage' for other objects documented here.
\arguments{
  \item{x}{object of the \code{aCGH} object}
  \item{imp}{logical variable indicating whether
    \code{log2.ratios.imputed} or \code{log2.ratios} slot of
    \code{\link{aCGH}} should be used. Defaults to imputed value (TRUE).
  }
  \item{Rowv}{determines if and how the row dendrogram should be computed
    and reordered.  Either a 'dendrogram' or a vector of values
    used to reorder the row dendrogram or 'NA' to suppress any
    row dendrogram (and reordering) or by default, 'NULL'
  }
  \item{Colv}{determines if and how the column dendrogram should be
    reordered.  Has the same options as the \code{Rowv} argument above
    and additionally when \code{x} is a square matrix, \code{Colv = "Rowv"}
    means that columns should be treated identically to the rows.
  }
  \item{distfun}{ function used to compute the distance (dissimilarity) between
    both rows and columns. Defaults to 'dist'.}
  \item{hclustfun}{function used to compute the hierarchical clustering when
    'Rowv' or 'Colv' are not dendrograms.  Defaults to 'hclust'}
  \item{add.expr}{expression that will be evaluated after the call to 'image'.
    Can be used to add components to the plot.}
  \item{symm}{logical indicating if 'x' should be treated *symm*etrically;
    can only be true when 'x' is a square matrix.}
  \item{revC}{logical indicating if the column order should be 'rev'ersed
    for plotting, such that e.g., for the symmetric case, the
    symmetry axis is as usual.}
  \item{scale}{character indicating if the values should be centered and
    scaled in either the row direction or the column direction,
    or none.  The default is "row" if \code{symm} false, and
    "none" otherwise.}
  \item{na.rm}{logical indicating whether 'NA''s should be removed.}
  \item{margins}{numeric vector of length 2 containing the margins (see
    'par(mar= *)') for column and row names, respectively.}
  \item{ColSideColors}{(optional) character vector of length 'ncol(x)'
    containing the color names for a horizontal side bar that may
    be used to annotate the columns of 'x'.}
  \item{RowSideColors}{(optional) character vector of length 'nrow(x)'
    containing the color names for a vertical side bar that may
    be used to annotate the rows of 'x'.}
  \item{cexRow, cexCol}{positive numbers, used as 'cex.axis' in for the row or
    column axis labeling.  The defaults currently only use number
    of rows or columns, respectively.}
  \item{labRow, labCol}{character vectors with row and column labels to use;
    these default to 'rownames(x)' or 'colnames(x)',
    respectively.}
  \item{main, xlab, ylab}{main, x- and y-axis titles;}
  \item{verbose}{logical indicating if information should be printed.}
  \item{methodR}{method to use for clustering rows. defaults to "ward"}
  \item{methodC}{method to use for clustering columns. defaults to "ward"}
  \item{zlm}{all the values greater or equal  than zlm are set to zlm -
    0.01. a;; value less or equal to -zlm are set to -zlm + 0.01
  }
  \item{\dots}{additional arguments passed on to 'image', e.g., 'col'
    specifying the colors.
  }
}
\details{
  This function is almost identical to the \code{\link{heatmap}} in base
  R. The slight modifications are that (1) a user can specify clustering
  method for rows and columns; (2) all the values outside specified
  limits are floored to be 0.01 less than a limit; (3) default values
  are different. Note that using default option of \code{imp} (TRUE)
  produces nicer looking plots as all missing values are removed.
}
\value{
  Invisibly, a list with components
  \item{crowInd}{row index permutation vector as returned by
    \code{\link{order.dendrogram}}
  }
  \item{colInd}{row index permutation vector as returned by
    \code{\link{order.dendrogram}}
  }
}
\references{heatmap function in base R}
\seealso{
  \code{\link{aCGH}}
  \code{\link{plotvalGenome.func}}
}
\examples{

#default plotting method for the aCGH object
data(colorectal)
plot(colorectal)

#to produce smoother looking heatmap, use imp = T: this will use imputed
#slot of aCGH object

plot(colorectal, imp = TRUE)

}
\keyword{hplot}
\keyword{cluster}

\eof
\name{human.chrom.info.Jul03}
\alias{human.chrom.info.Jul03}
\docType{data}
\title{
  Basic Chromosomal Information for UCSC Human Genome Assembly July 2003
  freeze
}
\description{
  This dataset contains basic chromosomal information for UCSC Human
  Genome Assembly July 2003 freeze.
}
\usage{
human.chrom.info.Jul03 is loaded automatically with
\code{\link{aCGH}} package.
}
\format{
  A data frame with 24 observations on the following 3 variables.
  \describe{
    \item{chrom}{Chromosomal index, X is coded as 23 and Y as 24.}
    \item{length}{Length of each chromosome in kilobases.}
    \item{centromere}{
      Location of the centromere on the chromosome (kb).
    }
  }
}
\details{
  This file is used for many plotting functions and needs to correspond
  to \code{\link{clones.info}} mapping file. The centromeric location is
  approximately extimated by taking mid-point between the last
  fish-mapped clone on the p-arm and the first fish-mapped clone on the
  q-arm using relevant UCSC freeze.
  For an alternative freeze, one needs to manually create a 3-column
  file of the format described above.
}
\source{
  http://genome.ucsc.edu/cgi-bin/hgText
}
% \references{
  
% }
\keyword{datasets}

\eof
\name{impute.lowess}
\alias{impute.lowess}
%- Also NEED an '\alias' for EACH other topic documented here.
\title{Imputing log2 ratios}
\description{
  Imputing log2 ratios
}
\usage{
impute.lowess(aCGH.obj, chrominfo = human.chrom.info.Jul03, maxChrom =
23, smooth = 0.1)
}
%- maybe also 'usage' for other objects documented here.
\arguments{
  \item{aCGH.obj}{Object of class aCGH.}
  \item{chrominfo}{a chromosomal information associated with the mapping of the data}
  \item{maxChrom}{Highest chromosome to impute.}
  \item{smooth}{smoothing parameter for the lowess procedure}
}
\details{
  There are two main reasons to impute data. One is that given that imputation is reasonable, one can increase the analytical power and improve results. Another, more practical, is that at the moment many widely used fuctions in R do not support missing values. While procedures such as kNN imputations is widely used for gene expression data, it is more powerful to take advantage of the genomic structure of the array CGH data and use a smoother. Note that we perform only one pass os smoothing. If there still remain missing values, they are imputed by the median on the chromosome or chromosomal arm where applicable,
}
\value{
  Computes and returns the imputed log2 ratio matrix of the aCGH object.
}
\seealso{
  \code{\link{aCGH}}
}
\examples{

datadir <- system.file("data", package = "aCGH")
clones.info <-
      read.table(file = file.path(datadir, "clones.info.ex.csv"),
                 header = TRUE, sep = "\t")
log2.ratios <-
      read.table(file = file.path(datadir, "log2.ratios.ex.csv"),
                 header = TRUE, sep = "\t")
ex.acgh <- create.aCGH(log2.ratios, clones.info)

## Imputing the log2 ratios 

log2.ratios.imputed(ex.acgh) <- impute.lowess(ex.acgh)

}
\keyword{models}% at least one, from doc/KEYWORDS

\eof
\name{mergeFunc}
\alias{mergeFunc}
%- Also NEED an `\alias' for EACH other topic documented here.
\title{Funtion to merge states based on their state means}
\description{
  This function takes an output of hmm.run.fun (states.hmm) with a
  particular model selection criterion andietratively merges the states
  which means are closer than a supplied threshold.
}
\usage{
mergeFunc(statesres = states.bic, minDiff = 0.1)
}
%- maybe also `usage' for other objects documented here.
\arguments{
  \item{statesres}{the sublist of the states.hmm list output from
    find.hmm.states for a given model selection crterion}
  \item{minDiff}{The states whose predicted values are less than minDiff
    apart are merged into one state and all the predicited values are
    recomputed}
}
\details{
  This function is intended to reduce effect of the possible small
  magnitude technological artifcats on the structure determination.
}
\value{
  List containing states.hmm object is returned.
}
\references{Application of Hidden Markov Models to the analysis of the
  array CGH data, Fridlyand et.al., \emph{JMVA}, 2004
}
\author{Jane Fridlyand}
\keyword{models}% at least one, from doc/KEYWORDS

\eof
\name{plotFreqStat}
\alias{plotFreqStat}
\alias{aCGH.test.old}
\alias{aCGH.test}
\alias{threshold.func}
\alias{changeProp.func}
\alias{changeProp.overall.func}
\alias{table.bac.func}
\alias{lengthGain.na}
\alias{propGain.na}
\alias{lengthLoss.na}
\alias{propLoss.na}
\alias{prop.na}
\alias{gainloss.func}
\alias{plotFreqStatColors}
\alias{plotFreqStatGrey}
\alias{plotFreqGivenStat}
\alias{plotfreqGivenStatFinalColors}
\alias{plotfreq.stat.final.func}
\alias{plotfreq.stat.chrom.final.func}
\alias{plotfreq.givenstat.final.colors.func}

%- Also NEED an '\alias' for EACH other topic documented here.
\title{frequency plots and significance analysis}
\description{
  The main application of this function is to plot the frequency of
  changes.
}
\usage{
plotFreqStat(aCGH.obj, resT, pheno,
             chrominfo = human.chrom.info.Jul03, X = TRUE, Y = FALSE,
             threshold = TRUE, minChanged = 0, all = FALSE,
             rsp.uniq = unique(pheno), nlim = 1, cutplot = 0,
             titles = rsp.uniq, thres = 0.2, ylm = c(-1, 1),
             ngrid = 2, p.thres = c(0.01, 0.05, 0.1),
             mincolors = 0.1, quant.col = 0.11, numaut = 22,
             onepage = TRUE, colored = TRUE, summarize.clones = TRUE)
}
%- maybe also 'usage' for other objects documented here.
\arguments{
  \item{aCGH.obj}{Object of class \code{aCGH}}
  \item{resT}{Data frame having the same structure as the result of
    applying \code{\link{mt.maxT}} or \code{\link{mt.minP}} functions
    from Bioconductor's \code{multtest} package for multiple testing.
    The result is a data frame including the following 4 components:
    'index', 'teststat', 'rawp' and 'adjp'.
  }
  \item{pheno}{phenotype to compare.}
  \item{chrominfo}{
    Chromosomal information. Defaults to
    \code{\link{human.chrom.info.Jul03}}
  }
  \item{X}{Include X chromosome? Defaults to yes.}
  \item{Y}{Include Y chromosome? Defaults to no.}
  \item{titles}{
    \code{titles} names of the groups to be used. Default is the unique
    levels of the \code{pheno}.
  }
  \item{thres}{\code{thres} is either a vector providing unique
    threshold for each sample or a vector of the same length as number
    of samples (columns in \code{data}) providing sample-specific
    threshold. Clone is considered to be gained if it is above the
    threshold and lost if it below negative threshold. Used for plotting
    the gain/loss freqeuncy data as well as for clone screening and for
    significance analysis when \code{threshold} is T.
  }
  \item{cutplot}{only clones with at least \code{cutplot} frequency of
    gain and loss are plotted.
  }
  \item{ylm}{\code{ylm} vertical limits for the plot}
  \item{p.thres}{
    \code{p.thres} vector of p-value ciut-off to be plotted. computed
    conservatively as the threshold corresponding to a given adjusted
    p-value.
  }
  \item{ngrid}{
    \code{ngrid} in how many grids red and green colors need to be
    divided.
  }
  \item{nlim}{\code{nlim} maximum value for colors.}
  \item{mincolors}{\code{mincolors} minimum value for colors.}
  \item{quant.col}{
    \code{quant.col} what quantile to use for color display.
  }
  \item{numaut}{\code{numaut} number of the autosomes}
  \item{onepage}{
    \code{onepage} whether all plots are to be plotted on one page or
    different pages.
  }
  \item{threshold}{
    \code{threshold} specifies whether significance analysis should be
    performed using original values (F) or thresholded (trimerized)
    values (T).
  }
  \item{minChanged}{
    Only clones that change in at least minChanged proportion of samples
    are plotted; is 0 by default.
  }
  \item{all}{
    \code{all} specifies whether samples should be analyzed by subgroups
    (T) or together (F).
  }
  \item{rsp.uniq}{\code{rsp.uniq} specified the codes for the groups of
    interest. Default is the unique levels of the phenotype. Not used
    when \code{all} is T.
  }
  \item{colored}{Is plotting in color or not? Default is T.}
  \item{summarize.clones}{Return frequency summaries for individual clones.}
}
% \details{
%   ~~ If necessary, more details than the __description__  above ~~
% }
\seealso{
  \code{\link{aCGH}}
}
\examples{

data(colorectal)

## Use mt.maxT function from multtest package to test
## differences in group means for each clone grouped by sex
colnames(phenotype(colorectal))
sex <- phenotype(colorectal)$sex
sex.na <- !is.na(sex)
colorectal.na <- colorectal[ ,sex.na, keep = TRUE ]
dat <- log2.ratios.imputed(colorectal.na)
resT.sex <- mt.maxT(dat, sex[sex.na], test = "t", B = 1000)

## Plot the result along the genome
plotFreqStat(colorectal.na, resT.sex, sex[sex.na],
             titles = c("Male", "Female"))

## Adjust the p.values from previous exercise with "fdr"
## method and plot them
resT.sex.fdr <- resT.sex
resT.sex.fdr$adjp <- p.adjust(resT.sex.fdr$rawp, "fdr")
plotFreqStat(colorectal.na, resT.sex.fdr, sex[sex.na],
             titles = c("Male", "Female"))

## Derive statistics and p-values for testing the linear association of
## age with the log2 ratios of each clone along the samples

age <- phenotype(colorectal)$age
age.na <- !is.na(age)
colorectal.na <- colorectal[ ,age.na, keep = TRUE ]
dat <- log2.ratios.imputed(colorectal.na)
stat.age <- sapply(1:nrow(dat),
                   function(i) {
                      if (i \%\% 100 == 0)
                         cat(i, "\n")
                      lm.fit <-
                          summary(lm(dat[i,] ~ age[age.na]))
                      c(lm.fit$fstatistic[1],
                        1 - pf(lm.fit$fstatistic[1],
                               lm.fit$fstatistic[2],
                               lm.fit$fstatistic[3])
                       )
                   }
                   )
## Make resT
resT.age <- data.frame(index = 1:ncol(stat.age),
                       teststat = stat.age[ 1, ],
                       rawp = stat.age[ 2, ],
                       adjp = p.adjust(stat.age[ 2, ], "fdr"))
#write out the table of results:
tbl.age <- data.frame(clones.info(colorectal), resT.age)
write.table(tbl.age, "table.age.txt", sep="\t", col.names = TRUE,
            row.names = FALSE, quote = FALSE)

}
\keyword{htest}
\keyword{hplot}

\eof
\name{plotGenome}
\alias{plotGenome}
\alias{plotGene}
\alias{plotGeneSign}
\alias{maPalette}
\alias{plotCGH.func}

%- Also NEED an '\alias' for EACH other topic documented here.
\title{Plots the genome}
\description{
  Basic plot of the log2 ratios for each array ordered along the genome.
}
\usage{
plotGenome(aCGH.obj, samples = 1:num.samples(aCGH.obj), naut = 22,
           Y = TRUE, X = TRUE, data = log2.ratios(aCGH.obj),
           chrominfo = human.chrom.info.Jul03,
           yScale = c(-2, 2), samplenames = sample.names(aCGH.obj),
           ylb = "Log2Ratio")
}
%- maybe also 'usage' for other objects documented here.
\arguments{
  \item{aCGH.obj}{an object of class aCGH}
  \item{samples}{vector containing indeces of the samples to be plotted.}
  \item{naut}{number of autosomes in the organism}
  \item{Y}{TRUE if chromosome Y is to be plotted, FALSE otherwise}
  \item{X}{TRUE if chromosome X is to be plotted, FALSE otherwise}
  \item{data}{a matrix containing values to use for plotting. defaults to the \code{log2.ratios(aCGH.obj)}.}
  \item{chrominfo}{a chromosomal information associated with the mapping
    of the data.}
  \item{yScale}{
    Minimum y-scale to use for plotting. Scale is expanded if any of the
    values exceed the positive or negative limit.
  }
  \item{samplenames}{sample names.}
  \item{ylb}{label for the Y-axis.}
}
\seealso{
  \code{\link{aCGH}}
}
\examples{

#plot samples in the order of descending quality 
data(colorectal)
order.quality <- order(sd.samples(colorectal)$madGenome)
postscript("plotGenome.orderByQuality.ps", paper="letter")
par(mfrow=c(2,1))
for(i in order.quality)
   plotGenome(colorectal, samples = i, Y = FALSE)
dev.off()

}
\keyword{hplot}% at least one, from doc/KEYWORDS

\eof
\name{plotHmmStates}
\alias{plotHmmStates}
\alias{plotHmmStatesNew}
%- Also NEED an '\alias' for EACH other topic documented here.
\title{
  Plotting the estimated hmm states and log2 ratios for each sample.
}
\description{
  This function displays the estimated hmm states and log2 ratios for
  each sample.
}
\usage{
plotHmmStates(aCGH.obj, sample.ind, chr = 1:num.chromosomes(aCGH.obj),statesres =
              hmm(aCGH.obj)$states.hmm[[1]],
              maxChrom = 23, chrominfo = human.chrom.info.Jul03,
              yScale = c(-2,2), samplenames = sample.names(aCGH.obj))
}
%- maybe also 'usage' for other objects documented here.
\arguments{
  \item{aCGH.obj}{object of class aCGH}
  \item{sample.ind}{index of the sample to be plotted relative to the
    data matrix (i.e. column index in the file)}
  \item{statesres}{matrix containing states informations. defaults to the states selected using the first  model selection criterionof \code{aCGH.obj}}
  \item{chr}{vector of chromosomes to be plotted}
  \item{yScale}{specified scale for Y-exis}
  \item{maxChrom}{highest chromosome to show}
  \item{chrominfo}{a chromosomal information associated with the mapping of the data}
  \item{samplenames}{vector of sample names}
}
\details{
  Each chromosome is plotted on a separate  page and contains two
  figures. The top figure shows the observed log2ratios and the bottom
  figure shows predicted values for all clones but outliers which show
  observed values. The genomic events are indicated on both figures as
  following. The first clone after transition is indicated with solid
  blue line and the last clone after transitions is shown with dotted
  green line. Focal aberrations clones are colored orange,
  amplifications are colored red and outliers are yellow.
}
\references{Application of Hidden Markov Models to the analysis of the
  array CGH data, Fridlyand et.al., \emph{JMVA}, 2004
}
\author{Jane Fridlyand}
\seealso{
  \code{\link{aCGH}}
  \code{\link{find.hmm.states}}
  \code{\link{plotGenome}}
}
\examples{

data(colorectal)
plotHmmStates(colorectal, 1)

}
\keyword{hplot}% at least one, from doc/KEYWORDS

\eof
\name{plotSummaryProfile}
\alias{plotSummaryProfile}
\title{plotSummaryProfile}
\description{
  This function display the genomic events and tests for the differences
  between groups if they are specified.
}
\usage{
plotSummaryProfile(aCGH.obj,
                   response = as.factor(rep("All", ncol(aCGH.obj))),
                   titles = unique(response[!is.na(response)]),
                   X = TRUE, Y = FALSE, maxChrom = 23,
                   chrominfo = human.chrom.info.Jul03)
}
%- maybe also 'usage' for other objects documented here.
\arguments{
  \item{aCGH.obj}{an object of aCGH class.}
  \item{response}{
    categorical phenotype to compare. defaults to all the samples being
    analyzed together.
  }
  \item{titles}{
    titles for the groups, defaults to the name of the \code{response}
  }
  \item{X}{
    logical indicating whether X needs to be shown
  }
  \item{Y}{
    logical indicating whether Y needs to be shown
  }  
  \item{maxChrom}{this parameter controls how many chromosomes will be
    plotted, from 1 to maxChrom.}
  \item{chrominfo}{
    a chromosomal information associated with the mapping of the data.
  }
}
\details{
  This function utilizes output of the \code{\link{find.genomic.events}}
  by plotting it and testing between groups. The test are performed
  using kruskal-wallis rank test.
}
% \value{
%   plot containing a number of events
% }
\seealso{
  \code{\link{aCGH}}
  \code{\link{find.genomic.events}}
}
\examples{

data(colorectal)

## Plotting summary of the sample profiles
plotSummaryProfile(colorectal)

}
\keyword{hplot}% at least one, from doc/KEYWORDS

\eof
\name{plotvalGenome.func}
\alias{plotvalGenome.func}
\alias{plotvalChrom.func}
\alias{plotValGenome}
\alias{plotValChrom}
\alias{plotChrom}
%- Also NEED an '\alias' for EACH other topic documented here.
\title{clustering and heatmap}
\description{
  This function clusters samples and shows their heatmap
}
\usage{
plotvalGenome.func(aCGH.obj,
                   response = as.factor(rep("All", ncol(aCGH.obj))),
                   chrominfo = human.chrom.info.Jul03, cutoff=1,
                   lowCol = "red", highCol = "green", midCol = "black",
                   ncolors = 50, byclass = FALSE, showaber = FALSE,
                   amplif = 1, homdel = -0.75,
                   samplenames = sample.names(aCGH.obj),
                   vecchrom = 1:23, titles = "Image Plot",
                   methodS = "ward", dendPlot = TRUE, imp = TRUE,
                   categoricalPheno = TRUE)
}
%- maybe also 'usage' for other objects documented here.
\arguments{
  \item{aCGH.obj}{object of class \code{aCGH} here}
  \item{response}{phenotype of interest. defaults to the same phenotype
    assigned to all samples}
  \item{chrominfo}{a chromosomal information associated with the mapping
    of the data}
  \item{cutoff}{maximum absolute value. all the values are floored to
    +/-cutoff depending on whether they are positive of
    negative. defaults to 1}
  \item{ncolors}{number of colors in the grid. input to
    \code{\link{maPalette}}. defaults to 50}
  \item{lowCol}{color for the low (negative) values. input to
    \code{\link{maPalette}}. defaults to "red"}
  \item{highCol}{color for the high (positive) values. input to
    \code{\link{maPalette}}. defaults to "green"}
  \item{midCol}{color for the values close to 0. input to
    \code{\link{maPalette}}. defaults to "black"}
  \item{byclass}{logical indicating whether samples should be clustered within each level of the phenotype or overall. defaults to F}
  \item{showaber}{logical indicating whether high level amplifications and homozygous deletions should be indicated on the plot. defaults to F}
  \item{amplif}{positive value that all observations equal or exceeding it  are marked by yellow dots indicating high-level changes. defaults to 1}
  \item{homdel}{negative value that all observations equal or below it  are marked by light blue dots indicating homozygous deletions. defaults to -0.75}
  \item{samplenames}{sample names}
  \item{vecchrom}{vector of chromosomal indeces to use for clustering and to display. defaults to 1:23}
  \item{titles}{plot title. defaults to "Image Plots"}
  \item{methodS}{clustering method to cluster samples. defaults to "ward"}
  \item{dendPlot}{logical indicating whether dendogram needs to be
    drawn. defaults to T.}
  \item{imp}{logical indicating whether imputed or original values should be used. defaults to T, i.e. imputed.}
  \item{categoricalPheno}{logical indicating whether phenotype is
    categorical. Continious phenotypes are treated as "no groups" except
    that their values are dispalyed.defaults to TRUE.
  }
}
\details{
  This functions is a more flexible version of the
  \code{\link{heatmap}}. It can cluster within levels of categorical
  phenotype as well as all of the samples while displaying phenotype
  levels in different colors. It also uses any combination of
  chromosomes that is requested and clusters samples based on these
  chromosomes only. It draws the chromosomal boundaries and displays
  high level changes and homozygous deletions. If phenotype if not
  categical, its values may still be displayed but groups are not formed
  and byclass = F.
  Image plot has the samples reordered according to clustering order.
}
% \value{
% }
%\references{ ~put references to the literature/web site here ~ }
\seealso{
  \code{\link{aCGH}}
  \code{\link{heatmap}}
}
\examples{ 
data(colorectal)

#cluster all samples using imputed data on all chromosomes (autosomes and X):

plotvalGenome.func(colorectal)

#cluster samples within sex groups based on 3 chromosomes individually. 
#use non-imputed data and  do not show dendogram. Indicate amplifications and 
#homozygous deletions.

plotvalGenome.func(colorectal, response = phenotype(colorectal)$sex,
                   byclass = TRUE, showaber = TRUE, vecchrom = c(4,8,9),
                   dendPlot = FALSE, imp = FALSE)

#cluster samples based on each chromosome individualy and display age. Show
#gains in red and losses in green. Show aberrations and use values < -1
#to identify homozgous deletions. Do not show dendogram.

pdf("plotimages.pdf", width = 11, height = 8.5)
for (i in 1:23)
    plotvalGenome.func(colorectal,
                       response = phenotype(colorectal)$age,
                       chrominfo = human.chrom.info.Jul03,
                       cutoff = 1, ncolors = 50, lowCol="green",
                       highCol="red", midCol="black", byclass = FALSE,
                       showaber = TRUE, homdel = -1, vecchrom = i,
                       titles = "Image Plot", methodS = "ward",
                       dendPlot = FALSE, categoricalPheno = FALSE)
dev.off()

}
\keyword{hplot}
\keyword{cluster}

\eof
\name{states.hmm.func}
\alias{states.hmm.func}
%- Also NEED an `\alias' for EACH other topic documented here.
\title{A function to fit unsupervised Hidden Markov model}
\description{
  This function is a workhorse of \code{\link{find.hmm.states}}. It
  operates on the individual chromosomes/samples and is not called
  directly by users.
}
\usage{
states.hmm.func(sample, chrom, dat, datainfo = clones.info, vr = 0.01,
                maxiter = 100, aic = FALSE, bic = TRUE, delta = 1,
                nlists = 1)
}
%- maybe also `usage' for other objects documented here.
\arguments{
  \item{sample}{ ~~Describe \code{sample} here~~ }
  \item{chrom}{ ~~Describe \code{chrom} here~~ }
  \item{dat}{ ~~Describe \code{dat} here~~ }
  \item{datainfo}{ ~~Describe \code{datainfo} here~~ }
  \item{vr}{ ~~Describe \code{vr} here~~ }
  \item{maxiter}{ ~~Describe \code{maxiter} here~~ }
  \item{aic}{ ~~Describe \code{aic} here~~ }
  \item{bic}{ ~~Describe \code{bic} here~~ }
  \item{delta}{ ~~Describe \code{delta} here~~ }
  \item{nlists}{ ~~Describe \code{nlists} here~~ }
}
\seealso{
  \code{\link{aCGH}}
}
\keyword{models}% at least one, from doc/KEYWORDS

\eof
