\name{Ley}
\alias{Ley}

\title{Gene Expression Data from Mouse Immune response study, (2002)}

\usage{
  data(Ley)
}

\description{
  Affymetrix GeneChip (12488 genes and 3 different conditions, each with 3 replicates) experiment was conducted by Dr. Ley (2002) to understand mouse immune response study. For demonstration purposes, we show data from 2 conditions, i.e. 
6 chips, only.
}

\format{
 \item{Ley}{matrix of 12488 genes by 7 columns. First column is the GeneID, 
	  next three columns are replicates in condition one, and 
last three columns are replicates in condition 2.
 For details, contact. Dr. Klaus Ley (kfl3f@virginia.edu) 
 http://hsc.virginia.edu/medicine/basic-sci/biomed/ley/}
}


\references{
	J.K. Lee and M.O.Connell(2003). \emph{An S-Plus library for the analysis of differential expression}. In The Analysis of Gene Expression Data: Methods and Software. Edited by G. Parmigiani, ES Garrett, RA Irizarry ad SL Zegar. Springer, NewYork.
	
	Jain et. al. (2003) \emph{Local pooled error test for identifying
      differentially expressed genes with a small number of replicated microarrays}. (To appear in Bioinformatics)
	See \url{http://hesweb1.med.virginia.edu/bioinformatics/research/expression/index.html}.
}

\keyword{datasets}

\eof
\name{am.trans}
\alias{am.trans}

\title{
  Transform replicated arrays into (A,M) format
}

\description{
  Transforms expression intensity measurements for replicated arrays of a single experimental condition into (A,M) format: A = (xi+xj)/2, M = (xi-xj) where x1, x2,..., xn are individual chips. This function is used in the estimation of within-bin variances in the LPE function, and not typically by the user.
}

\usage{
  am.trans(y)
}

\arguments{
  \item{y}{y is an ngene by n matrix of expression intensity measurements for
     replicated arrays under a single experimental condition.}
}

\value{
  Returns matrix with 2 columns cbind(A.M) and rows comprising all permutations of individual chip columns of the input matrix y. Note that for each pair of chips M is calculated twice, once for xi-xj and once for xj-xi. The resulting marix thus has twice the number of rows as the input matrix y.  
}

\references{
	J.K. Lee and M.O.Connell(2003). \emph{An S-Plus library for the analysis of differential expression}. In The Analysis of Gene Expression Data: Methods and Software. Edited by G. Parmigiani, ES Garrett, RA Irizarry ad SL Zegar. Springer, NewYork.

		Jain et. al. (2003) \emph{Local pooled error test for identifying
      differentially expressed genes with a small number of replicated microarrays}, Bioinformatics, 1945-1951.

	
}

\seealso{
  \code{\link{lpe}}
}

\examples{
  library(LPE)
  # Loading the LPE library
 
  # Test data with duplicates
  temp1 <- matrix(c(1,20,1.5,23),nrow=2)
  am.trans(temp1)
  # It gives a matrix of (4*2) as only two permutaions
  # are possible for each row with duplicates (X1-X2, and X2-X1)
  
  
  # Another test data with three replicates
  temp2 <- matrix(c(1,20,1.5,23,0.8,19),nrow=2)
  am.trans(temp2)
  # Now it returns matrix of (12*2) as there are
  # 6 possible permutations for each row with triplicates
  # (X1-X2, X1-X3, X2-X3, X2-X1, X3-X1 and X3-X2)
  
}

\keyword{methods} % from KEYWORDS.db

\eof
\name{baseOlig.error}
\alias{baseOlig.error}

\title{
 Evaluates LPE variance function of M for quantiles of A within and 
 experimental condition and then interpolates it for all genes.
}

\description{
 Calls baseOlig.error.step1 and baseOlig.error.step2 functions in order to
 calculate the baseline distribution.
}

\usage{
  baseOlig.error(y, stats=median, q=0.01, min.genes.int=10,div.factor=1)
}

\arguments{
 \item{y}{y is a preprocessed matrix or data frame of expression intensities in which columns are expression intensities for a particular experimental
          condition and rows are genes.}
 \item{stats}{It determines whether mean or median is to be used for the replicates}
 \item{q}{q is the quantile width; q=0.01 corresponds to 100 quantiles
 	  i.e. percentiles. Bins/quantiles have equal number of genes and
 	  are split according to the average intensity A.}
 \item{min.genes.int}{Determines the minimum number of genes in a subinterval for selecting the adaptive intervals.}
 \item{div.factor}{Determines the factor by which sigma needs to be divided for
 selecting adaptive intervals.}
}
 
\value{
  Returns object of class baseOlig comprising a data frame with 2 columns: A
  and var M, and rows for each quantile specified. The A column contains
  the median values of A for each quantile/bin and the M columns contains 
  the pooled variance of the replicate chips for genes within each quantile/bin.
}


\references{
	J.K. Lee and M.O.Connell(2003). \emph{An S-Plus library for the analysis of differential expression}. In The Analysis of Gene Expression Data: Methods and Software. Edited by G. Parmigiani, ES Garrett, RA Irizarry ad SL Zegar. Springer, NewYork.

	Jain et. al. (2003) \emph{Local pooled error test for identifying
      differentially expressed genes with a small number of replicated microarrays}, Bioinformatics, 1945-1951.

}


\seealso{
  \code{\link{lpe}}
}

\examples{

  # Loading the library and the data
  library(LPE)
  data(Ley)
  
  dim(Ley)
  # Gives 12488 by 7
  Ley[1:3,]
   # Returns 
  #       ID           c1   c2   c3    t1    t2    t3
#   1  AFFX-MurIL2_at 4.06 3.82 4.28 11.47 11.54 11.34
#   2 AFFX-MurIL10_at 4.56 2.79 4.83  4.25  3.72  2.94
#   3  AFFX-MurIL4_at 5.14 4.10 4.59  4.67  4.71  4.67

  Ley[,2:7] <- preprocess(Ley[,2:7],data.type="MAS5")
  
  subset <- 1:1000
  Ley.subset <- Ley[subset,]
  
  # Finding the baseline distribution of subset of the data
  # condition one (3 replicates)
  var.1 <- baseOlig.error(Ley.subset[,2:4], q=0.01)
  dim(var.1)
  # Returns a matrix of 1000 by 2 (A,M) format
}
\keyword{methods} % from KEYWORDS.db

\eof
\name{baseOlig.error.step1}
\alias{baseOlig.error.step1}

\title{
 Evaluates LPE variance function of M for quantiles of A within and experimental condition by divinding the A in 100 intervals.
}

\description{
  Genes are placed in bins/quantiles according to their average
  expression intensity. The function baseOlig.error calculates a 
  pooled variance of M for genes within these bins/quantiles of A 
  for the replicates of the experimental condition contained in y.
  Here the assumption is that variance of the genes in each interval
  is similar.
}

\usage{
  baseOlig.error.step1(y, stats=median, q=0.01, df=10)
}

\arguments{
 \item{y}{y is a preprocessed matrix or data frame of expression intensities in which columns are expression intensities for a particular experimental
          condition and rows are genes.}
 \item{stats}{It determines whether mean or median is to be used for the replicates}
 \item{q}{q is the quantile width; q=0.01 corresponds to 100 quantiles
 	  i.e. percentiles. Bins/quantiles have equal number of genes and
 	  are split according to the average intensity A.}
 \item{df}{df stands for degrees of freedom. It is used in 
 	   smooth.spline function to interpolate the variances
	   of all genes. Default value is 10.}
}
 
\value{
  Returns object of class baseOlig, comprising a data frame with 2 columns: A
  and var M. The A column contains the median values of each gene 
  and the M columns contains the corresponding variance.  Number of 
  rows of the data-frame is same as that of the number of genes.
}


\references{
	J.K. Lee and M.O.Connell(2003). \emph{An S-Plus library for the analysis of differential expression}. In The Analysis of Gene Expression Data: Methods and Software. Edited by G. Parmigiani, ES Garrett, RA Irizarry ad SL Zegar. Springer, NewYork.

	Jain et. al. (2003) \emph{Local pooled error test for identifying
      differentially expressed genes with a small number of replicated microarrays}, Bioinformatics, 1945-1951.

}


\seealso{
  \code{\link{lpe}}
}

\examples{

  # Loading the library and the data
  library(LPE)
  data(Ley)
  
  dim(Ley)
  # Gives 12488 by 7
  Ley[1:3,]
   # Returns 
  #       ID           c1   c2   c3    t1    t2    t3
#   1  AFFX-MurIL2_at 4.06 3.82 4.28 11.47 11.54 11.34
#   2 AFFX-MurIL10_at 4.56 2.79 4.83  4.25  3.72  2.94
#   3  AFFX-MurIL4_at 5.14 4.10 4.59  4.67  4.71  4.67

  Ley[1:1000,2:7] <- preprocess(Ley[1:1000,2:7],data.type="MAS5")
  # Finding the baseline distribution of subset of the data
  # condition one (3 replicates)
  var.1 <- baseOlig.error.step1(Ley[1:1000,2:4], q=0.01)
  dim(var.1)
  # Returns a matrix of 1000 by 2 (A,M) format
}
\keyword{methods} % from KEYWORDS.db

\eof
\name{baseOlig.error.step2}
\alias{baseOlig.error.step2}

\title{
 Evaluates LPE variance function of M for quantiles of A within and 
 experimental condition. It is based on the adaptive number of intervals.
}

\description{
 Similar to baseOlig.error.step1 function, except that now the number of bins
 are chosen adaptively instead of fixed 100.
 }

\usage{
  baseOlig.error.step2(y,baseOlig.error.step1.res, df=10, stats=median, min.genes.int=10, div.factor=1)
}

\arguments{
 \item{y}{y is a preprocessed matrix or data frame of expression 
      intensities in which columns are expression intensities for
      a particular experimental condition and rows are genes.}
 \item{baseOlig.error.step1.res}{It is the result obtained from 
 	baseOlig.error.step1 function, in which number of bins are fixed=100}
 \item{df}{df stands for degrees of freedom. It is used in 
 	   smooth.spline function to interpolate the variances
	   of all genes. Default value is 10.}
 \item{stats}{It determines whether mean or median is to be used for the replicates}
  \item{min.genes.int}{Determines the minimum number of genes in a subinterval for selecting the adaptive intervals.}
 \item{div.factor}{Determines the factor by which sigma needs to be divided for
 selecting adaptive intervals.}

}
 
\value{
 Returns object of class baseOlig comprising a data frame with 2 columns: A
  and var M, and rows for each quantile specified. The A column contains
  the median values of A for each quantile/bin and the M columns contains 
  the pooled variance of the replicate chips for genes within each quantile/bin.
}


\references{
	J.K. Lee and M.O.Connell(2003). \emph{An S-Plus library for the analysis of differential expression}. In The Analysis of Gene Expression Data: Methods and Software. Edited by G. Parmigiani, ES Garrett, RA Irizarry ad SL Zegar. Springer, NewYork.

	Jain et. al. (2003) \emph{Local pooled error test for identifying
      differentially expressed genes with a small number of replicated microarrays}, Bioinformatics, 1945-1951.

}


\seealso{
  \code{\link{lpe}}
}

\examples{

  # Loading the library and the data
  library(LPE)
  data(Ley)
  
  dim(Ley)
  # Gives 12488 by 7
  Ley[1:3,]
   # Returns 
  #       ID           c1   c2   c3    t1    t2    t3
#   1  AFFX-MurIL2_at 4.06 3.82 4.28 11.47 11.54 11.34
#   2 AFFX-MurIL10_at 4.56 2.79 4.83  4.25  3.72  2.94
#   3  AFFX-MurIL4_at 5.14 4.10 4.59  4.67  4.71  4.67

  Ley[1:1000,2:7] <- preprocess(Ley[1:1000,2:7],data.type="MAS5")
  # Finding the baseline distribution of subset of the data
  # condition one (3 replicates)
  var.1 <- baseOlig.error.step1(Ley[1:1000,2:4], q=0.01, df=10)
  dim(var.1)
  var.11 <- baseOlig.error.step2(Ley[1:1000,2:4], var.1, df=10)
  # Returns a matrix of 1000 by 2 (A,M) format
}
\keyword{methods} % from KEYWORDS.db

\eof
\name{fdr.adjust}
\alias{fdr.adjust}

\title{
  FDR adjustment procedures
}

\description{
  Based on the type of adjustment, eg: resampling, BH, BY, etc, 
  calls appropriate functions for fdr adjustment
}

\usage{
 fdr.adjust(lpe.result,adjp="resamp",target.fdr=c(10^-3 ,seq(0.01,0.10,0.01), 0.15, 0.20, 0.50),iterations=5,ALL=FALSE )
}

\arguments{
 \item{lpe.result}{Data frame obtained from calling lpe function}
 \item{adjp}{Type of adjustment procedure}
 \item{target.fdr}{Desired FDR level (used only for resampling
 	based adjustment)}
 \item{iterations}{Number of iterations for stable z-critical.}
 \item{ALL}{If TRUE, the FDR corresponding to all the z-statistics, i.e.
 	for every gene intensity is given.}
}


\details{
 Returns the output similar to lpe function, including adjusted fdr
 }

\references{
	J.K. Lee and M.O.Connell(2003). \emph{An S-Plus library for the analysis of differential expression}. In The Analysis of Gene Expression Data: Methods and Software. Edited by G. Parmigiani, ES Garrett, RA Irizarry ad SL Zegar. Springer, NewYork.
	

	Jain et. al. (2003) \emph{Local pooled error test for identifying
      differentially expressed genes with a small number of replicated microarrays}, Bioinformatics, 1945-1951.

	}

\examples{

 # Loading the library and the data
 library(LPE)
 data(Ley)
 
 dim(Ley)
 # Gives 12488*7 
 # First column is ID.

 Ley[,2:7] <- preprocess(Ley[,2:7],data.type="MAS5")

 # Subsetting the data
 subset.Ley <- Ley[1:1000,]
  
   
 # Finding the baseline distribution of condition 1 and 2.
 var.1 <- baseOlig.error(subset.Ley[,2:4], q=0.01)
 var.2 <- baseOlig.error(subset.Ley[,5:7], q=0.01)

 # Applying LPE
 lpe.result <- lpe(subset.Ley[,2:4],subset.Ley[,5:7], var.1, var.2,
                probe.set.name=subset.Ley[,1])


 final.result <- fdr.adjust(lpe.result, adjp="resamp", target.fdr=c(0.01,0.05), iterations=1)

}

\keyword{methods} % from KEYWORDS.db



 


  

\eof
\name{fixbounds.predict.smooth.spline}
\alias{fixbounds.predict.smooth.spline}

\title{
	Makes the predicted variance non negative
}

\description{
	Makes the predicted variance non negative
}

\usage{
fixbounds.predict.smooth.spline(object,x, deriv=0)
}

\arguments{
  \item{object}{variance from baseOlig.error function}
  \item{x}{vector for which variance needs to be predicted}
  \item{deriv}{derivative of the vetor required, default =0}
}

\value{
  Returns the predicted variance for the given vector based on the
  baseline error distribution. Maximum and minimum predicted values 
  for the vetor are same as those of baseline error distribution 
}

\references{
	J.K. Lee and M.O.Connell(2003). \emph{An S-Plus library for the analysis of differential expression}. In The Analysis of Gene Expression Data: Methods and Software. Edited by G. Parmigiani, ES Garrett, RA Irizarry ad SL Zegar. Springer, NewYork.
	
	Jain et. al. (2003) \emph{Local pooled error test for identifying
      differentially expressed genes with a small number of replicated microarrays}, Bioinformatics, 1945-1951.



}


\examples{
# Loading the library and the data
 library(LPE)
 data(Ley)
 
 dim(Ley)
 # Gives 12488*7 
 # First column is ID.


 # Subsetting the data
 subset.Ley <- Ley[1:1000,]
  
  subset.Ley[,2:7] <- preprocess(subset.Ley[,2:7],data.type="MAS5")
  # preprocess the data
  
 # Finding the baseline distribution of condition 1 and 2.
 var.1 <- baseOlig.error(subset.Ley[,2:4], q=0.01)
  median.x <- apply(subset.Ley[,2:4], 1, median)

 sf.x <- smooth.spline(var.1[, 1], var.1[, 2], df = 10)
  
 var.test <- fixbounds.predict.smooth.spline(sf.x, median.x)$y

}

\keyword{methods} %from KEYWORD.db

\eof
\name{iqr}
\alias{iqr}

\title{
  Inter-quartile range
}

\description{
  Finds inter-quartile range of the data = {75th percentile - 25th percentile}.
}

\usage{
  iqr(x)
}

\arguments{
  \item{x}{x is a vector for which IQR has to be found.}
}

\value{
  Returns a numeric value representing the difference of 75th percentile 
  and 25th percentile of the vector. It is used for normalization across 
  the chips - basic assumption is that net differential expression of 
  the middle half of the genes in microarray experiment is zero, which 
  is conservative assumption as typically only 5-10% genes show 
  differential expression.
}

\references{
	J.K. Lee and M.O.Connell(2003). \emph{An S-Plus library for the analysis of differential expression}. In The Analysis of Gene Expression Data: Methods and Software. Edited by G. Parmigiani, ES Garrett, RA Irizarry ad SL Zegar. Springer, NewYork.
	
	Jain et. al. (2003) \emph{Local pooled error test for identifying
      differentially expressed genes with a small number of replicated microarrays}, Bioinformatics, 1945-1951.



}



\seealso{
  \code{\link{lpe}}
}

\examples{
  library(LPE)
  # Loading the LPE library
 
  iqr(1:5) 
  % returns 2 [= 75th percentile(4)- 25th percentile(2)]
}

\keyword{methods} % from KEYWORDS.db

\eof
\name{lowess.normalize}
\alias{lowess.normalize}

\title{
  lowess normalization of the data (based on M vs A graph)
}

\description{
 All the chips are normalized w.r.t. 1st chip 
}

\usage{
  lowess.normalize(x,y)
}

\arguments{
  \item{x}{x is the chip data w.r.t. which other chips would be normalized}
  \item{y}{y is the chip data which would be normalized}
}

\value{
  Returns the lowess normalized chip intensity.
}

\references{
	J.K. Lee and M.O.Connell(2003). \emph{An S-Plus library for the analysis of differential expression}. In The Analysis of Gene Expression Data: Methods and Software. Edited by G. Parmigiani, ES Garrett, RA Irizarry ad SL Zegar. Springer, NewYork.
	

	Jain et. al. (2003) \emph{Local pooled error test for identifying
      differentially expressed genes with a small number of replicated microarrays}, Bioinformatics, 1945-1951.


}



\seealso{
  \code{\link{lpe}}
}

\examples{
  library(LPE)
  # Loading the LPE library
 
  data(Ley)
  # Loading the data set
  dim(Ley) #gives 12488 * 7
  Ley[1:3,]

  Ley[1:1000,2:7] <- preprocess(Ley[1:1000,2:7],data.type="MAS5")
  Ley[1:3,]
 
}

\keyword{methods} % from KEYWORDS.db

\eof
\name{lpe}
\alias{lpe}

\title{
  Evaluates local pooled error significance test
}

\description{
  The local pooled error test attempts to reduce dependence on the within-gene estimates in tests for differential expression, by pooling error estimates within regions of similar intensity. Note that with the large number of genes there will be genes with low within-gene error estimates by chance, so that some signal-to-noise ratios will be large regardless of mean expression intensities and fold-change. The local pooled error attempts to avert this by combining within-gene error estimates with those of genes with similar expression intensity.
}

\usage{
  lpe(x, y, basevar.x,basevar.y, df=10, array.type="olig", 
      probe.set.name="OLIG.probe.name", trim.percent=5)
}

\arguments{
 \item{x}{Replicated data from first experimental condition (as matrix 
			or data-frame)}.
  	  
 \item{y}{Replicated data from second experimental condition (as matrix 
			or data-frame)}.
 \item{basevar.x}{Baseline distribution of first condition obtained from 
		 function baseOlig.error}
 \item{basevar.y}{Baseline distribution of second condition obtained from 
		 function baseOlig.error}
 \item{df}{Degrees of freedom used in fitting smooth.spline to estimates
	   of var.M for bins in A}
 \item{array.type}{Currently supports oligo arrays}
 \item{probe.set.name}{Gene IDs}
 \item{trim.percent}{Percent of (A, var.M) estimates to trim from low 
	end of A}
 }

\details{
  The LPE test statistic numerator is the difference in medians between the
  two experimental conditions. The test statistic denominator is the combined pooled standard error for the two experimental conditions obtained by looking up the var.M from each baseOlig.error variance function. The conversion to p-values is based on the Gaussian distribution for difference if order statistics (medians). The user may select bith the smoother degrees of freedom (smaller is smoother) and the trim percent to obtain a variance fucntion to suit particular 
issues i.e. variability of genes with low expression intensity.
}

\value{
  Data frame including x, median of x, y, median of y, median difference of (x,y), pooled standard deviation of difference, LPE p-value, outlier flag, probability of an outlier within x or y.
}


\references{
	J.K. Lee and M.O.Connell(2003). \emph{An S-Plus library for the analysis of differential expression}. In The Analysis of Gene Expression Data: Methods and Software. Edited by G. Parmigiani, ES Garrett, RA Irizarry ad SL Zegar. Springer, NewYork.

	Jain et. al. (2003) \emph{Local pooled error test for identifying
      differentially expressed genes with a small number of replicated microarrays}, Bioinformatics, 1945-1951.

}

\examples{
   
 # Loading the library and the data
 library(LPE)
 data(Ley)
 
 dim(Ley)
 # Gives 12488*7 
 # First column is ID.



 # Subsetting the data
 subset.Ley <- Ley[1:1000,]
  
  subset.Ley[,2:7] <- preprocess(subset.Ley[,2:7],data.type="MAS5")
  
 # Finding the baseline distribution of condition 1 and 2.
var.11 <- baseOlig.error.step1(subset.Ley[,2:4]) 
 var.1 <- baseOlig.error(subset.Ley[,2:4], q=0.01)
 var.2 <- baseOlig.error(subset.Ley[,5:7], q=0.01)
 
 # Applying LPE
 lpe.result <- lpe(subset.Ley[,2:4],subset.Ley[,5:7], var.1, var.2,
		probe.set.name=subset.Ley[,1])
  
}

\keyword{methods} % from KEYWORDS.db

\eof
\name{mt.rawp2adjp}
\alias{mt.rawp2adjp}
\title{Adjusted p-values for simple multiple testing procedures}
\description{
  This function computes adjusted \eqn{p}-values for simple
  multiple testing procedures from a vector of raw (unadjusted)
  \eqn{p}-values. The procedures include the Bonferroni, Holm (1979),
  Hochberg (1988), and Sidak procedures for strong control of the
  family-wise Type I error rate (FWER), and the Benjamini & Hochberg
  (1995) and Benjamini & Yekutieli (2001) procedures for (strong)
  control of the false discovery rate (FDR). 
}
\usage{
mt.rawp2adjp(rawp, proc=c("Bonferroni", "Holm", "Hochberg", "SidakSS", "SidakSD", "BH", "BY"))
}
\arguments{
  \item{rawp}{A vector of raw (unadjusted) \eqn{p}-values for each
    hypothesis under consideration. These could be nominal
    \eqn{p}-values, for example, from t-tables, or permutation
    \eqn{p}-values as given in \code{mt.maxT} and \code{mt.minP}. If the
    \code{mt.maxT} or \code{mt.minP} functions are used, raw
    \eqn{p}-values should be given in the original data order,
    \code{rawp[order(index)]}.} 
  \item{proc}{A vector of character strings containing the names of the
    multiple testing procedures for which adjusted \eqn{p}-values are to
    be computed. This vector should include any of the following:
    \code{"Bonferroni"}, \code{"Holm"}, \code{"Hochberg"},
    \code{"SidakSS"}, \code{"SidakSD"}, \code{"BH"}, \code{"BY"}. 
}
}
\details{
  Adjusted \eqn{p}-values are computed for simple FWER and FDR
  controlling procedures based on a vector of raw (unadjusted)
  \eqn{p}-values.\cr 
\item{Bonferroni}{Bonferroni single-step adjusted \eqn{p}-values
for strong control of the FWER.}
\item{Holm}{Holm (1979) step-down adjusted \eqn{p}-values for
strong control of the FWER.}
\item{Hochberg}{Hochberg (1988) step-up adjusted \eqn{p}-values
for 
strong control of the FWER (for raw (unadjusted) \eqn{p}-values
satisfying the Simes inequality).}
\item{SidakSS}{Sidak single-step adjusted \eqn{p}-values for
strong control of the FWER (for positive orthant dependent test
statistics).}
\item{SidakSD}{Sidak step-down adjusted \eqn{p}-values for
strong control of the FWER (for positive orthant dependent test
statistics).}
\item{BH}{adjusted \eqn{p}-values for the Benjamini & Hochberg
(1995) step-up FDR controlling procedure (independent and positive
regression dependent test statistics).}
\item{BY}{adjusted \eqn{p}-values for the Benjamini & Yekutieli
(2001) step-up FDR controlling procedure (general dependency
structures).} 
}

\value{
A list with components
  \item{adjp}{A matrix of adjusted \eqn{p}-values, with rows
    corresponding to hypotheses and columns to multiple testing
    procedures. Hypotheses are sorted in increasing order of their raw
    (unadjusted) \eqn{p}-values.} 
  \item{index}{A vector of row indices, between 1 and
    \code{length(rawp)}, where rows are sorted according to 
    their raw (unadjusted) \eqn{p}-values. To obtain the adjusted
    \eqn{p}-values in the original data order, use
    \code{adjp[order(index),]}. 
}
}
\references{
Y. Benjamini and Y. Hochberg (1995). Controlling the false discovery
rate: a practical and powerful approach to multiple
testing. \emph{J. R. Statist. Soc. B}. Vol. 57: 289-300.\cr 

Y. Benjamini and D. Yekutieli (2001). The control of the false discovery
rate in multiple hypothesis testing under dependency. \emph{Annals of
  Statistics}. Accepted.\cr 

S. Dudoit, J. P. Shaffer, and J. C. Boldrick (Submitted). Multiple
hypothesis testing in microarray experiments.\cr 

Y. Ge, S. Dudoit, and T. P. Speed (In preparation). Fast algorithm for 
resampling-based \eqn{p}-value adjustment in multiple testing. \cr

Y. Hochberg (1988). A sharper Bonferroni procedure for multiple tests of
significance, \emph{Biometrika}. Vol. 75: 800-802.\cr 

S. Holm (1979). A simple sequentially rejective multiple test
procedure. \emph{Scand. J. Statist.}. Vol. 6: 65-70. 
}
 
\author{
  Sandrine Dudoit, \url{http://www.stat.berkeley.edu/~sandrine},\cr
  Yongchao Ge, \email{gyc@stat.berkeley.edu}. 
}

\seealso{\code{\link{lpe}}}

%\seealso{\code{\link{mt.maxT}}, \code{\link{mt.minP}},
%  \code{\link{mt.plot}}, \code{\link{mt.reject}}, \code{\link{golub}}.}

\examples{
# Loading the library and the data
 library(LPE)
 data(Ley)
 
 dim(Ley)
 # Gives 12488*7 
 # First column is ID.


 # Subsetting the data
 subset.Ley <- Ley[1:1000,]

 subset.Ley[,2:7] <- preprocess(subset.Ley[,2:7],data.type="MAS5")
  
 # Finding the baseline distribution of condition 1 and 2.
 var.1 <- baseOlig.error(subset.Ley[,2:4], q=0.01)
 var.2 <- baseOlig.error(subset.Ley[,5:7], q=0.01)
 
 # Applying LPE
 lpe.result <- lpe(subset.Ley[,2:4],subset.Ley[,5:7], var.1, var.2,
		probe.set.name=subset.Ley[,1])
 fdr.BH <- fdr.adjust(lpe.result, adjp="BH")
}

\keyword{htest}






\eof
\name{n.genes.adaptive.int}
\alias{n.genes.adaptive.int}

\title{
 Calcuates the number of genes in various intervals adaptively.
}

\description{
  Instead of dividing the genes equally in 100 intervals, this
  function divides them adaptively based on three rules:
  a) min. number of genes (default =10), b) max. number of genes = total/100;
  c) based on Median + fraction(SD) from the starting gene of each
  interval
}

\usage{
  n.genes.adaptive.int(baseOlig.error.step1.res,
  		min.genes.int=10, div.factor=1)
}

\arguments{
 \item{baseOlig.error.step1.res}{It is the result 
 		from baseOlig.error.step1 function.}
 \item{min.genes.int}{It is the minimum number of genes in the interval,
 	default=10.}
 \item{div.factor}{(1/div.factor) is the fraction of Standard Deviation
 		which we wish to include in each interval to calculate
		number of genes in each interval}
}
 
\value{
  Returns a vector respresenting the number of genes in each interval.
}


\references{
	J.K. Lee and M.O.Connell(2003). \emph{An S-Plus library for the analysis of differential expression}. In The Analysis of Gene Expression Data: Methods and Software. Edited by G. Parmigiani, ES Garrett, RA Irizarry ad SL Zegar. Springer, NewYork.

	Jain et. al. (2003) \emph{Local pooled error test for identifying
      differentially expressed genes with a small number of replicated microarrays}, Bioinformatics, 1945-1951.

}


\seealso{
  \code{\link{lpe}}
}

\examples{

  # Loading the library and the data
  library(LPE)
  data(Ley)
  
  dim(Ley)
  # Gives 12488 by 7
  Ley[1:3,]
   # Returns 
  #       ID           c1   c2   c3    t1    t2    t3
#   1  AFFX-MurIL2_at 4.06 3.82 4.28 11.47 11.54 11.34
#   2 AFFX-MurIL10_at 4.56 2.79 4.83  4.25  3.72  2.94
#   3  AFFX-MurIL4_at 5.14 4.10 4.59  4.67  4.71  4.67

  Ley[1:1000,2:7] <- preprocess(Ley[1:1000,2:7],data.type="MAS5")
  # Finding the baseline distribution of subset of the data
  # condition one (3 replicates)
  var.1 <- baseOlig.error.step1(Ley[1:1000,2:4], q=0.01)
  dim(var.1)
  # Returns a matrix of 1000 by 2 (A,M) format
  n.genes.subint <- n.genes.adaptive.int(var.1, min.genes.int=10, div.factor=1)
}
\keyword{methods} % from KEYWORDS.db

\eof
\name{permute}
\alias{permute}

\title{
	Calculating all possible permutations of a vector
}
\description{ 
	Given a vector, all possible combinations of vector are obtained 
}
\usage{
	permute(a)
}
\arguments{
 	\item{a}{a is any numeric vector.}
}

\value{
	A vector containing the possible combinations.
}

\details{
 Used in am.trans. Does all permutations for columns within an experimental 
 condition so that A and M can be calculated for all permutations of chips within a treatment.
}


\references{
	J.K. Lee and M.O.Connell(2003). \emph{An S-Plus library for the analysis of differential expression}. In The Analysis of Gene Expression Data: Methods and Software. Edited by G. Parmigiani, ES Garrett, RA Irizarry ad SL Zegar. Springer, NewYork.
	
	Jain et. al. (2003) \emph{Local pooled error test for identifying
      differentially expressed genes with a small number of replicated microarrays}, Bioinformatics, 1945-1951.

}

\seealso{
  \code{\link{lpe}}
}

\examples{
  # Loading LPE library
  library(LPE)
  
  # A test vector
  permute(1:3)  
  
 # Returns a 2 by 3 matrix 
 #       [,1] [,2] [,3]
 #  [1,]   2    3    1
 #  [2,]   3    1    2

}
\keyword{methods} %from KEYWORD.db

\eof
\name{preprocess}
\alias{preprocess}

\title{
  Preprocessing the data (IQR normalization, thresholding, log-
  transformation, and lowess normalization)
}

\description{
  Finds inter-quartile range of the data = (75th percentile - 25th percentile),
  thresholds low intensity MAS4, MAS5 and dChip data to 1, then log transforms
  the data (base 2), and dows lowess normalization  
}

\usage{
  preprocess(x, data.type="MAS5",threshold=1,LOWESS=TRUE)
}

\arguments{
  \item{x}{x is the data-set which needs preprocessing.}
  \item{data.type}{Three types of data accepted in the current version :
          MAS4 (Microarray suite software) , MAS5 and dChip}
  \item{threshold}{threshold is the 'thresholding value' below which
                  all data would be thresholded (default = 1).}
  \item{LOWESS}{LOWESS is a logical variable which determines if lowess 
  	normalization needs to be performed.}

}

\value{
  Returns a data-set of same dimensions as that of the input data. It has 
  IQR normalization for MAS4 and MAS5 data. Low intensities of MAS4, MAS5
  and dChip data are thresholded to 1. Then data is transformed to base 2, and
  finally lowess based normalization is applied.
}

\references{
	J.K. Lee and M.O.Connell(2003). \emph{An S-Plus library for the analysis of differential expression}. In The Analysis of Gene Expression Data: Methods and Software. Edited by G. Parmigiani, ES Garrett, RA Irizarry ad SL Zegar. Springer, NewYork.
	
	Jain et. al. (2003) \emph{Local pooled error test for identifying
      differentially expressed genes with a small number of replicated microarrays}, Bioinformatics, 1945-1951.
}



\seealso{
  \code{\link{lpe}}
}

\examples{
  library(LPE)
  # Loading the LPE library
 
  data(Ley)
  # Loading the data set
  dim(Ley) #gives 12488 * 7
  Ley[1:3,]

  Ley[1:1000,2:7] <- preprocess(Ley[1:1000,2:7],data.type="MAS5",
  	threshold=1, LOWESS=TRUE)
  Ley[1:3,]
 
}

\keyword{methods} % from KEYWORDS.db

\eof
\name{quan.norm}
\alias{quan.norm}

\title{
  Finding quartile range
}

\description{
  Finds quartile range of the data (default is IQR = 75th percentile - 25th percentile).
}

\usage{
  quan.norm(x, percent=50)
}

\arguments{
  \item{x}{x is a vector for which quartile range has to be found.}
  \item{percent}{Percentage for which quartile range is needed}
}

\value{
  Returns a numeric value representing the difference of 75th percentile 
  and 25th percentile of the vector. It is used for normalization across 
  the chips - basic assumption is that net differential expression of 
  the middle half of the genes in microarray experiment is zero, which 
  is conservative assumption as typically only 5-10% genes show 
  differential expression.
}

\references{
	J.K. Lee and M.O.Connell(2003). \emph{An S-Plus library for the analysis of differential expression}. In The Analysis of Gene Expression Data: Methods and Software. Edited by G. Parmigiani, ES Garrett, RA Irizarry ad SL Zegar. Springer, NewYork.
	
	Jain et. al. (2003) \emph{Local pooled error test for identifying
      differentially expressed genes with a small number of replicated microarrays}, Bioinformatics, 1945-1951.
}

\seealso{
  \code{\link{lpe}}
}

\examples{
  library(LPE)
  # Loading the LPE library
 
  quan.norm(1:5) 
  % returns 2 [= 75th percentile(4)- 25th percentile(2)]
}

\keyword{methods} % from KEYWORDS.db

\eof
\name{quartile.normalize}
\alias{quartile.normalize}

\title{
  Normalization based on quartile range
}

\description{
Does Normalization based on quartile range 
}

\usage{
  quartile.normalize(x, percent=50)
}

\arguments{
  \item{x}{x is a matrix or dataframe on which normalization has to be performed.}
  \item{percent}{Percentage for which normalization is needed}
}

\value{
Returns the normalized data based on quartile normalization 
}

\references{
	J.K. Lee and M.O.Connell(2003). \emph{An S-Plus library for the analysis of differential expression}. In The Analysis of Gene Expression Data: Methods and Software. Edited by G. Parmigiani, ES Garrett, RA Irizarry ad SL Zegar. Springer, NewYork.
	
	Jain et. al. (2003) \emph{Local pooled error test for identifying
      differentially expressed genes with a small number of replicated microarrays}, Bioinformatics, 1945-1951.
}

\seealso{
  \code{\link{lpe}}
}

\examples{
  library(LPE)
  # Loading the LPE library

  data(Ley) 
 
 dim(Ley)
 # Gives 12488*7
 # First column is ID.

  subset <- 1:1000
  Ley[subset,2:7] <- quartile.normalize(Ley[subset,2:7],percent=50)


}

\keyword{methods} % from KEYWORDS.db

\eof
\name{resamp.adj}
\alias{resamp.adj}

\title{
   Resampling based fdr adjustment
}

\description{
  Adjusts the fdr based on rank invariant genes
}

\usage{
 resamp.adj(x,y, q=0.01, iterations=5, min.genes.int=10) 
}

\arguments{
 \item{x}{Replicated data from first experimental condition (as matrix 
			or data-frame)}.
 \item{y}{Replicated data from second experimental condition (as matrix 
			or data-frame)}.
 \item{q}{q is the quantile width; q=0.01 corresponds to 100 quantiles}.
 \item{iterations}{Number of iterations to be performed to obtain critical z-statistics}.
 \item{min.genes.int}{Determines the minimum number of genes in a subinterval for selecting the adaptive intervals.}
}

\details{
 Returns the z-statistics for the null distribution, obtained from resampling
 the rank invariant genes within each quantile. These z-statistic values are
 compared with z-statiscs from the original data, and fdr is calculated.
}


\references{
	J.K. Lee and M.O.Connell(2003). \emph{An S-Plus library for the analysis of differential expression}. In The Analysis of Gene Expression Data: Methods and Software. Edited by G. Parmigiani, ES Garrett, RA Irizarry ad SL Zegar. Springer, NewYork.
	
	Jain et. al. (2003) \emph{Local pooled error test for identifying
      differentially expressed genes with a small number of replicated microarrays}, Bioinformatics, 1945-1951.
      
      }


\examples{
 
  # Loading the library and the data
 library(LPE)
 data(Ley)
 
 dim(Ley)
 # Gives 12488*7 
 # First column is ID.

 # Subsetting the data
 subset.Ley <- Ley[1:1000,]
  
  subset.Ley[,2:7] <- preprocess(subset.Ley[,2:7],data.type="MAS5")
   
 # Finding the baseline distribution of condition 1 and 2.
 var.1 <- baseOlig.error(subset.Ley[,2:4], q=0.01)
 var.2 <- baseOlig.error(subset.Ley[,5:7], q=0.01)
 
 # Applying LPE
 lpe.result <- lpe(subset.Ley[,2:4],subset.Ley[,5:7], var.1, var.2,
		probe.set.name=subset.Ley[,1])
  

 
 z.stats.null <- resamp.adj(subset.Ley[,2:4], subset.Ley[,5:7], q=0.01, iterations=2,min.genes.int=10 )

}

\keyword{methods} % from KEYWORDS.db



\eof
