.packageName <- "splicegear"
## Laurent 2003
## Implemented from Genome Research, Hu et. al. , p.1244


## x <- matrix(1:20, 5, 4); tissue <- factor(c("a", "a", "b", "b"))

## SPLICE algorithm:
getRelSignStrength <- function(x, tissue=as.factor(1:ncol(x)), fun=mean, nipt=30, nitt=30, ...) {
  ## x: matrix. One probe per line, one column per chip
  ## tissue: a factor for the 'tissue' covariate
  ## returns the 'RSS'
  
  ## cutoff values
  x[ x < 20 ] <- 20
  x[ x > 5000 ] <- 5000

  ## mean (or 'fun') of all probe pairs in tissue X 
  avgDix.perx <- unlist(tapply(seq(along=tissue), tissue, function(y) fun(x[ , y])))
  avgDix.perx.indices <- tapply(seq(along=tissue), tissue, function(y) y)
  avgDix <- matrix(NaN, nc=ncol(x), nr=nrow(x))
  for (i in seq(along=avgDix.perx.indices))
    avgDix[, avgDix.perx.indices[[i]]] <- avgDix.perx[i]

  ## mean or 'fun' of a particular probe pair across different tissues
  ##avgDi <- t(apply(x, 1, function(y) tapply(y, tissue, fun)), ...)
  avgDi <- apply(x, 1, fun, ...)
  
  relsignstr <- x / avgDix[, as.integer(tissue)]

  ## non-informative probe threshold
  nip <- avgDi < nipt
  attr(relsignstr, "nip") <- nip

  ## non-informative tissue threshold
  nit <- avgDix < nitt
  attr(relsignstr, "nit") <- nit

  return(relsignstr)
}

getFinalRatio <- function(x, tissue=as.factor(1:ncol(x)), fun=mean, ...) {
  x <- getRelSignStrength(x, tissue=tissue, fun=fun, ...)
  ex <- sapply(tissue, function(x) tissue != x, simplify=FALSE)
  avgrss <- t(apply(x, 1, function(y) unlist(lapply(ex, function(z) fun(y[z])))))
  fr <- log(x / avgrss)

  return(fr)
}

as.data.frame.SpliceSites <- function(x, row.names=NA, optional=NA) {
  
  if (! inherits(x, "SpliceSites"))
    stop("the argument should inherit class 'SpliceSites'")
  
  nr.typeII <- length(x@spsiteIIpos)

  pdat.I <- x@spsiteIpos.pData@pData
  rownames(pdat.I) <- seq(1, nrow(pdat.I), length=nrow(pdat.I))
  pdat.II <- x@spsiteIIpos.pData@pData
  rownames(pdat.II) <- seq(nrow(pdat.I)+1, nrow(pdat.I) + nr.typeII, length=nr.typeII)

  if (nrow(pdat.I) == 0 && nrow(pdat.II) == 0) {
    m <- list()
  } else if (nrow(pdat.I) == 0) {
    m <- pdat.II
  } else if (nrow(pdat.II) == 0) {
    m <- pdat.I
  } else {
    m <- merge(pdat.I, pdat.II, by.x = c(0, seq(along=pdat.I)), by.y = c(0, seq(along=pdat.II)), all = TRUE)
  }
  
  rv <- do.call("data.frame", c(list(begin = c(x@spsiteIpos[, 1], x@spsiteIIpos),
                                     end = c(x@spsiteIpos[, 2], rep(NA, nr.typeII)),
                                     lapply(m, function(x) {x})
                                     ))
                )
  
  
  return(rv)
}
as.data.frame.SpliceExprSet <- function(x, row.names=NA, optional=NA) {
  
  if (! inherits(x, "SpliceExprSet"))
    stop("the argument should inherit class 'SpliceExprSet'")

  spSites <- x@spliceSites
  eset <- x@eset
  probes <- x@probes
  
  nc.eset <- ncol(exprs(eset))
  nr.eset <- nrow(exprs(eset))

  ## sanity check
  ## (because slots of objects can be tweaked by the user)
  if (nr.eset != nrow(probes@pos)) {
    stop("inconsistency beetween slots 'eset' and 'probes'")
  }
  
  ## build the probe position column
  i.probes <- seq(1, nr.eset)
  i.probes.expand <- rep(i.probes, nc.eset)
  
  ## build the 'is in type x' columns
  probeOnSpSite <- isProbeOnSpliceSite(probes, spSites)

  ## build the probe level intensities columns
  rv.eset <- as.data.frame.exprSet(eset)
  
  rv <- do.call("data.frame", c(list(begin = probes@pos[, 1][i.probes.expand],
                                     end = probes@pos[, 2][i.probes.expand],
                                     isintypeI = probeOnSpSite$isintypeI[i.probes.expand],
                                     isintypeII = probeOnSpSite$isintypeII[i.probes.expand]),
                                lapply(rv.eset, function(x) return(x)),
                                lapply(probes@info, function(x, i) return(x[i]), i.probes.expand)
                                )
                )
  
  return(rv)
}
barplot.SpliceSites <- function(height, type.as=c("typeI", "typeII", "all"),
                                info="tissue", ...) {

  type.as <- match.arg(type.as)
  dt <- switch(type.as,
               typeI = pData(height@spsiteIpos.pData),
               typeII = pData(height@spsiteIIpos.pData),
               all = rbind(pData(height@spsiteIpos.pData), pData(height@spsiteIIpos.pData))
               )

  if (nrow(dt)>0 && ! info %in% colnames(dt))
    stop(paste(paste("parameter 'info=", info,"' is not one of ", sep=""),
               colnames(dt), collapse=", "))

  dt <- dt[[info]]

  if (is.null(dt))
    ct <- 0
  else
    ct <- table(dt)

  r <- barplot(ct, ...)

  invisible(r)
  
}
buildSpliceSites <- function(xml, verbose=TRUE) {

  get.pData <- function(this.result, what) {
    leaf <- this.result[[what]]
    if (is.null(leaf)) {
      r <- NA
    } else {
      r <- as.character(xmlChildren(leaf)$text)[6]
    }
    return(r)
  }
  
  resultQuery <- xmlChildren(xml$doc$children[["ResultQuery"]])

  query.status <- as.character(xmlChildren(resultQuery[["query"]][["query-status"]])$text)[6]

  if (query.status != "OK") {
    stop(paste("Query status is:", query.status))
  }
  
  entries.i <- which(names(resultQuery) == "Entry")

  n.EST.hit <- sum(unlist(lapply(resultQuery[entries.i],
                                 function (x) length(names(x)) - 1)))
  
  spsites.list <- vector("list", length=length(entries.i))
  names(spsites.list) <- rep(NULL, length=length(entries.i))
  ##spsites.list <- vector("list", length=n.EST.hit)
  ##names(spsites.list) <- rep(NULL, n.EST.hit)

  ##i.list <- 1

  if (verbose)
    cat(length(entries.i), " entrie(s) in the set.\n")
  
  for (i in seq(along=entries.i)) {
    e.i <- entries.i[i]

    entries.k <- which(names(resultQuery[[e.i]]) == "Alt-Splice")
    ug.id <- resultQuery[[e.i]][["Reference-sequence"]][["ug-cluster-id"]]
    ug.id <- as.character(xmlChildren(ug.id)$text)[6]
    seq.len <- resultQuery[[e.i]][["Reference-sequence"]][["ref-len"]]
    seq.len <- as.integer(as.character(xmlChildren(seq.len)$text)[6])
    seq.string <- resultQuery[[e.i]][["Reference-sequence"]][["ref-seq"]]
    if (is.null(seq.string))
      seq.string <- ""
    else
      seq.string <- as.character(xmlChildren(seq.string)$text)[6]
    
    n.ASinfo <- sum(unlist(lapply(resultQuery[[e.i]][entries.k],
                                 function(x) sum(names(x) == "Hit-info"))))
    spsiteIpos <- vector("list", length = n.ASinfo)
    spsiteIIpos <- vector("list", length = n.ASinfo)
    pData.tissue <- vector("character", length = n.ASinfo)
    pData.histology <- vector("character", length = n.ASinfo)
    pData.cellline <- vector("character", length = n.ASinfo)
    pData.site <- vector("character", length = n.ASinfo)
    ##pData.other <- vector("list", length=n.ASinfo)
    j.offset <- 0
    
    if (verbose)
      cat(" entrie ", i, " has ", n.ASinfo, " element(s).\n")
  
    for (k in seq(along=entries.k)) {
      e.k <- entries.k[k]
      
      i.ASinfo <- names(resultQuery[[e.i]][[e.k]]) == "Hit-info"

      entries.j <- which(i.ASinfo)

      this.result <- resultQuery[[e.i]][[e.k]]
      typeAS <- xmlAttrs(this.result)["Type"]
      typeAS <- as.integer(typeAS)

      if (typeAS == 1) {
        pos.i <- which(names(this.result[["Site-info"]]) == "Pos")
        pos1 <- this.result[["Site-info"]][[pos.i[1]]]
        pos1 <- as.character(xmlChildren(pos1)$text)[6]
        pos2 <- this.result[["Site-info"]][[pos.i[2]]]
        pos2 <- as.character(xmlChildren(pos2)$text)[6]
      }
      if (typeAS == 2) {
        pos2 <- resultQuery[[e.i]][[e.k]][["Site-info"]][["Pos"]]
        pos2 <- as.character(xmlChildren(pos2)$text)[6]
      }
      
      if (verbose)
        cat("  sub-entrie ", k, " has ", length(entries.j), " element(s).\n")

      for (j in seq(along=entries.j)) {
        ##cat ("j=", j, "(j.offset=", j.offset, ")\n")
        j.offset <- j.offset + 1
        e.j <- entries.j[j]
        if (typeAS == 1) {
          spsiteIpos[[j.offset]] <- c(as.integer(pos1),
                                      as.integer(pos2))
        }
        if (typeAS == 2) {
          spsiteIIpos[[j.offset]] <- c(as.integer(pos2))
        }
        pData.tissue[j.offset] <- get.pData(this.result[[e.j]], "Hit-tissue")
        pData.histology[j.offset] <- get.pData(this.result[[e.j]], "Hit-histology")
        
        ##pData.other[[j.offset]] <- other.pdata
        pData.site[j.offset] <- k
      }

    }
    spsiteIpos.i <- ! unlist(lapply(spsiteIpos, is.null))
    
    if (sum(spsiteIpos.i) == 0)
      spsiteIpos <- matrix(0, 0, 0)
    else
      spsiteIpos <- matrix(unlist(spsiteIpos[spsiteIpos.i]), nc=2, byrow=TRUE)
    
    spsiteIIpos.i <- ! unlist(lapply(spsiteIIpos, is.null))
    
    if (sum(spsiteIIpos.i) == 0)
      spsiteIIpos <- integer(0)
    else
      spsiteIIpos <- unlist(spsiteIIpos[spsiteIIpos.i])
    
    ##spsiteIpos.pData <- new("phenoData", pData=data.frame(tissue = spsiteIpos.pData.tissue[spsiteIpos.i]))
    spsiteIpos.pData <- new("phenoData")
    spsiteIpos.pData@pData <-
      data.frame(tissue = pData.tissue[spsiteIpos.i],
                 histology = pData.histology[spsiteIpos.i],
                 site = pData.site[spsiteIpos.i])
    spsiteIIpos.pData <- new("phenoData")
    spsiteIIpos.pData@pData <-
      data.frame(tissue = pData.tissue[spsiteIIpos.i],
                 histology = pData.histology[spsiteIIpos.i],
                 site = pData.site[spsiteIIpos.i])
    
    spsites.list[[i]] <- new("SpliceSites", seq.length = seq.len,
                             seq = seq.string,
                             spsiteIpos = spsiteIpos,
                             spsiteIIpos = spsiteIIpos,
                             spsiteIpos.pData = spsiteIpos.pData,
                             spsiteIIpos.pData = spsiteIIpos.pData
                             )
    
    names(spsites.list)[i] <- ug.id
    ##i.list <- i.list + 1
  }
  
  return(spsites.list)
  
}


# group.pdata <- function(spsites, type.as) {

#   if (type.as == 1) {
#     u <- unique(spsites@spsiteIpos)
#     pdat <- pData(spsites@spsiteIpos.pData)
#     pos <- spsites@spsiteIpos
#   } else if (type.as ==2) {
#     pos <- matrix(spsites@spsiteIIpos, nc = 1)
#     u <- unique(spsites@spsiteIIpos)
#     pdat <- pData(spsites@spsiteIIpos.pData)
#   }
  
#   u.n <- nrow(u)
  
#   r <- vector("list", length=u.n)
#   names(r) <- apply(u, 1, paste, collapse="..")

#   for (r.i in seq(1, u.n, length=u.n)) {
#     i <- apply(pos, 1, function(x) all(x == u[r.i, ]))
#     r[[r.i]] <- pdat[i, ]
#   }

#   return(r)
# }
require(Biobase, quietly=TRUE) || stop("Could not load the package 'Biobase'")

  
setClass("SpliceSites",
         representation(
                        seq = "character",   # the genomic sequence (if available)
                        seq.length = "integer", 
                        spsiteIpos = "matrix", # a two-columns matrix (window for type I splice site),
                          spsiteIIpos = "integer", # position for type II splice site
                        spsiteIIIpos = "matrix", # a two-columns matrix (window for type III splice site),
                        spsiteIpos.pData = "phenoData",
                        spsiteIIpos.pData = "phenoData",
                        spsiteIIIpos.pData = "phenoData"
                        ))


  setMethod("show", signature(object = "SpliceSites"),
            function(object) {
              cat("Alternative splicing sites (SpliceSites):\n")
              cat("\tseq is ", object@seq.length, " bp long", sep="")
              if (object@seq == "")
                cat(" (warning: sequence not included).\n")
              else
                cat(".\n")
              cat("\t", nrow(object@spsiteIpos), " type I splice site(s)\n", sep="")
              show(object@spsiteIpos.pData)
              cat("\t", length(object@spsiteIIpos), " type II splice site(s)\n", sep="")
              show(object@spsiteIIpos.pData)
              cat("\t", nrow(object@spsiteIIIpos), " type III splice site(s)\n", sep="")
            })


if( !isGeneric("plot") )
  setGeneric("plot", function(x, y, ...)
             standardGeneric("plot"))


setMethod("plot",
          signature(x="SpliceSites", y="missing"),
          function(x, ...) {
              plot.SpliceSites(x, ...)
            })


setMethod("initialize", "SpliceSites",
          function(.Object, 
                   seq = "", seq.length = as.integer(-1),
                   spsiteIpos = matrix(0, 0, 0),
                   spsiteIIpos = integer(0),
                   spsiteIIIpos = matrix(0, 0, 0),
                   spsiteIpos.pData = new("phenoData"),
                   spsiteIIpos.pData = new("phenoData"),
                   spsiteIIIpos.pData = new("phenoData"),
                   ...)
          {
            if (seq == "" && seq.length == -1)
              stop("'seq' or 'seq.length' must be defined.")
            .Object <- callNextMethod()
            .Object@seq = seq
            .Object@seq.length = seq.length
            .Object@spsiteIpos = spsiteIpos
            .Object@spsiteIIpos = spsiteIIpos
            .Object@spsiteIIIpos = spsiteIIIpos
            .Object@spsiteIpos.pData = spsiteIpos.pData
            .Object@spsiteIIpos.pData = spsiteIIpos.pData
            .Object@spsiteIIIpos.pData = spsiteIIIpos.pData
            return(.Object)
          })


if( !isGeneric("grid.plot") )
  setGeneric("grid.plot", function(x, y, ...)
             standardGeneric("grid.plot"))

setMethod("grid.plot",
          signature(x="SpliceSites", y="missing"),
          function(x, ...) {
              grid.plot.SpliceSites(x, ...)
            })
## This class could be useful in the pack matchprobes too...

setClass("Probes",
         representation(pos="matrix", info="data.frame"))


setMethod("initialize", "Probes",
          function(.Object, pos, info=NULL) {
            .Object@pos <- pos
            if (! is.null(info)) {
              if (nrow(pos) != nrow(info))
                stop("length mismatch between 'pos' and 'info'.")
              .Object@info <- info
            }
            return(.Object)
          })

setMethod("show", "Probes",
          function(object) {
            cat("Probes object:\n")
            cat(" ", nrow(object@pos), "probe(s)\n")
          })

if( !isGeneric("plot") )
  setGeneric("plot", function(x, y, ...)
             standardGeneric("plot"))

setMethod("plot",
          signature(x="Probes", y="missing"),
          function(x, probepos.yscale=NULL, ...){ ##, fig.yratio=c(2,1)) {
            plot.Probes(x, probepos.yscale=NULL, ...) ##, fig.yratio=c(2,1))
          })

setMethod("plot",
          signature(x="Probes", y="SpliceSites"),
          function(x, y, probes.opt=list(), spsites.opt=list(), fig.yratio = c(2,1), probepos.yscale = NULL, ...) {
            
            if (is.null(probepos.yscale)) {
              if (nrow(x@pos) <= 1)
                ypos <- c(0,1)
              else
                ypos <- 1:nrow(x@pos)
            } else {
              ypos <- probepos.yscale
            }
            
            ylim <- range(ypos, 0)
            
            ylim <- range(ylim, ylim[1] - 1/4 * (ylim[2] - ylim[1]))
            xlim <- c(0, y@seq.length)
            
            plot(x=0, y=0,
                 xlab="seq", ylab="splice variants / probes",
                 xlim=xlim, ylim=ylim,
                 type="n", ...)
            
            do.call("plot.SpliceSites", c(list(y, add=TRUE, ylim=ylim), spsites.opt))
            p.ylim <- do.call("plot.Probes",
                              c(list(x, xlim=c(0, y@seq.length), add=TRUE, probepos.yscale = probepos.yscale),
                                probes.opt))
            
            abline(h=0, col="grey")


            invisible(ylim)
            
          }
          )


if( !isGeneric("grid.plot") )
  setGeneric("grid.plot", function(x, y, ...)
             standardGeneric("grid.plot"))

setMethod("grid.plot",
          signature(x="Probes", y="missing"),
          function(x, probepos.yscale=NULL, vp = NULL, ...){ ##, fig.yratio=c(2,1)) {
            grid.plot.Probes(x, probepos.yscale=NULL, vp = vp, ...) ##, fig.yratio=c(2,1))
          })


setMethod("grid.plot",
          signature(x="Probes", y="SpliceSites"),
          function(x, y, probes.opt=list(), spsites.opt=list(), fig.yratio = c(2/3, 1/3),
                   probepos.yscale = NULL, add=FALSE, vp = NULL, ...) {
            
            if (is.null(probepos.yscale)) {
              if (nrow(x@pos) <= 1)
                ypos <- c(0,1)
              else
                ypos <- 1:nrow(x@pos)
            } else {
              ypos <- probepos.yscale
            }
            
            ylim <- range(ypos, 0)
            
            ylim <- range(ylim, ylim[1] - 1/4 * (ylim[2] - ylim[1]))
            xlim <- c(0, y@seq.length)



            if (! add) {
              grid.newpage()
              figscale <- 0.9 
              ##vp <- viewport(xscale = xlim, yscale = ylim, w=0.9, h=0.9)
            } else {
              push.viewport(vp)
              on.exit(pop.viewport())
              figscale <- 1
            }
            
            top.lt <- grid.layout(2, 1, widths = figscale * 1,
                                    heights = figscale * fig.yratio,
                                  default.units = "npc",
                                  respect = matrix(c(1, 1), 2, 1))
            
            temp.vp <- viewport(layout = top.lt)
            push.viewport(temp.vp)

            ##spliceSites
            panel.vp <- viewport(layout.pos.row = 2, layout.pos.col = 1)
            do.call("grid.plot", c(list(y, add=TRUE, ylim=ylim, vp=panel.vp), spsites.opt))
            ##probes
            panel.vp <- viewport(layout.pos.row = 1, layout.pos.col = 1)
            ## trick to have the background:
            spsites.opt.hack <- spsites.opt
            spsites.opt.hack$col.typeI <- 0
            do.call("grid.plot", c(list(y, add=TRUE, ylim=ylim, vp=panel.vp),
                                   spsites.opt.hack))
            p.ylim <- do.call("grid.plot",
                              c(list(x, xlim=c(0, y@seq.length), add=TRUE, vp=panel.vp,
                                     probepos.yscale = probepos.yscale),
                                probes.opt))
            
            ##abline(h=0, col="grey")
            
            invisible(ylim)
            
          }
          )



matchprobes2Probes <- function(mpo, probes.length, names=NULL) {
  if (! identical(names(mpo), c("match", "pos")))
    stop("Expected a list with names 'match' and 'probes'\n(as returned by the package 'matchprobes').")

  n.seq <- length(mpo$pos)
  
  p.list <- vector("list", length=n.seq)

  for (i in seq(along=mpo$pos)) {
    info <- data.frame(probe.index = mpo$match[[i]])
    
    p.list[[i]] <- new("Probes", pos=cbind(mpo$pos[[i]],
                                   mpo$pos[[i]] + probes.length),
                       info=info)
    
  }
  
  return(p.list)
}


#   if( !isGeneric("getPos") )
#     setGeneric("getPos", function(object)
#                standardGeneric("getPos"), where=where)
 
#   setMethod("getPos", signature(object="Probes"),
#             function(object, what=c("begin", "end")){
#               what <- match.arg(what)
#               object@info[what][[1]]
#             }, where = where)

##}
##.initSpliceSitesGenomicMethods <- function(where=where) {

    setClass("SpliceSitesGenomic",
             representation(variants = "list"),
             contains="SpliceSites")
##             where=where)

    setMethod("plot", signature(x="SpliceSitesGenomic", y="missing"),
              function(x, ...) {
                plot.SpliceSitesGenomic(x, ...)
              })
##, where = where)
    
##  }


##a <- new("SpliceSitesGenomic", seq.length=as.integer(10), spsiteIpos=matrix(c(1, 3, 5, 2, 3.5, 8), nc=2),
##         variants=list(c(1,2,3), c(2,3)))
## copyright. Laurent 2003
## under the L-GPL license

## classe to describe splice variants w/ expression values
##
require(Biobase, quietly=TRUE) || stop("Could not load the package 'Biobase'")

setClass("SpliceExprSet",
         representation(spliceSites="SpliceSites",
                        probes = "Probes",
                        eset="exprSet")) ## exprs: one row per probe,
## one column per experiment
## phenoData: covariate info.

## -- accessors --
if (is.null(getGeneric("spliceSites")))
  setGeneric("spliceSites", function(object)
               standardGeneric("spliceSites"))

setMethod("spliceSites", signature(object = "SpliceExprSet"),
          function(object) {
            object@spliceSites
          })


setMethod("exprs", signature(object="SpliceExprSet"),
          function(object) exprs(object@eset))

setReplaceMethod("exprs", "SpliceExprSet",
                 function(object, value) {
                   exprs(object@eset) <- value
                   if (nrow(exprs(object)) != nrow(object@probes@pos))
                     stop("mismatch between the number of probes and the size of the expression matrix.")
                   return(object)
                 })

##setMethod("probepos", signature(object="SpliceExprSet"),
##          function(object) { return(object@probepos) }, where=where)

##setMethod("variant", signature(object="SpliceExprSet"),
##          function(object) { return(object@variant) }, where=where)

## -- other methods --
setMethod("show", signature(object = "SpliceExprSet"),
          function(object) {
            cat("Alternative splicing expression set (SpliceExprSet):\n")
            cat("\t", ncol(exprs(object@eset)), " chip(s)\n", sep="")
            cat("\t", nrow(exprs(object)), " probe(s) on the sequence\n", sep="")
              cat(spliceSites(object))
          })


setMethod("plot", signature(x = "SpliceExprSet", y = "missing"),
          function(x, probes.opt = list(), expr.opt = list(col=NA, lty = 1:6),
                   fig.xratio=c(2,1), fig.yratio=c(2,1),
                   probepos.yscale=NULL, ...) {
            plot.SpliceExprSet(x, probes.opt = probes.opt,
                               expr.opt = expr.opt,
                               fig.xratio=fig.xratio, fig.yratio=fig.yratio,
                               probepos.yscale=probepos.yscale, ...)
          })

if( !isGeneric("grid.plot") )
  setGeneric("grid.plot", function(x, y, ...)
             standardGeneric("grid.plot"))

setMethod("grid.plot",
          signature(x="SpliceExprSet", y="missing"),
          function(x, ...) {
            grid.plot.SpliceExprSet(x, ...)
          })


##setMethod("sort", signature(x = "SpliceExprSet"),
sort.SpliceExprSet <- function(x, fun=function(x) order(x@probes@pos[, 1]), reverse=FALSE)
{
  
  o <- fun(x)
  
  if (reverse) {
    o <- rev(o)
  }
  
  spSites <- x@spliceSites
  eset <- x@eset
  probes <- x@probes
  probes@pos <- probes@pos[o, , drop=FALSE]
  exprs(eset) <- exprs(eset)[o, , drop=FALSE]
  ##if (! is.null(spSites@variant))
  ##  spSites@variant <- spSites@variant[o]

  x@spliceSites <- spSites
  x@eset <- eset
  x@probes <- probes
  
  return(x)
}

##  setMethod("isProbeOnSpliceSite", "SpliceExprSet",
isSpliceSiteOnProbe <- function(spSites, probes) {
  error("Not implemented (yet)")
}
##, where=where)

isProbeOnSpliceSite <- function(probes, spSites) {
  
  ## ensure that pos1 & pos2 are correctly ordered
  r.ppos <- apply(probes@pos, 1, range)
  
  isWithin <- function(x,y) {
    ## x: pos1 and pos2 for a probe
    ## y: matrix of two rows and n columns (pos1 and pos2 for n type-II splice sites)
    ## check if (pos1 of x is greater the pos1 of y and lower than the value pos2 of y
    ## for any column (i.e. for any type-II splice site) of y.
    ## In other words, it returns TRUE if the probe x has at least a partial overlap with
    ## one of the type-II site, FALSE otherwise
    any(x[1] >= y[1, ] & x[1] <= y[2, ]) | any(x[2] <= y[2, ] & x[2] >= y[1, ])
  }

  if(length(spSites@spsiteIpos) > 0)
    isintypeI <- apply(r.ppos, 2, isWithin,
                       apply(spSites@spsiteIpos, 1, range))
  else
    isintypeI <- rep(FALSE, nrow(probes@pos))
  
  hasSite <- function(x, y) {
    any(x[1] >= y & x[2] <= y)
  }

  if(length(spSites@spsiteIIpos) > 0)
    isintypeII <- apply(r.ppos, 2, hasSite,
                        spSites@spsiteIIpos)
  else
    isintypeII <- rep(FALSE, nrow(probes@pos))

  return(isintypeI=isintypeI, isintypeII=isintypeII)
}

## HOWTO:
## - ProbeSet to SpliceExprSet:
##            get the pm from the probe set and put in slot exprs
##
# grid.numeric2npc <- function(x, xlim=NULL, lower.blank=0, upper.blank=0) {
#   if (is.null(xlim)) {
#     offset <- min(x, na.rm=TRUE)
#     scale <- max(x, na.rm=TRUE)
#   } else {
#     offset <- min(xlim)
#     scale <- max(xlim, na.rm=TRUE)
#   }
#   scale <- scale - offset
#   offset <- offset + lower.blank
#   scale <- scale + upper.blank
#   x <- (x - offset) / scale
#   return(x)
# }

grid.make.numeric2npc <- function(x=NULL, xlim=NULL, lower.blank=0, upper.blank=0) {
  if (! is.null(x)) {
    offset <- min(x, na.rm=TRUE)
    scale <- max(x, na.rm=TRUE)
  } else if (! is.null(xlim)) {
    offset <- min(xlim)
    scale <- max(xlim, na.rm=TRUE)
  } else {
    stop("Only 'x' or 'xlim' are expected to be defined !")
  }
  scale <- scale - offset
  offset <- offset + lower.blank
  scale <- scale + upper.blank
  
  f <- function(x) {
    x <- (x - offset) / scale
    return(x)
  }
  
  return(f)
}

# grid.expand.gp <- function(n, parlist=list(), ...) {
#   parlist <- c(parlist, substitute(list(...)))
#   lapply(parlist, function(x) rep(x, length=n))
# }

grid.expand.gp <- function(n, parlist=list()) {
  gp <- vector("list", length=n)
  parlist <- lapply(parlist, function(x) rep(x, length=n))
  for (i in seq(1, n, length=n)) {
    gp[[i]] <- do.call("gpar", lapply(parlist, function(x) x[i]))
  }
  return(gp)
}

grid.plot.Probes <- function(x,
                             col="black",
                             add=FALSE,
                             probepos.yscale=NULL, xlim=NULL, vp = NULL, ...) {

  if (is.null(probepos.yscale)) {
    if (nrow(x@pos) <= 1)
      ypos <- c(0,1)
    else
      ypos <- 1:nrow(x@pos)
  } else {
    ypos <- probepos.yscale
  }

  ylim <- range(ypos)
  
  if (! add) {
    grid.newpage()

    ##grid.rect(gp=gpar(fill="grey"))
    if (is.null(xlim))
      xlim <- range(x@pos)
    
    ## plot separator
    ##abline(h=0, col="grey")
    vp <- viewport(xscale = xlim, yscale = ylim, w=0.9, h=0.9)
    
  } else {
    push.viewport(vp)
    on.exit(pop.viewport())
    vp <- viewport(xscale = xlim, yscale = ylim)
  }
  
  
  grid.xaxis(vp = vp)

  if (nrow(x@pos) > 0) {
    col <- rep(col, length=length(ypos))
    for (i in seq(along=ypos)) {
      grid.segments(x@pos[i, 1], ypos[i],
                    x@pos[i, 2], ypos[i],
                    default.units = "native",
                    gp = gpar(col=col[i]),
                    vp = vp)
##      segments(x@pos[i, 1], ypos[i], x@pos[i, 2], ypos[i], col=col[i])
    }
  }

  invisible(ylim)
  
}
grid.plot.SpliceExprSet <- function(x, probes.opt = list(),
                               expr.opt = list(col=NA, lty = 1:6),
                               fig.xratio=c(2,1), fig.yratio=c(2,1),
                               probepos.yscale=NULL, ylim = NULL, ...) {
  
  spSites <- x@spliceSites
  probes <- x@probes
  eset <- x@eset

  if (! all(is.list(probes.opt) , is.list(expr.opt)))
    stop("probes.opt and expr.opt should be lists !")
    
  if (all(is.na(expr.opt$col))) {
    expr.opt$col <- rainbow(ncol(exprs(eset)))
  }
  
  grid.newpage()
  ##top.vp <- top.vp <- viewport(y = 0, height = unit(1, "npc") - unit(1.5, 
  ##      "lines"), just = c("centre", "bottom"))
  top.lt <- grid.layout(2, 2, widths = 0.9 * c(0.5),
                        heights = 0.9 * 1, ##c(0.5),
                        default.units = "npc",
                        respect = matrix(c(1, 1), 2, 2))
  temp.vp <- viewport(layout = top.lt)
  push.viewport(temp.vp)

  panel.vp <- viewport(layout.pos.row = 2, layout.pos.col = 1)  
  grid.plot(spSites, vp=panel.vp, add=TRUE)
  
  panel.vp <- viewport(layout.pos.row = 1, layout.pos.col = 1)
  ylim <- do.call("grid.plot", c(list(probes, xlim=c(0, spSites@seq.length), vp=panel.vp, add=TRUE),
                                 probes.opt))
  
  
  if (is.null(probepos.yscale)) {
    ypos <- 1:nrow(probes@pos)
  } else {
    ypos <- probepos.yscale
  }

  ##ylim <- range(ypos)
    
  if (is.null(ylim))
    ylim <- m.ylim
  
#     if (nrow(probes@pos) <= 1)
#     ylim <- c(0,1)
#   else
#     ylim <- c(0,nrow(probes@pos))

  
  if (nrow(exprs(eset)) != nrow(probes@pos))
    stop("length mismatch between number of probes and number of expression values")
  
  opar.mar <- par()$mar
  on.exit(par(mar=opar.mar))
  npar.mar <- opar.mar
  npar.mar[2] <- 1
  par(mar=opar.mar)

  ylim <- range(ylim, ylim[1] - 1/4 * (ylim[2] - ylim[1]))
  xlim <- c(0, spSites@seq.length)

  ##scale.x <- grid.make.numeric2npc(xlim)
  ##scale.y <- grid.make.numeric2npc(ylim)

  gp <- grid.expand.gp(nrow(exprs(eset)), parlist=expr.opt)

  panel.vp <- viewport(layout.pos.row = 1, layout.pos.col = 2)
  push.viewport(panel.vp)
  vp <- viewport(xscale = xlim, yscale = ylim)
  for (i in seq(1, nrow(exprs(eset)), length=nrow(exprs(eset))))
    grid.lines(exprs(eset)[i, ], ypos,
               gp = gp[[i]], vp = panel.vp)
  
  return()
  
  do.call("matplot", c(list(exprs(eset), matrix(ypos, ncol=1),
                            ylim=ylim,
                            xlab="expression", ylab="probes",
                            type="l"), expr.opt))
  ##overlay typeI
                                        #for (i in 1:nrow(spSites@spsiteIpos))
                                        #  rect(ylim[1], min(spSites@spsiteIpos[i, ]), ylim[2], max(spSites@spsiteIpos[i, ]),
                                        #       col="yellow", border="transparent" )
  ##overlay typeII
                                        #for (i in seq(along=spSites@spsiteIIpos))
                                        #  ##segments(x@spsiteIIpos[i], ylim[1], x@spsiteIIpos[i], ylim[2], col="red")
                                        #  segments(ylim[1], spSites@spsiteIIpos[i], ylim[2], spSites@spsiteIIpos[i], col="red")
  
  ##boxplot(exprs(eset), horizontal=TRUE, add=TRUE)
  
}
grid.plot.SpliceSites <- function(x, col.typeI="orange", col.typeI.window="yellow", col.typeII="red", add=FALSE, ylim=NULL, vp = NULL, ...) {

  ## type I splice sites
  if (nrow(x@spsiteIpos) > 0) {
    spliceI.pos <- 1:nrow(x@spsiteIpos)
  } else {
    spliceI.pos <- integer(0)
  }
  
  ## plot the upper part:

  xlim <- range(c(1, x@seq.length))
  ylim <- range(0, - nrow(x@spsiteIpos))

  if (! add) {
    grid.newpage()
    vp <- viewport(xscale = xlim, yscale = ylim, w=0.9, h=0.9)
    grid.xaxis(vp = vp, main=FALSE)
  } else {
    push.viewport(vp)
    on.exit(pop.viewport())
    vp <- viewport(xscale = xlim, yscale = ylim)
  }
  
  
  scale.x <- grid.make.numeric2npc(xlim=xlim)
  scale.y <- grid.make.numeric2npc(xlim=ylim)
  ## plot type I splice sites
  for (i in spliceI.pos)
    grid.rect(scale.x(min(x@spsiteIpos[i, ])), ##scale.y(ylim[1]),
              width = scale.x(diff(range(x@spsiteIpos[i, ]))),
              height = 0.99,
              just ="left",
              gp = gpar(fill=col.typeI.window, col="transparent"),
              vp = vp)
  
  ## plot type II splice sites
  for (i in seq(along=x@spsiteIIpos))
    grid.segments(scale.x(x@spsiteIIpos[i]), scale.y(ylim[1]),
                  scale.x(x@spsiteIIpos[i]), scale.y(ylim[2]),
                  gp = gpar(col=col.typeII),
                  vp = vp)
  
#   ## plot type III splice sites
#   if (nrow(x@spsiteIIIpos) > 0) {
#     splice.pos <- 1:nrow(x@spsiteIIIpos)
#   } else {
#     splice.pos <- integer(0)
#   }
  
#   for (i in splice.pos) {
#     grid.segments(scale.x(min(x@spsiteIIIpos[i, ])), scale.y(ylim[1]),
#                   scale.x(min(x@spsiteIIIpos[i, ])), scale.y(ylim[2]),
#                   gp = gpar(col="orange"),
#                   vp = vp)
#     grid.segments(scale.x(max(x@spsiteIIIpos[i, ])), scale.y(ylim[1]),
#                   scale.x(max(x@spsiteIIIpos[i, ])), scale.y(ylim[2]),
#                   gp = gpar(col="orange"),
#                   vp = vp)
#   }
  
  ## plot the lower part
  ##FIXME
  ##siteI.ypos <- seq(min(ylim), min(ypos), length=length(spliceI.pos)+1)
  siteI.ypos <- seq(min(ylim), min(c(ylim[2], 0)),
                    length=length(spliceI.pos)+1)
  
  col.typeI <- rep(col.typeI, length=length(spliceI.pos))
  
  for (i in seq(along=spliceI.pos))
    grid.segments(scale.x(min(x@spsiteIpos[i, ])), scale.y(siteI.ypos[i]),
                  scale.x(max(x@spsiteIpos[i, ])), scale.y(siteI.ypos[i]),
                  gp = gpar(col=col.typeI[i]),
                  vp = vp)

  ## plot separator
  ##if (! add)
  ##  abline(h=0, col="grey")
  
  invisible(ylim)
  
}

plot.Probes <- function(x,
                        col="black",
                        add=FALSE,
                        probepos.yscale=NULL, xlim=NULL, ...) {

  if (is.null(probepos.yscale)) {
    if (nrow(x@pos) <= 1)
      ypos <- c(0,1)
    else
      ypos <- 1:nrow(x@pos)
  } else {
    ypos <- probepos.yscale
  }

  ylim <- numeric(0)
  
  if (! add) {
    if (is.null(xlim)) {
      if (nrow(x@pos) == 0)
        xlim <- c(0,1)
      else
        xlim <- range(x@pos)
    }
    ## plot the upper part:
    
    ylim <- range(ypos, 0)
    
    plot(x=0, y=0,
         xlab="seq", ylab="probes",
         xlim=xlim, ylim=ylim,
         type="n", ...)
    
    ## plot separator
    abline(h=0, col="grey")

  }
  
  if (nrow(x@pos) > 0) {
    col <- rep(col, length=length(ypos))
    for (i in seq(along=ypos)) {
      segments(x@pos[i, 1], ypos[i], x@pos[i, 2], ypos[i], col=col[i])
    }
  }

  if (identical(ylim, numeric(0)))
    ylim <- ypos
  else
    ylim <- range(ypos)
  
  invisible(ylim)
  
}
plot.SpliceExprSet <- function(x, probes.opt = list(),
                               expr.opt = list(col=NA, lty = 1:6),
                               fig.xratio=c(2,1), fig.yratio=c(2,1),
                               probepos.yscale=NULL, ylim = NULL, ...) {
  
  spSites <- x@spliceSites
  probes <- x@probes
  eset <- x@eset

  if (! all(is.list(probes.opt) , is.list(expr.opt)))
    stop("probes.opt and expr.opt should be lists !")
    
  if (all(is.na(expr.opt$col))) {
    expr.opt$col <- rainbow(ncol(exprs(eset)))
  }
  
  layout(matrix(c(1,2), 1, 2), width=fig.xratio, height=fig.yratio) #fig.yratio useless for now
  
  if (is.null(probepos.yscale)) {
    ypos <- 1:nrow(probes@pos)
  } else {
    ypos <- probepos.yscale
  }

  ##ylim <- range(ypos)

  ##plot(probes, spSites, fig.yratio=fig.yratio, probepos.yscale=ypos, ...)
  m.ylim <- plot(probes, spSites, fig.yratio=fig.yratio, probepos.yscale=ypos, ...)
  if (is.null(ylim))
    ylim <- m.ylim
  
#     if (nrow(probes@pos) <= 1)
#     ylim <- c(0,1)
#   else
#     ylim <- c(0,nrow(probes@pos))

  
  if (nrow(exprs(eset)) != nrow(probes@pos))
    stop("length mismatch between number of probes and number of expression values")
  
  opar.mar <- par()$mar
  on.exit(par(mar=opar.mar))
  npar.mar <- opar.mar
  npar.mar[2] <- 1
  par(mar=opar.mar)

  ylim <- range(ylim, ylim[1] - 1/4 * (ylim[2] - ylim[1]))
  xlim <- c(0, spSites@seq.length)

  do.call("matplot", c(list(exprs(eset), matrix(ypos, ncol=1),
                            ylim=ylim,
                            xlab="expression", ylab="probes",
                            type="l"), expr.opt))
  ##overlay typeI
                                        #for (i in 1:nrow(spSites@spsiteIpos))
                                        #  rect(ylim[1], min(spSites@spsiteIpos[i, ]), ylim[2], max(spSites@spsiteIpos[i, ]),
                                        #       col="yellow", border="transparent" )
  ##overlay typeII
                                        #for (i in seq(along=spSites@spsiteIIpos))
                                        #  ##segments(x@spsiteIIpos[i], ylim[1], x@spsiteIIpos[i], ylim[2], col="red")
                                        #  segments(ylim[1], spSites@spsiteIIpos[i], ylim[2], spSites@spsiteIIpos[i], col="red")
  
  ##boxplot(exprs(eset), horizontal=TRUE, add=TRUE)
  
}
plot.SpliceSites <- function(x, col.typeI="orange", col.typeI.window="yellow", col.typeII="red", add=FALSE, ylim=NULL, ...) {

  ## type I splice sites
  if (nrow(x@spsiteIpos) > 0) {
    spliceI.pos <- 1:nrow(x@spsiteIpos)
  } else {
    spliceI.pos <- integer(0)
  }
  
  ## plot the upper part:

  if (! add) {
    xlim <- range(c(1, x@seq.length))
    ylim <- range(0, - nrow(x@spsiteIpos))
    
    plot(x=0, y=0,
         xlab="seq", ylab="splice variants",
         xlim=xlim, ylim=ylim,
         type="n",
         yaxt="n",
         ...)
  
  } else {
    if (is.null(ylim))
      stop("ylim is missing !")
  }
  
  ## plot type I splice sites
  for (i in spliceI.pos)
    rect(min(x@spsiteIpos[i, ]), ylim[1], max(x@spsiteIpos[i, ]), ylim[2],
         col=col.typeI.window, border="transparent" )

  ## plot type II splice sites
  for (i in seq(along=x@spsiteIIpos))
    segments(x@spsiteIIpos[i], ylim[1], x@spsiteIIpos[i], ylim[2], col=col.typeII)

  ## plot type III splice sites
  if (nrow(x@spsiteIIIpos) > 0) {
    splice.pos <- 1:nrow(x@spsiteIIIpos)
  } else {
    splice.pos <- integer(0)
  }
  
  for (i in splice.pos) {
    segments(min(x@spsiteIIIpos[i, ]), ylim[1], min(x@spsiteIIIpos[i, ]), ylim[2], col="orange")
    segments(max(x@spsiteIIIpos[i, ]), ylim[1], max(x@spsiteIIIpos[i, ]), ylim[2], col="orange")
  }
  
  ## plot the lower part
  ##FIXME
  ##siteI.ypos <- seq(min(ylim), min(ypos), length=length(spliceI.pos)+1)
  siteI.ypos <- seq(min(ylim), min(c(ylim[2], 0)),
                    length=length(spliceI.pos)+1)
  
  col.typeI <- rep(col.typeI, length=length(spliceI.pos))
  
  for (i in seq(along=spliceI.pos))
    segments(min(x@spsiteIpos[i, ]), siteI.ypos[i],
             max(x@spsiteIpos[i, ]), siteI.ypos[i], col=col.typeI[i])

  ## plot separator
  if (! add)
    abline(h=0, col="grey")
  
  invisible(ylim)
  
}
plot.SpliceSitesGenomic <- function(x, col.variant=par("col"), col.exon="white",
                                    split=FALSE, main=names(x@variants), ...) {
  makeBackground <- function(xlim, main) {
    plot.new()
    plot.window(xlim, c(-1, 1))
    title(main=main)
    segments(xlim[1], 0, xlim[2], 0, col="grey")
    for (i in seq(1, nrow(pos), length=nrow(pos))) {
      rect(pos[i, 1], -.5, pos[i, 2], +.5, col=col.exon[i])
    }
  }
  pos <- x@spsiteIpos
  col.variant <- rep(col.variant, length=length(x@variants))
  col.exon <- rep(col.exon, length=nrow(pos))
  xlim <- range(0, pos, x@seq.length)
  if (split)
    main <- paste("variant ", rep(main, length=length(x@variants)))
  else
    makeBackground(xlim, main)
  for (i in seq(along=x@variants)) {
    if (split) {
      makeBackground(xlim, main=main[i])
    }
    variant <- x@variants[[i]]
    for (j in seq(along=variant)[-1]) {
      x1 <- mean(c(pos[variant[j-1], 1], pos[variant[j-1], 2]))
      x2 <- mean(c(pos[variant[j], 1], pos[variant[j], 2]))
      up <- (-1)^i
      segments(x1, up*.5, (x1+x2)/2, up*.8, col=col.variant[i])
      segments((x1+x2)/2, up*.8, x2, up*.5, col=col.variant[i])
    }
  }
}
getPALSdbURL <- function(query, disp = c("data", "browser"),
                        field = c("keyword", "ug.id", "gb.id",
                          "human.cytoband", "mouse.cytoband", "cluster_count"),
                        species = c("human", "mouse"),
                        e.value = "1e-1",
                        ident.threshold = c("90% 50b", "95% 50b", "90% 45b")) {

  require(annotate, quietly=TRUE) || stop()
  
  url.base <- "http://palsdb.ym.edu.tw/cgi-bin/palsdb/big_xml.cgi"
  ##url.base <- "http://140.129.151.155/~laurent/cgi-bin/palsdb/big_xml.cgi"

  disp <- match.arg(disp)
  field <- match.arg(field)
  species <- match.arg(species)
  ident.threshold <- match.arg(ident.threshold)
  
  format <- switch(disp,
                   data="xml",
                   browser="html")

  field <- switch(field,
                  keyword="A",
                  ug.id="B",
                  gb.id="C",
                  human.cytoband="E",
                  mouse.cytoband="F")

  query.tag <- "keyword"
  
  ident.threshold <- switch(ident.threshold,
                            "95% 45b"="A",
                            "90% 50b"="B",
                            "95% 50b"="C",
                            "90% 45b"="D"
                            )

  url <- paste(url.base,
               paste(paste("format", format, sep="="),
                     paste(query.tag, query, sep="="),
                     paste("field", field, sep="="),
                     paste("species", species, sep="="),
                     paste("evalue", e.value, sep="="),
                     paste("constraint", ident.threshold, sep="="),
                     "submit=Submit", ## ?!
                     sep="&"),
               sep="?")
}

queryPALSdb <- function(query, disp = c("data", "browser"),
                        field = c("keyword", "ug.id", "gb.id",
                          "human.cytoband", "mouse.cytoband", "cluster_count"),
                        species = c("human", "mouse"),
                        e.value = "1e-1",
                        ident.threshold = c("90% 50b", "95% 50b", "90% 45b")) {

  disp <- match.arg(disp)
  
  url <- getPALSdbURL(query=query, disp = disp,
                      field = field,
                      species = species,
                      e.value = e.value,
                      ident.threshold = ident.threshold)
  
  if (disp == "data") {
    require(XML, quietly=TRUE) || stop("Library XML required !")
    return(.handleXML(url))
    ##return(paste(readLines(url(url)), collapse=""))
  }
  else {
    browseURL(url)
  }
}
split.SpliceSites <- function(x, f=list(typeI=NA, typeII=NA)) {

  get.factor <- function(type.as, f) {
    
    dt <- switch(type.as,
                 typeI = pData(x@spsiteIpos.pData),
                 typeII = pData(x@spsiteIIpos.pData)
                 )

    f <- factor(numeric(0))
    if (is.character(f) && length(f) == 1 && nrow(dt) > 0) {
      if (! f %in% colnames(dt))
        stop(paste(paste("parameter 'f=", f, sep=""),
             "' is not one of ", colnames(dt), collapse=", "))
      else
        f <- dt[f]
    }
    return(f)
  }
  
  get.pdata <- function(type.as, f) {
    if (length(f) == 0) {
      dt <- list()
    } else {
      dt <- switch(type.as,
                   typeI = cbind(x@spsiteIpos, pData(x@spsiteIpos.pData)),
                   typeII = cbind(x@spsiteIIpos, pData(x@spsiteIIpos.pData))
                   )
      
      dt <- split(dt, f)
    }
    
    return(dt)
  }
  
  if (! is.factor(f$typeI)) {
    if (is.na(f$typeI))
      f$typeI <- get.factor("typeI", "site")
    else
      f$typeI <- get.factor("typeI", f$typeI)
  }
  
  if (! is.factor(f$typeII)) {
    if (is.na(f$typeII))
      f$typeII <- get.factor("typeII", "site")
    else
      f$typeII <- get.factor("typeII", f$typeII)
  }
  
  
  r.I <- get.pdata("typeI", f$typeI)
  r.II <- get.pdata("typeII", f$typeII)
  
  r <- vector("list", length=length(r.I) + length(r.II))

  for (i in seq(along=r.I)) {
    r[[i]] <- new("SpliceSites", seq = x@seq, seq.length = x@seq.length,
                  spsiteIpos = as.matrix(r.I[[i]][, 1:2]),
                  spsiteIpos.pData = new("phenoData", pData=r.I[[i]][, -c(1,2)],
                    varLabels=as.list(names(r.I[[i]])[-c(1,2)]))
                  )
  }

  for (i in seq(along=r.II)) {
    r[[i+length(r.I)]] <- new("SpliceSites", seq = x@seq, seq.length = x@seq.length,
                              spsiteIIpos = as.integer(as.vector(r.II[[i]][, 1])),
                              spsiteIIpos.pData = new("phenoData", pData=r.II[[i]][, -c(1)],
                                varLabels=as.list(names(r.II[[i]])[-c(1)]))
                              )
  }
  
  return(r)
}
.First.lib <- function(libname, pkgname, where) {

  message <- TRUE
  
  if (missing(where)) {
    where <- match(paste("package:", pkgname, sep=""), search())
    if(is.na(where)) {
      warning(paste("Not a package name: ", pkgname))
      return()
    }
    where <- pos.to.env(where)
  }
  
  require(methods, quietly=TRUE) || stop("The package 'methods' is required !")
  require(Biobase, quietly=TRUE) || stop("The package 'Biobase' is required !")
  require(grid, quietly=TRUE) || stop("The package 'grid' is required !")

  if (message) {
    cat("splicegear loaded.\n")
    if ("1.3.25" > package.description("Biobase")[c("Version")]) {
      cat("Please source the function 'as.data.frame.exprSet' available at\n",
          "http://www.cbs.dtu.dk/laurent/download/splicegear/")
    }
  }
  
}
