/* Copyright R. Gentleman, 2003, all rights reserved 

 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */ 
#include "Ruuid.h" 

#ifdef WIN32
uuid_t *UUID2uuidt(UUID curUUID) {
    struct uuid tmpUuid;
    uuid_t *newuuid;
    
    tmpUuid.time_low = curUUID.Data1;
    tmpUuid.time_mid = curUUID.Data2;
    tmpUuid.time_hi_and_version = curUUID.Data3;
    /* Copy the first two bytes of Data4 into clock_seq */
    memcpy(&tmpUuid.clock_seq, curUUID.Data4, 2);
    /* And the last 6 bytes into node */
    memcpy(tmpUuid.node, curUUID.Data4+2, 6);
    /* Now have a complte tsouid structure, convert to Ts'o uuid_t */
    newuuid = (uuid_t *)R_alloc(1,sizeof(uuid_t));
    uuid_pack(&tmpUuid, newuuid);

    return(newuuid);
}

SEXP Ruuid_getuuid(void)
{
    uuid_t *newuuid;
    char x[UUIDSTRLEN], z[UUIDTLEN];
    unsigned char *tmpStr;
    SEXP RuClass, curRuuid, uClass, curUuidt;
    
    RuClass = MAKE_CLASS("Ruuid");
    uClass = MAKE_CLASS("uuidt");
    PROTECT(curRuuid = NEW_OBJECT(RuClass));

    newuuid = getuuid();
    if (newuuid == NULL)
	return(R_NilValue);
    
    strncpy(x,uuid2str(newuuid),UUIDSTRLEN);

    SET_SLOT(curRuuid, Rf_install("stringRep"), mkString(x)); 
    
    memcpy(z, *newuuid, sizeof(uuid_t));
    z[UUIDTLEN] = '\0';

    PROTECT(curUuidt = NEW_OBJECT(uClass));
    SET_SLOT(curUuidt, Rf_install("uuid"), mkString(z));

    SET_SLOT(curRuuid, Rf_install("uuid"), curUuidt);
    
    UNPROTECT(2);

    return(curRuuid);
}

uuid_t *getuuid(void) {
    UUID newWinUUID;
    uuid_t *newuuid;

    /* Create the Win32 UUID */
    if (UuidCreate(&newWinUUID) == RPC_S_UUID_NO_ADDRESS) {
	/* Error, can not generate the UUID */
	return(NULL);
    }
    newuuid = (uuid_t *)R_alloc(1,sizeof(uuid_t));
    newuuid = UUID2uuidt(newWinUUID);
    return(newuuid);
}
#else
SEXP Ruuid_getuuid(void)
{
    uuid_t *newuuid;
    char x[UUIDSTRLEN], z[UUIDTLEN];
    SEXP RuClass, curRuuid, uClass, curUuidt, temp;

    
    RuClass = MAKE_CLASS("Ruuid");
    uClass = MAKE_CLASS("uuidt");
    PROTECT(curRuuid = NEW_OBJECT(RuClass));

    newuuid = getuuid();
    strncpy(x,uuid2str(newuuid),UUIDSTRLEN);

    PROTECT(temp = mkString(x));
    SET_SLOT(curRuuid, Rf_install("stringRep"), temp); 
    UNPROTECT(1);

    memcpy(z, *newuuid, sizeof(uuid_t));
    z[UUIDTLEN-1] = '\0';

    PROTECT(curUuidt = NEW_OBJECT(uClass));
    PROTECT(temp = mkString(z));
    SET_SLOT(curUuidt, Rf_install("uuid"), temp);
    UNPROTECT(1);

    SET_SLOT(curRuuid, Rf_install("uuid"), curUuidt);
    
    UNPROTECT(2);
    return(curRuuid);
}

uuid_t *getuuid(void) {
    uuid_t *newuuid;

    newuuid = (uuid_t *)R_alloc(1,sizeof(uuid_t));

    uuid_generate(*newuuid);

    return(newuuid);
}
#endif /* WIN32 */

char *uuid2str(uuid_t *x) {
    char *y;
    
    y = (char *)R_alloc(UUIDSTRLEN,sizeof(char));
    
    /* Perform the conversion */
    uuid_unparse(*x, y);

    /* Need to manuallyadd in the NULL */
    y[UUIDSTRLEN] = '\0';

    return(y);
}

uuid_t *str2uuid(char *x) {
    uuid_t *y;
    int err;
    
    /* Check that the string is UUIDSTRLEN-1 (subtraction for the NULL
       character) */
    if (strlen(x) != (UUIDSTRLEN-1))
	error("Invalid string parameter");

    y = (uuid_t *)R_alloc(1,sizeof(uuid_t));

    /* Convert */
    err = uuid_parse(x, *y);
    if(err) 
	error("Error transforming: %d \n", err);

    return(y);
}

SEXP Ruuid_uuid2str(SEXP x)
{
  uuid_t myuuid;

  memcpy(myuuid, STR(GET_SLOT(x,Rf_install("uuid"))), sizeof(uuid_t));

  return(mkString(uuid2str(&myuuid)));
}


SEXP Ruuid_str2uuid(SEXP x)
{
    SEXP uClass, curUuid;
    char outStr[UUIDTLEN];
    uuid_t *y;

    if(!isString(x))
	error("require a string argument");

    y = str2uuid(STR(x));

    uClass = MAKE_CLASS("uuidt");
    PROTECT(curUuid = NEW_OBJECT(uClass));

    memcpy(outStr, *y, sizeof(uuid_t));
    outStr[UUIDTLEN-1] = '\0';

    SET_SLOT(curUuid, Rf_install("uuid"), mkString(outStr));
    UNPROTECT(1);

  return(curUuid);
}

SEXP Ruuid_printuuid(SEXP uuid)
{
    char outStr[100];
    uuid_t uu;
    int i;

    strcpy(outStr,"\0");

    memcpy(uu, STR(GET_SLOT(uuid,Rf_install("uuid"))), sizeof(uuid_t));

    for(i=0; i<16; i++) 
	sprintf(outStr,"%s:%d:",outStr,uu[i]);
    
    return(mkString(outStr));
}


