.packageName <- "marrayPlots"
############################################################################
# maBasicPlots.R
#
# Wrapper for diagnostic plots for two-color cDNA microarrays
#
###########################################################################
# Pre- and post-normalization plots

maDiagnPlots1<-function(mraw, title=NULL, save=TRUE,
	fname=paste(as.character(maLabels(maTargets(mraw))[1]),".ps",sep=""),
	dev=c("postscript","jpeg"))
{

   mraw<-mraw[,1]
   # Default loess normalization within print-tip-group
   mnorm<-maNorm(mraw,norm="p")
  
  if(save==TRUE)
    do.call(dev,list(fname))
  
  layout(matrix(c(1:4,9,10,5:8,11,12),2,6,byrow=TRUE),width=c(5.5,3,5.5,3,6,6))

  # maImage
  stats<-c("maGb", "maRb", "maM","maM")
  titl<-c("Gbg","Rbg","Unnormalized M","Normalized M")
  Gcoltmp <- maPalette(low="white", high="green", k=50)
  Rcoltmp <- maPalette(low="white", high="red", k=50)
  RGcoltmp <- maPalette(low="green", high="red", k=50)
  cols<-list(Gcoltmp,Rcoltmp,RGcoltmp,RGcoltmp)
  m<-c("mraw","mraw","mraw","mnorm")
  for(i in 1:4)
  {
    x.bar<-do.call("maImage",list(m=eval(as.symbol(m[i])), x=stats[i], subset=TRUE, col=cols[[i]], contours=FALSE, bar=FALSE,main=titl[i]))$x.bar
     maColorBar(x.bar,horizontal=FALSE,col=cols[[i]],main="")
  }

  # maBoxplot
  maBoxplot(mraw,"maPrintTip","maM",main="Unnormalized M")
  maBoxplot(mnorm,"maPrintTip","maM",main="Normalized M")

  # maPlot
  defs<-maDefaultPar(mraw[,1],"maA","maM","maPrintTip")
  legend.func<-do.call("maLegendLines",defs$def.legend)
  args.lines<-c(list(TRUE,f=0.3),defs$def.lines)
  lines.func<-do.call("maLowessLines",args.lines)
  
  maPlot(mraw,"maA","maM","maPrintTip",lines.func,text.func=maText(),legend.func,main="Unnormalized MA-plot")
  maPlot(mnorm,"maA","maM","maPrintTip",lines.func,text.func=maText(),legend.func,main="Normalized MA-plot")

  # Back to defaults
  layout(1)

  if(!is.null(title))
    mtext(title, line=3)
  else
    mtext(paste(as.character(maLabels(maTargets(mraw))[1]),": Pre- and post- print-tip-group loess normalization",sep=""),line=3)

  if(save==TRUE)
    dev.off()
}


########################################################################### 
# Pre-normalization plots

maRawPlots<-function(mraw, title=NULL, save=TRUE, 
	fname=paste(as.character(maLabels(maTargets(mraw))[1]),".ps",sep=""), 
	dev=c("postscript","jpeg"))
{
 
  mraw<-mraw[,1]
  if(save==TRUE)
    do.call(dev,list(fname))

  layout(matrix(c(1:6,7,7,8,8,9,9),2,6,byrow=TRUE),width=c(6,2.5,6,2.5,6,2.5))

  # maImage
  stats<-c("maGb", "maRb", "maM")
  titl<-c("Gbg","Rbg","Unnormalized M")
  m<-c("mraw","mraw","mraw")
  Gcoltmp <- maPalette(low="white", high="green", k=50)
  Rcoltmp <- maPalette(low="white", high="red", k=50)
  RGcoltmp <- maPalette(low="green", high="red", k=50)
  cols<-list(Gcoltmp,Rcoltmp,RGcoltmp)
  for(i in 1:3)
  {
    x.bar<-do.call("maImage",list(m=eval(as.symbol(m[i])), x=stats[i], subset=TRUE, col=cols[[i]], contours=FALSE, bar=FALSE,main=titl[i]))$x.bar
    maColorBar(x.bar,horizontal=FALSE,col=cols[[i]],main="")
  }

  # maBoxplot
  maBoxplot(mraw,x="maPrintTip", y="maM",main="Unnormalized M")
  maBoxplot(mraw,x="maPlate", y="maM",main="Unnormalized M")
  
  # maPlot
  defs<-maDefaultPar(mraw,"maA","maM","maPrintTip")
  legend.func<-do.call("maLegendLines",defs$def.legend)
  args.lines<-c(list(TRUE,f=0.3),defs$def.lines)
  lines.func<-do.call("maLowessLines",args.lines)
  
  maPlot(mraw,"maA","maM","maPrintTip",lines.func,text.func=maText(),legend.func,main="Unnormalized MA-plot")
  
  # Back to defaults
  layout(1)

  if(!is.null(title))
    mtext(title, line=3)
  else
    mtext(paste(as.character(maLabels(maTargets(mraw))[1]),": Pre-normalization",sep=""),line=3)

  
  if(save==TRUE)
    dev.off()
}


########################################################################## 
# Post-normalization plots

maNormPlots<-function(mnorm, title=NULL, save=TRUE, 
	fname=paste(as.character(maLabels(maTargets(mnorm))[1]),".ps",sep=""),
	dev=c("postscript","jpeg"))
{
  
  mnorm<-mnorm[,1]
  if(save==TRUE)
    do.call(dev,list(fname))

  layout(matrix(c(1:4,5,5,6,6),2,4,byrow=TRUE),width=c(7.5,2.5,7.5,2.5))

  # maImage
  stats<-c("maMloc", "maM")
  titl<-c("Loc. normalization", "Normalized M")
  RGcoltmp <- maPalette(low="green", high="red", k=50)
  cols<-list(RGcoltmp,RGcoltmp)
  for(i in 1:2)
  {
    x.bar<-do.call("maImage",list(m=mnorm, x=stats[i], subset=TRUE, col=cols[[i]], contours=FALSE, bar=FALSE,main=titl[i]))$x.bar
    maColorBar(x.bar,horizontal=FALSE,col=cols[[i]],main="")
  }

  # maBoxplot
  maBoxplot(mnorm,x="maPrintTip",y="maM",main="Normalized M")
  
  # maPlot
  defs<-maDefaultPar(mnorm,"maA","maM","maPrintTip")
  legend.func<-do.call("maLegendLines",defs$def.legend)
  args.lines<-c(list(TRUE,f=0.3),defs$def.lines)
  lines.func<-do.call("maLowessLines",args.lines)
  
  maPlot(mnorm,"maA","maM","maPrintTip",lines.func,text.func=maText(),legend.func,main="Normalized MA-plot")
  

  # Back to defaults
  layout(1)

  if(!is.null(title))
    mtext(title, line=3)
  else
    mtext(paste(as.character(maLabels(maTargets(mnorm))[1]), ": ", as.character(list(maNormCall(mnorm)))," normalization",sep=""),line=3)

  if(save==TRUE)
    dev.off()
}

########################################################################### 
############################################################################
# maPlot.R
#
# Diagnostic plots for two-color cDNA microarrays
#
###########################################################################
require(marrayInput)
require(marrayClasses)
require(modreg)

###########################################################################
# Default plotting parameters for microarray objects
###########################################################################
# Compare default and ... plotting parameters and let ... overwrite defaults

maDotsDefaults<-function(dots, defaults)
{
  args<-c(dots,defaults[setdiff(names(defaults),names(dots))])
  return(args)
}

#########################
# Default parameters for microarray objects of class marrayRaw and marrayNorm
maDefaultPar<-function(m,x,y,z)
{
    m<-m[,1]
    main<-as.character(maLabels(maTargets(m)))

    xlab<-ylab<-zlab<-""
    col<-2
    lty<-1
    lwd<-2.5
    las<-1
    names<-""
    def.legend<-list()

    ylab<-strsplit(y,"ma")[[1]]
    if(ylab[1]=="")
     ylab<-ylab[2]

    if(!is.null(x))
    {
      xlab<-strsplit(x,"ma")[[1]]
      if(xlab[1]=="")
        xlab<-xlab[2]
    }

    if(!is.null(z))
    {
      zz<-eval(call(z,m))
      zlab<-strsplit(z,"ma")[[1]]
      if(zlab[1]=="")
        zlab<-zlab[2]

      if(z!="maPrintTip")
      {
        names<-paste(zlab,unique(zz),sep=" ")
        col<-(1:length(names))
        lty<-rep(1,length(names))
        las<-1
        ncol<-1
        ord<-order(lty,col)
        def.legend<-list(legend=names[ord],col=col[ord], lty=lty[ord],
	  lwd=lwd, ncol=ncol)
      }

      if(z=="maPrintTip")
      {
        which<-unique(zz)
        Ig<-maNgr(m)
        Jg<-maNgc(m)
        lg.names<- paste("(",sort(rep(1:Ig,Jg)),",",rep(1:Jg,Ig),")",sep="")
        lg.col<-sort(rep(2:(Ig+1),Jg))
        lg.lty<-rep(1:Jg,Ig)
        names<-lg.names[which]
        col<-lg.col[which]
        lty<-lg.lty[which]
        las<-3
        ncol<-Jg
        ord<-order(lg.lty,lg.col)
        def.legend<-list(legend=lg.names[ord],col=lg.col[ord], lty=lg.lty[ord], 	  lwd=lwd, ncol=ncol)
      }
   }

    def.box<-list(xlab=xlab,ylab=ylab,names=names,col=col,las=las,main=main)
    def.plot<-list(xlab=xlab,ylab=ylab,pch=20,col=1,main=main)
    def.lines<-list(col=col,lty=lty,lwd=lwd)
    def.text<-list(pch=16,col="purple")

    return(list(def.box=def.box,def.plot=def.plot,def.lines=def.lines,
		def.legend=def.legend,def.text=def.text))
}

###########################################################################
# maBoxplot: Boxplot methods
###########################################################################
# Boxplots for a single and multiple arrays

#.initMarrayBoxplot <- function(where)
#{

#setGeneric("maBoxplot", function(m, x="maPrintTip", y="maM", ...) {

maBoxplot<- function(m, x="maPrintTip", y="maM", ...) {
    opt<-list(...)

    if(maNsamples(m)==1)
    {

      yy<-as.numeric(eval(call(y,m)))

      if(is.null(x))
        xx<-rep(1,length(yy))
      if(!is.null(x))
        xx<-eval(call(x,m))

      # Optional graphical parameter defaults
      def<-maDefaultPar(m,x,y,x)$def.box
      if(!is.null(opt))
        def<-maDotsDefaults(opt,def)


      args<-c(list(yy~xx),def)
      do.call("boxplot",args)
   }
   if(maNsamples(m)>1)
   {
     yy<-as.data.frame(eval(call(y,m)))

     # Optional graphical parameter defaults
     if(length(maLabels(maTargets(m))) != 0)
       def <- list(names=maLabels(maTargets(m)),ylab=strsplit(y,"ma")[[1]][2],col=2)
     else
       def <- list(names=dimnames(yy)[[2]], ylab=strsplit(y,"ma")[[1]][2],col=2)

     if(!is.null(opt))
        def<-maDotsDefaults(opt,def)

      args<-c(list(yy),def)
      do.call("boxplot",args)
    }
    if(y=="maM") abline(h=0,col="gray",lwd=2.5)
}
#  },where=where)


#setMethod("maBoxplot",signature(m="marrayRaw",x="character",y="character"),
#  function(m,x,y, ...) maBoxplot(m,x,y,...), where=where)

#setMethod("maBoxplot",signature(m="marrayNorm",x="character",y="character"),
#  function(m,x,y, ...) maBoxplot(m,x,y,...), where=where)
#}

###########################################################################
# maPlot: Scatter-plot methods with fitted lines and points highlighted
###########################################################################
# General function for scatter-plot of y vs. x with fitted lines within
# values of z and subset of points highlighted

maPlot.func<-function(x, y, z,
	lines.func=maLowessLines(subset=TRUE,f=0.3,col=1:length(unique(z)),
		lty=1,lwd=2.5),
	text.func=maText(),
	legend.func=maLegendLines(legend=as.character(unique(z)),
		col=1:length(unique(z)), lty=1,lwd=2.5,ncol=1),
	...)
{
  plot(x,y,...)

  # Plot fitted curves
  if(!is.null(lines.func))
    lines.func(x,y,z)

  # Legend
  if(!is.null(legend.func))
    legend.func(x,y)

  # Label a subset of points
  if(!is.null(text.func))
    text.func(x,y)

}

#########################
# Label a subset of points
# Jean: Modify (Oct 21, 2002)  line "tmp <- length(c(1:length(subset))[subset])"

maText <-function (subset = NULL, labels = as.character(1:length(subset)),
    ...)
{
    function(x, y) {
      tmp <- length(c(1:length(subset))[subset])
      if (tmp > 0) {
            if (length(subset) < length(labels))
                text(x[subset], y[subset], labels[subset], ...)
            if (length(subset) > length(labels))
                text(x[subset], y[subset], labels, ...)
            if ((length(subset) == length(labels)) & is.logical(subset))
                text(x[subset], y[subset], labels[subset], ...)
            if ((length(subset) == length(labels)) & is.numeric(subset))
                text(x[subset], y[subset], labels, ...)
        }
    }
}

#########################
# Plot fitted lines

# Lowess
maLowessLines<-function(subset=TRUE, f=0.3, col=2, lty=1, lwd=2.5,...)
{
  function(x,y,z)
  {
    subset<-maNum2Logic(length(x), subset)
    g<-unique(z[subset])
    if(length(col)<length(g))
      col<-rep(col[1],length(g))
    if(length(lty)<length(g))
      lty<-rep(lty[1],length(g))

    for(i in (1:length(g)))
    {
      which<-z[subset]==g[i]
      xx<-x[subset][which]
      yy<-y[subset][which]
      ind <- is.na(xx) | is.na(yy) | is.infinite(xx) | is.infinite(yy)
      fit<- lowess(xx[!ind], yy[!ind], f=f)
      lines(fit,col=col[i],lty=lty[i],lwd=lwd,...)
    }
  }
}

# Loess

maLoessLines<-function(subset=TRUE, weights=NULL,
                        loess.args=list(span=0.4, degree=1, family="symmetric",
                          control=loess.control(trace.hat="approximate",
                            iterations=5,surface="direct")),col=2, lty=1, lwd=2.5, ...)
{
  function(x,y,z)
  {
    subset<-maNum2Logic(length(x), subset)
    g<-unique(z[subset])
    if(length(col)<length(g))
      col<-rep(col[1],length(g))
    if(length(lty)<length(g))
      lty<-rep(lty[1],length(g))

    for(i in (1:length(g)))
    {
      which<-z[subset]==g[i]
      xx<-x[subset][which]
      yy<-y[subset][which]
      ww<-weights[subset][which]
      args<-c(list(yy ~ xx, weights=ww),loess.args)
      fit<-do.call("loess",args)
      xf<-seq(quantile(xx,0.005,na.rm=TRUE),quantile(xx,0.995,na.rm=TRUE),length=100)
      yf<-predict(fit,data.frame(xx=xf))
      lines(xf,yf,col=col[i],lty=lty[i],lwd=lwd,...)
    }
  }
}

#########################
# Add legend to existing plot

maLegendLines<-function(legend="", col=2, lty=1, lwd=2.5, ncol=1, ...)
{
  function(x,y)
  {
    a<-min(x[!(is.na(x)|is.infinite(x))])
    b<-max(y[!(is.na(y)|is.infinite(y))])
    legend(a,b,legend=as.character(legend),col=col,lty=lty,lwd=lwd,ncol=ncol,...)
  }
}

###########################################################################
# Methods for microarray objects: wrapper around maPlot.func

#.initMarrayPlot <- function(where)
#{

#setGeneric("maPlot", function(m, x="maA", y="maM", z="maPrintTip",lines.func,text.func,legend.func,...)

## Jean April 9,2003 modified default to maLoessLines
##

maPlot <- function(m, x="maA", y="maM", z="maPrintTip",lines.func,text.func,legend.func, ...)
{

  m<-m[,1]
  # Default plotting arguments
  defs<-maDefaultPar(m,x,y,z)

  if(missing(lines.func))
    lines.func<-do.call("maLoessLines", c(list(subset=TRUE,
                                               loess.args=list(span=0.4, degree=1, family="symmetric",
                                                 control=loess.control(trace.hat="approximate",
                                                   iterations=5,surface="direct"))),
                                               defs$def.lines))
  if(missing(text.func))
    text.func<-maText()
  if(missing(legend.func))
    legend.func<-do.call("maLegendLines",defs$def.legend)

  xx<-as.numeric(eval(call(x,m)))
  yy<-as.numeric(eval(call(y,m)))
  if(is.null(z))
     zz<-rep(1,length(xx))
  if(!is.null(z))
    zz<-eval(call(z,m))

  opt<-list(...)
  if(!is.null(opt))
    def.plot<-maDotsDefaults(opt,defs$def.plot)

  do.call("maPlot.func", c(list(x=xx,y=yy,z=zz,lines.func=lines.func,text.func=text.func,legend.func=legend.func),def.plot))
  if(y=="maM") abline(h=0,col="gray",lwd=2.5)
}

#},where=where)

#setMethod("maPlot",
#signature(m="marrayRaw",x="character",y="character",z="character",
#          lines.func="function",text.func="function",legend.func="function"),
#function(m,x,y,z,lines.func,text.func,legend.func,...)
#  maPlot(m,x,y,z,lines.func,text.func,legend.func,...), where=where)

#setMethod("maPlot",
#signature(m="marrayNorm",x="character",y="character",z="character",
#           lines.func="function",text.func="function",legend.func="function"),
#function(m,x,y,z,lines.func,text.func,legend.func,...)
#  maPlot(m,x,y,z,lines.func,text.func,legend.func,...), where=where)

#}

###########################################################################
# maImage: image methods for microarray objects
###########################################################################
# Function for 2D spatial image of spot statistics x

maImage.func<-function(x, L, subset=TRUE, col=heat.colors(12), contours=FALSE, ...)
{

  # When only a subset of spots are stored in marray object, pad with NA
  subset<-maNum2Logic(maNspots(L), subset)
  z<-rep(NA,maNspots(L))
  z[maSub(L)][subset]<-x[subset]

  Ig<-maNgr(L)
  Jg<-maNgc(L)
  Is<-maNsr(L)
  Js<-maNsc(L)
  # Create a "full layout"
  L0<-read.marrayLayout(ngr=Ig, ngc=Jg, nsr=Is, nsc=Js)
  nr<-Is*Ig
  nc<-Js*Jg
  row.ind<-(maGridRow(L0)-1)*Is+maSpotRow(L0)
  col.ind<-(maGridCol(L0)-1)*Js+maSpotCol(L0)
  ord<-order(row.ind,col.ind)


  z<-matrix(z[ord],nrow=nr,ncol=nc,byrow=TRUE)
  z<-t(z)[,nr:1]

  # Image of spot statistics
  image(1:nc, 1:nr, z, axes=FALSE, col=col, ...)
  axis(3, at = (0:(Jg-1))*Js + Js/2, labels = 1:Jg)
  axis(2, at = (0:(Ig-1))*Is + Is/2, labels = Ig:1,las=1)
  if(contours)
    contour(1:nc,1:nr,z,add=TRUE, ...)
  box(lwd=4)
  abline(v=((1:Jg-1)*Js + 0.5),lwd=3)
  abline(h=((1:Ig-1)*Is + 0.5),lwd=3)

}

#########################
# Methods for microarray objects: wrapper around maImage.func

#.initMarrayImage <- function(where)
#{

# setGeneric("maImage", function(m, x="maM", subset=TRUE, col, contours=FALSE, bar=TRUE, ...)


# Methods for microarray objects: wrapper around maImage.func
# Modified by Jean : Sept 15, 2002 to include centering of color

maImage <- function(m, x="maM", subset=TRUE, col, contours=FALSE, bar=TRUE, ...)
{
  subset<-maNum2Logic(maNspots(m), subset)
  m<-m[,1]
  if(missing(col))
  {
    col<-rainbow(50)
    if(is.element(x,c("maGb","maGf","maLG")))
      col<-maPalette(low="white", high="green", k=50)
    if(is.element(x,c("maRb","maRf","maLR")))
      col<-maPalette(low="white", high="red", k=50)
    if(is.element(x,c("maM","maMloc","maMscale")))
      col<-maPalette(low="blue", high="yellow", k=50)
  }

  xx<-as.numeric(eval(call(x,m)))

  # Set color range
  tmp<-xx[subset]
  tmp<-tmp[!(is.na(tmp)|is.infinite(tmp))]

  zmax <- ceiling(max(tmp))
  zmin <- floor(min(tmp))
  if(zmin < 0){
    ztmp <- max(abs(zmin), zmax)
    zrange <- c(-ztmp, ztmp)
  }
  else
    zrange <- c(zmin, zmax)

  # Optional graphical parameter defaults

  def<-list(xlab="",ylab="",main=paste(maLabels(maTargets(m)), ": image of ", strsplit(x,"ma")[[1]][2], sep=""), zlim=zrange)
  opt<-list(...)
  if(!is.null(opt))
    def<-maDotsDefaults(opt,def)
  args<-c(list(x=xx, L=maLayout(m), subset=subset, col=col, contours=contours),def)
  x.bar <- seq(args$zlim[1], args$zlim[2], length=41)

  if(!bar)
    do.call("maImage.func",args)

  if(bar)
  {
    layout(matrix(c(1,2),1,2),width=c(9,1))
    par(mar=c(4,4,5,3))
    do.call("maImage.func",args)
    par(mar=c(4,0,5,3))
    maColorBar(x.bar,horizontal=FALSE,col=col,main="")
    layout(1)
    par(mar=c(5,4,4,2) + 0.1)
  }

  return(list(x.col=col[1:length(x.bar)], x.bar=x.bar,
              summary=summary(xx[subset])))
}
#}, where=where)

#setMethod("maImage",signature(m="marrayRaw", x="character", subset="logical",
#	col="character", contours="logical", bar="logical"),
#	function(m, x, subset, col, contours, bar, ...)
#		maImage(m, x, subset, col, contours, bar, ...)
#, where=where)

#setMethod("maImage",signature(m="marrayNorm", x="character", subset="logical",
#	col="character", contours="logical", bar="logical"),
#	function(m, x, subset, col, contours, bar, ...)
#		maImage(m, x, subset, col, contours, bar, ...)
#, where=where)

#}
###########################################################################
# Color bar for calibration

maPalette <- function(low = "white",
                      high = c("green", "red"),
                      mid=NULL,
                      k =50)
{
    low <- col2rgb(low)/255
    high <- col2rgb(high)/255

    if(is.null(mid)){
        r <- seq(low[1], high[1], len = k)
        g <- seq(low[2], high[2], len = k)
        b <- seq(low[3], high[3], len = k)
      }
    if(!is.null(mid)){
      k2 <- round(k/2)
      mid <- col2rgb(mid)/255
      r <- c(seq(low[1], mid[1], len = k2),
             seq(mid[1], high[1], len = k2))
      g <- c(seq(low[2], mid[2], len = k2),
             seq(mid[2], high[2], len = k2))
      b <- c(seq(low[3], mid[3], len = k2),
             seq(mid[3], high[3], len = k2))
    }
    rgb(r, g, b)
}


maColorBar<-function(x, horizontal = TRUE, col=heat.colors(50),
scale=1:length(x), k=10,  ...)
{
  if(is.numeric(x))
  {
    x <- x
    colmap <- col
  }
  else
  {
    colmap <- x
    low<-range(scale)[1]
    high<-range(scale)[2]
    x <- seq(low, high, length=length(x))
  }

  if(length(x)>k)
    x.small<-seq(x[1], x[length(x)],length=k)
  else
    x.small<-x

  if(horizontal)
  {
    image(x, 1, matrix(x,length(x),1), axes=FALSE, xlab="", ylab="", col=colmap, ...)
    axis(1, at=rev(x.small), labels=signif(rev(x.small),2), srt=270)
  }
  if(!horizontal)
  {
    image(1, x, matrix(x,1,length(x)), axes=FALSE, xlab="", ylab="", col=colmap, ...)
    par(las=1)
    axis(4, at=rev(x.small), labels=signif(rev(x.small), 2))
    par(las=0) # Back to default
  }
  box()
}

###########################################################################
# Functions to filter genes, return a logical vector

maTop<-function(x, h=1, l=1)
{
  x<-as.vector(x)
  return((x>=quantile(x, 1-h, na.rm=TRUE)) | (x<=quantile(x, l, na.rm=TRUE)))
}

###########################################################################
## R function for finding gene patterns
## Created Date: Sept. 18, 2002

maDiagnPlots <- function(mraw,
                         mNorm = NULL,
                         save = TRUE,
                         fname = NULL,
                         dev = c("jpeg", "postscript"),
                         pch,
                         col,
                         DEBUG=FALSE,
                         ...) 
{
  rm.na <- 
  function (x) 
    {
      ind <- is.na(x) | is.nan(x) | is.infinite(x)
      return(x[!ind])
    }
  
  opt <- list(...)
  data <- mraw[,1]
  if(is.null(mNorm))
    {
      ifelse(length(maW(data)) != 0, 
             defs <- list(norm="p", subset=eval(maW(data)==0)),
             defs <- list(norm="p"))
      args <-  maDotsDefaults(opt, defs)
      normdata <- do.call("maNorm", c(list(data),args))
    }
  
  if(is.null(fname))
    f <- colnames(maGf(data)) 

  tmp <- unlist(strsplit(f, "\\."))
  ifelse(length(maGb(data))!=0, bg <- "Plot", bg <- "Plot.nbg")
  fstart <- paste(tmp[-length(tmp)], collapse=".")

  if(length(grep(f, dir())) != 0)
    {
      tmp <- readLines(f, n=40)
      subnames <- paste(tmp[grep("DateTime", tmp)], tmp[grep("PMT", tmp)])
    }
  else
    subnames <- ""
  
  if(dev == "jpeg")
    {
      fname <- paste(bg, fstart, "jpeg", sep=".")
      def <- list(dev=list(quality=100, width=1400, height=1400),
                  main= paste(fname, ": Pre- and post- normalization"))
    }

  if(dev == "postscript")
    {
      fname <- paste(bg, fstart, "ps", sep=".")
      def <- list(dev=list(paper="special", width=14, height=14),
                  main= paste(fname, ": Pre- and post- normalization"))
    }
 
  if (!is.null(opt)) 
    def <- maDotsDefaults(opt, def)
  args <- c(list(fname), def$dev)
  
  if(save) 
    do.call(dev, args)
  
  ## Some controls info
  ifelse(missing(col), colcode <- unique(as.integer(maControls(data))+1),
         colcode <- col)
  ctlcode <- levels(maControls(data))
  names(colcode) <- ctlcode


  if(DEBUG) print("start 2")
  ## Layout 
  layout(matrix(c(13, 1,1,2,2,13,3,3,5,5,13,4,4,6,6,13,7,7,
                  9,10,13,8,8,9,10, 13,11,11,12,12), 5, 6),
         height=c(1, 5, 5, 5, 5), 
         width = c(5.5, 5.5, 2 ,5.5, 2, 5.5))
  
  ## 1) MA-plot (Before Normalization)
  par(mar=c(3,2,3,2))
  y <- max(maM(data), na.rm=TRUE) + 2
  x <- min(maA(data), na.rm=TRUE) 
  defs <- maDefaultPar(data, x="maA", y="maM", z="maPrintTip")
  flags <- as.integer(as.factor(maW(data)))
  badspot.func <- maText(maW(data) != 0,
                         labels=as.character(flags[maW(data) != 0]),
                         col=flags[maW(data) != 0], cex=0.5)
  args <- maDotsMatch(maDotsDefaults(opt, defs), formals(args("maPlot")))
  maPlot(data, ylim=c(min(maM(data), na.rm=TRUE),y), text.func = badspot.func,
         legend.func=NULL, main="MA-plot: raw", cex=0.6)
  legend.func <- do.call("maLegendLines", defs$def.legend)
  legend.func(x, y)

  if(DEBUG) print("start 2")
  ## 2) MA-plot (After Normalization)
  par(mar=c(3,2,3,2))
  y <- max(maM(normdata), na.rm=TRUE) + 2
  x <- min(maA(normdata), na.rm=TRUE) 
  defs <- maDefaultPar(normdata, x="maA", y="maM", z="maPrintTip")
  Cindex <- as.character(maControls(data)) != "probes"
  if(length(c(1:length(Cindex))[Cindex]) != 0)
    {
      speccode <- rep(NA, length(Cindex))
      for(i in 1:length(ctlcode))
        speccode[maControls(data)==ctlcode[i]] <- colcode[i]
      qualspot.func <- maText(Cindex,
                              labels=rep("*", length(Cindex))[Cindex],
                              col=speccode[Cindex], cex=1)
    }
  else
    {
      qualspot.func <- NULL
    }
  maPlot(normdata, ylim=c(min(maM(normdata), na.rm=TRUE),y),text.func = qualspot.func,
         lines.func=NULL, legend.func =NULL, main="MA-plot: Norm", cex=0.6)
  if(length(c(1:length(Cindex))[Cindex]) != 0)
    legend(x, y,  ctlcode[ctlcode!="probes"], col=colcode, pch="*")

  if(DEBUG) print("start 3, 4")
  ## 3 & 4) maM (Before Normalization)
  par(mar=c(2,3,5,2))
  RGcol <- maPalette(high="red", low="green", mid="white", k=50)
  tmp <- maImage(data, x="maM", main="Spatial: M-Raw", bar=FALSE, col=RGcol)
  par(mar=c(2,1,2,4))
  maColorBar(tmp$x.bar, horizontal = FALSE, col = RGcol,  main = "")

  if(DEBUG) print("start 5 & 6")
  ## 5 & 6) maM (After Normalization)
  par(mar=c(2,3,5,2))
  tmp <- maImage(normdata, x="maM", main="Spatial: M-Norm", bar=FALSE, col=RGcol)
  par(mar=c(2,1,2,4))
  maColorBar(tmp$x.bar, horizontal = FALSE, col = RGcol,  main = "")

  if(DEBUG) print("start 7 & 8")
  ## 7 & 8) maA 
  par(mar=c(2,3,5,2))
  Bcol <- maPalette(high="blue", low="white", k=50)
  tmp <- maImage(data, x="maA", main="Spatial: A", bar=FALSE, col=Bcol)
  par(mar=c(2,1,2,4))
  maColorBar(tmp$x.bar, horizontal = FALSE, col = Bcol,  main = "")

  if(DEBUG) print("start 9")
  ## 9
  ifelse(length(maRb(data))!=0 , RS2N <- as.vector(log(maRf(data) / maRb(data),2)),
         RS2N <- as.vector(log(maRf(data),2)))
  lab <- paste("mean:", round(mean(RS2N), 2),",", "var:", round(var(rm.na(RS2N)), 2))
  hist(rm.na(RS2N), main=lab, col="red", freq=FALSE, ylim=c(0,1.1));

  if(length(maControls(data))!=0)
    {
      tmp <- split(RS2N, maControls(data))
      tmp2 <- lapply(tmp, function(x){density(rm.na(x), sd(rm.na(RS2N))/4, na.rm=TRUE)})
      for(i in 1:length(tmp2))
        lines(tmp2[[i]], lwd=2, col=colcode[i])
      xrange <- range(rm.na(RS2N))
      xcood <- xrange[1] + (xrange[2]-xrange[1]) * 0.7
      legend(xcood, 1, names(tmp), lty=1, lwd=2, col=colcode, cex=0.8)
    }
  
  if(DEBUG) print("start 10")
  ##10
  ifelse(length(maGb(data))!=0, GS2N <- as.vector(log(maGf(data) / maGb(data),2)),
         GS2N <- as.vector(log(maGf(data),2)))
  lab <- paste("mean:", round(mean(GS2N), 2),",", "var:", round(var(rm.na(GS2N)), 2))
  hist(rm.na(GS2N), main=lab, col="green", freq=FALSE, ylim=c(0,1.1));

  if(length(maControls(data))!=0)
    {
      tmp <- split(GS2N, maControls(data))
      tmp2 <- lapply(tmp, function(x){density(rm.na(x), sd(rm.na(GS2N))/4, na.rm=TRUE)})
      for(i in 1:length(tmp2))
        lines(tmp2[[i]], lwd=2, col=colcode[i])
      xrange <- range(rm.na(RS2N))
      xcood <- xrange[1] + (xrange[2]-xrange[1]) * 0.7
      legend(xcood, 1, names(tmp), lty=1, lwd=2, col=colcode, cex=0.8)
    }

  if(DEBUG) print("start 11")
  ## 11
  if(length(maControls(data))!=0)
    maDotPlots(data, x="maM", col=colcode)

  if(DEBUG) print("start 12")
  ## 12
  if(length(maControls(data))!=0)
    maDotPlots(data, x="maA",  col=colcode)
  
  ## 13
  layout(1)
  par(mar=c(2,2,4,2))
  mtext(def$main, line=3)
  mtext(subnames, line=2, cex = 0.7)
  mtext(paste("Call:", maNormCall(normdata)[3]), line=1, cex = 0.7)
  

  ## Finishing
  if (save == TRUE) {
    cat(paste("save as", fname, "\n"))
    dev.off()
  }
}


maDotPlots <- function(data,
                       x=list("maA"),
                       id="ID",
                       pch,
                       col,
                       nrep=3,
                       ...)
  {
    newdata <- NULL
    for(i in x)
      newdata <- cbind(newdata, eval(call(i, data)))

    if(!is.null(newdata))
      {
        colnames(newdata) <- x
        xlim <- range(newdata, na.rm=TRUE)
      }
    else
      stop("No specified data")

    Cindex <- maControls(data) != "probes"
    ifelse(missing(pch), pchcode <- (1:ncol(newdata))+15, pchcode <- pch)
    ifelse(missing(col), colcode <- unique(as.integer(maControls(data))+1),
           colcode <- col)
    names(colcode) <- levels(maControls(data))
    Ctl <- cbind(maInfo(maGnames(data)), maControls(data))

    IDindex <- grep(id, colnames(Ctl))
    y <- split(Ctl, Ctl[,ncol(Ctl)])
    if(length(y[names(y) != "probes"]) != 0)
      {
        exty <- lapply(y[names(y) != "probes"], function(x){
          ext <- split(x, x[, IDindex])
          extid <- lapply(ext, function(xx){as.integer(row.names(xx))})
          extid[lapply(extid, length) > nrep]
        })
        exty <- exty[lapply(exty, length) != 0]
        
        ylim <- c(1, sum(unlist(lapply(exty, length))))
        
        par(mar=c(4,7,2,2))
        plot(1,1, type="n", xlim=xlim, ylim=ylim, axes=FALSE, xlab=unlist(x), ylab="")
        ii <- 1
        for(i in 1:length(exty))
          for(j in 1:length(exty[[i]]))
            {
              ind <- exty[[i]][[j]]
              for(k in 1:ncol(newdata))
                {
                  points(newdata[ind,k], rep(ii, length(newdata[ind,k])), pch=pchcode[k],
                         col=colcode[names(exty)[i]])
                  points(median(newdata[ind, k], na.rm=TRUE), ii, pch=18, col="black")
                }
              ii <- ii + 1
            }
        axis(1)
        lab <- paste(unlist(lapply(exty, names)), " (n=",
                     unlist(lapply(exty, lapply, length)), ") ", sep="")
        axis(2, at=1:ylim[2], labels=lab, las=2, cex.axis=0.6) 
        box()
      }
    else
      {
        plot(1, 1, axes=FALSE, xlab="", ylab="", type="n")
        text(1, 1, "No Control Genes")
        box()
      }
    return()
  }
############################################################################
# maPlot3.R
#
# S4 methods
# Diagnostic plots for two-color cDNA microarrays
#
###########################################################################


  if(!isGeneric("boxplot"))  setGeneric("boxplot")
  setMethod("boxplot", signature(x="marrayRaw"), function (x, xvar = "maPrintTip", yvar = "maM", ...)
            {
              maBoxplot(m=x, x=xvar, y=yvar, ...)
            }
            )
  setMethod("boxplot", signature(x="marrayNorm"), function (x, xvar = "maPrintTip", yvar = "maM", ...)
            {
              maBoxplot(m=x, x=xvar, y=yvar, ...)
            }
            )


  if(!isGeneric("image"))  setGeneric("image")
  setMethod("image", signature(x="marrayRaw"), function (x, xvar = "maM", subset = TRUE, col, contours = FALSE,  bar = TRUE, ...)
            {
              maImage(m=x, x=xvar, subset=subset, col=col, contours=contours, bar=bar, ... )
            }
            )

  setMethod("image", signature(x="marrayNorm"), function (x, xvar = "maM", subset = TRUE, col, contours = FALSE,  bar = TRUE, ...)
            {
              maImage(m=x, x=xvar, subset=subset, col=col, contours=contours, bar=bar, ... )
            }
            )

.First.lib <- function(libname, pkgname, where) {

  require("marrayClasses") || stop("marrayClasses is needed")
  require("marrayInput") || stop("marrayInput is needed")
  require("modreg") || stop("modreg is needed")
}

