\name{.aaf.character}
\alias{.aaf.character}
\title{Generic constructor for classes extending character}
\description{
  Retrieve annotation from a character data source.
}

\usage{
.aaf.character(probeids, chip, type, class)
}
\arguments{
  \item{probeids}{character vector containing probe ids}
  \item{chip}{name of chip}
  \item{type}{type of annotation}
  \item{class}{class of object to be created}
}

\value{
  A list of objects of class \code{class} containing the annotation 
  data of from the \code{type} dataset for the given \code{probeids}.
  NA values are returned as empty objects.
}
\note{
  Written at the NASA Center for Computational Astrobiology \cr
  \url{http://cca.arc.nasa.gov/}
}
\author{Colin A. Smith, \email{annaffy@colinsmith.org}}
\keyword{ file }

\eof
\name{.aaf.goterm}
\alias{.aaf.goterm}
\title{Fetch Gene Ontology term information}
\description{
  Given a Gene Ontology number, return its type and name.
}

\usage{
.aaf.goterm(num)
}
\arguments{
  \item{num}{Gene Ontology number - should be formatted \code{GO:XXXXXXX}}
}

\value{
  A list with components
  \item{type}{Type of GO record, either Biological Process, Cellular Component,
    or Molecular Function.}
  \item{name}{A character vector containing the GO name.}
}
\note{
  Written at the NASA Center for Computational Astrobiology \cr
  \url{http://cca.arc.nasa.gov/}
}
\author{Colin A. Smith, \email{annaffy@colinsmith.org}}
\keyword{ file }

\eof
\name{aaf.handler}
\alias{aaf.handler}
\title{Handle feching annotation data columns}
\description{
  Dispatches requests for annotation data to the correct function. Alternatively
  returns a list of all the columns it supports.
}

\usage{
  aaf.handler(probeids, chip, name)
}
\arguments{
  \item{probeids}{character vector containing probe ids}
  \item{chip}{name of chip}
  \item{name}{name of the column of data to return}
}
\value{
  An \code{aafList} containing objects of the propper class.
  
  If no arguments are passed, it will return a character vector of the columns
  currently supported.
}
\note{
  Written at the NASA Center for Computational Astrobiology \cr
  \url{http://cca.arc.nasa.gov/}
}
\author{Colin A. Smith, \email{annaffy@colinsmith.org}}
\keyword{ file }

\eof
\name{.aaf.integer}
\alias{.aaf.integer}
\title{Generic constructor for classes extending integer}
\description{
  Retrieve annotation from an integer data source.
}

\usage{
.aaf.integer(probeids, chip, type, class)
}
\arguments{
  \item{probeids}{character vector containing probe ids}
  \item{chip}{name of chip}
  \item{type}{type of annotation}
  \item{class}{class of object to be created}
}

\value{
  A list of objects of class \code{class} containing the annotation 
  data of from the \code{type} dataset for the given \code{probeids}.
  NA values are returned as empty objects.
}
\note{
  Written at the NASA Center for Computational Astrobiology \cr
  \url{http://cca.arc.nasa.gov/}
}
\author{Colin A. Smith, \email{annaffy@colinsmith.org}}
\keyword{ file }

\eof
\name{.aaf.raw}
\alias{.aaf.raw}
\title{Fetch raw annotation data}
\description{
  Retrieve annotation data from a data package, loading the library if 
  necessary.
}

\usage{
.aaf.raw(probeids, chip, type)
}
\arguments{
  \item{probeids}{character vector containing probe ids}
  \item{chip}{name of chip, see details}
  \item{type}{type of annotation, see details}
}
\details{
  The core workings of this function depend on an (informal) protocol
  used in creating the BioConductor Affymetrix annotation data
  packages. Based on currently published (and unpublished) data packages,
  the current protocol includes the following features:

  The package is named after the chip, \code{<chip name>} \cr
  The package contains datasets named \code{<chip name><data type>}
}

\value{
  A list of annotation data for the given probeids. Each list contains a
  sub-list containing the actual data.
}
\note{
  Written at the NASA Center for Computational Astrobiology \cr
  \url{http://cca.arc.nasa.gov/}
}
\author{Colin A. Smith, \email{annaffy@colinsmith.org}}
\keyword{ file }

\eof
\name{aafChromLoc-class}
\docType{class}
\alias{aafChromLoc-class}
\title{Class aafChromLoc, a class for gene chromosome locations}
\description{
  An abstraction for gene chromosome locations from Bioconductor
  data packages.
}

\section{Objects from the Class}{
Objects are generally created by the \code{\link{aafChromLoc}}
constructor.  Objects can also be created manually by calls of the
form \code{new("aafChromLoc", description)}.
}
\section{Slots}{
  \describe{
    \item{\code{.Data}:}{Object of class \code{integer}}
  }
}
\section{Extends}{
Class \code{integer}, from data part.
}

\section{Methods}{
No methods defined with class "aafChromLoc" in the signature.
See generic implementations of \code{\link{getText}}, \code{\link{getURL}},
\code{\link{getHTML}}, \code{\link{getTD}}, and \code{\link{getCSS}}.
}

\author{Colin A. Smith, \email{annaffy@colinsmith.org}}

\seealso{\link{aafChromLoc}}

\keyword{classes}

\eof
\name{aafChromLoc}
\alias{aafChromLoc}
\title{Constructor for aafChromLoc objects}
\description{
  For the given \code{probeids}, constructs an \code{aafList} of
  \code{aafChromLoc} objects containing annotation data from the
  \code{chip} data package.
}

\usage{
aafChromLoc(probeids, chip)
}
\arguments{
  \item{probeids}{character vector containing probe ids}
  \item{chip}{name of the chip data package}
}

\value{
  An \code{aafList} of \code{aafChromLoc} objects. NA values are
  returned as empty objects.
}

\author{Colin A. Smith, \email{annaffy@colinsmith.org}}

\examples{
probes <- ls(annaffyCHRLOC)
locations <- aafChromLoc(probes, "annaffy")
show(locations[6:10])
}

\seealso{\link{aafChromLoc-class}}

\keyword{ file }

\eof
\name{aafChromosome-class}
\docType{class}
\alias{aafChromosome-class}
\title{Class aafChromosome, a class for gene chromosome assignments}
\description{
  An abstraction for gene gene chromosome assignments from Bioconductor
  data packages.
}

\section{Objects from the Class}{
Objects are generally created by the \code{\link{aafChromosome}}
constructor.  Objects can also be created manually by calls of the
form \code{new("aafChromosome", description)}.
}
\section{Slots}{
  \describe{
    \item{\code{.Data}:}{Object of class \code{character}}
  }
}
\section{Extends}{
Class \code{character}, from data part.
}

\section{Methods}{
No methods defined with class "aafChromosome" in the signature.
See generic implementations of \code{\link{getText}}, \code{\link{getURL}},
\code{\link{getHTML}}, \code{\link{getTD}}, and \code{\link{getCSS}}.
}

\author{Colin A. Smith, \email{annaffy@colinsmith.org}}

\seealso{\link{aafChromosome}}

\keyword{classes}

\eof
\name{aafChromosome}
\alias{aafChromosome}
\title{Constructor for aafChromosome objects}
\description{
  For the given \code{probeids}, constructs an \code{aafList} of
  \code{aafChromosome} objects containing annotation data from the
  \code{chip} data package.
}

\usage{
aafChromosome(probeids, chip)
}
\arguments{
  \item{probeids}{character vector containing probe ids}
  \item{chip}{name of the chip data package}
}

\value{
  An \code{aafList} of \code{aafChromosome} objects. NA values are
  returned as empty objects.
}

\author{Colin A. Smith, \email{annaffy@colinsmith.org}}

\examples{
probes <- ls(annaffyCHR)
chromosomes <- aafChromosome(probes, "annaffy")
show(chromosomes[6:10])
}

\seealso{\link{aafChromosome-class}}

\keyword{ file }

\eof
\name{aafCytoband-class}
\docType{class}
\alias{aafCytoband-class}
\alias{getText,aafCytoband-method}
\alias{getURL,aafCytoband-method}
\alias{show,aafCytoband-method}
\title{Class aafCytoband, a class for cytoband data}
\description{
  An abstraction for cytoband data from Bioconductor data packages.
}

\section{Objects from the Class}{
Objects are generally created by the \code{\link{aafCytoband}}
constructor.  Objects can also be created manually by calls of the
form \code{new("aafCytoband", band, genbank)}.
}
\section{Slots}{
  \describe{
    \item{\code{band}:}{
      Object of class \code{character} containing genomic cytoband
    }
    \item{\code{genbank}:}{
      Object of class \code{character} containing containing GenBank
      accession number
    }
  }
}
\section{Methods}{
  \describe{
    \item{getText}{
      (aafCytoband): Returns text of \code{band}.
    }
    \item{getURL}{
      (aafCytoband): Returns a URL corresponding entry in NCBI's
      cytoband map viewer.
    }
  }
  See generic implementations of \code{\link{getHTML}},
  \code{\link{getTD}}, and \code{\link{getCSS}}.
}

\author{Colin A. Smith, \email{annaffy@colinsmith.org}}

\seealso{
  \link{aafCytoband}
}

\keyword{classes}

\eof
\name{aafCytoband}
\alias{aafCytoband}
\title{Constructor for aafCytoband objects}
\description{
  For the given \code{probeids}, constructs an \code{aafList} of
  \code{aafCytoband} objects containing annotation data from the
  \code{chip} data package.
}

\usage{
aafCytoband(probeids, chip)
}
\arguments{
  \item{probeids}{character vector containing probe ids}
  \item{chip}{name of the chip data package}
}

\value{
  An \code{aafList} of \code{aafCytoband} objects. NA values are returned as
  empty objects.
}

\author{Colin A. Smith, \email{annaffy@colinsmith.org}}

\examples{
probes <- ls(annaffyMAP)
bands <- aafCytoband(probes, "annaffy")
show(bands[6:10])
}

\seealso{\link{aafCytoband-class}}

\keyword{ file }

\eof
\name{aafDescription-class}
\docType{class}
\alias{aafDescription-class}
\title{Class aafDescription, a class for gene descriptions}
\description{
  An abstraction for gene description from Bioconductor data packages.
}

\section{Objects from the Class}{
Objects are generally created by the \code{\link{aafDescription}}
constructor.  Objects can also be created manually by calls of the
form \code{new("aafDescription", description)}.
}
\section{Slots}{
  \describe{
    \item{\code{.Data}:}{Object of class \code{character}}
  }
}
\section{Extends}{
Class \code{character}, from data part.
}

\section{Methods}{
No methods defined with class "aafDescription" in the signature.
See generic implementations of \code{\link{getText}}, \code{\link{getURL}},
\code{\link{getHTML}}, and \code{\link{getTD}}.
}

\author{Colin A. Smith, \email{annaffy@colinsmith.org}}

\seealso{\link{aafDescription}}

\keyword{classes}

\eof
\name{aafDescription}
\alias{aafDescription}
\title{Constructor for aafDescription objects}
\description{
  For the given \code{probeids}, constructs an \code{aafList} of
  \code{aafDescription} objects containing annotation data from the
  \code{chip} data package.
}

\usage{
aafDescription(probeids, chip)
}
\arguments{
  \item{probeids}{character vector containing probe ids}
  \item{chip}{name of the chip data package}
}

\value{
  An \code{aafList} of \code{aafDescription} objects. NA values are returned as
  empty objects.
}

\author{Colin A. Smith, \email{annaffy@colinsmith.org}}

\examples{
probes <- ls(annaffyGENENAME)
descriptions <- aafDescription(probes, "annaffy")
show(descriptions[6:10])
}

\seealso{\link{aafDescription-class}}

\keyword{ file }

\eof
\name{aafExpr}
\alias{aafExpr}
\title{Sample exprSet used for demonstration purposes}
\description{
Contains expression values for 250 probe ids with 8 samples. Two
covariates are provided. Expression comes from the \code{hgu95av2}
chip.
}
\details{
The data is real but anonymized. 250 genes expression values were
chosen at random from an existing exprSet. Another 250 probe ids
were selected at random and were assigned to the expression values.
That way, expression values do not correspond to the true probe
ids.

Post-processing was done with \code{rma()} in \code{affy} 1.2.23.
}

\keyword{datasets}


\eof
\name{aafFunction-class}
\docType{class}
\alias{aafFunction-class}
\title{Class aafFunction, a class for gene product functions}
\description{
  An abstraction for gene product function from Bioconductor data packages.
}

\section{Objects from the Class}{
Objects are generally created by the \code{\link{aafFunction}}
constructor.  Objects can also be created manually by calls of the
form \code{new("aafFunction", description)}.
}
\section{Slots}{
  \describe{
    \item{\code{.Data}:}{Object of class \code{character}}
  }
}
\section{Extends}{
Class \code{character}, from data part.
}

\section{Methods}{
No methods defined with class "aafFunction" in the signature.
See generic implementations of \code{\link{getText}}, \code{\link{getURL}},
\code{\link{getHTML}}, \code{\link{getTD}}, and \code{\link{getCSS}}.
}

\author{Colin A. Smith, \email{annaffy@colinsmith.org}}

\seealso{\link{aafFunction}}

\keyword{classes}

\eof
\name{aafFunction}
\alias{aafFunction}
\title{Constructor for aafFunction objects}
\description{
  For the given \code{probeids}, constructs an \code{aafList} of
  \code{aafFunction} objects containing annotation data from the
  \code{chip} data package.
}

\usage{
aafFunction(probeids, chip)
}
\arguments{
  \item{probeids}{character vector containing probe ids}
  \item{chip}{name of the chip data package}
}

\value{
  An \code{aafList} of \code{aafFunction} objects. NA values are
  returned as empty objects.
}

\author{Colin A. Smith, \email{annaffy@colinsmith.org}}

\examples{
probes <- ls(annaffySUMFUNC)
functions <- aafFunction(probes, "annaffy")
show(functions[6:10])
}

\seealso{\link{aafFunction-class}}

\keyword{ file }

\eof
\name{aafGO-class}
\docType{class}
\alias{aafGO-class}
\alias{getText,aafGO-method}
\alias{getURL,aafGO-method}
\alias{getHTML,aafGO-method}
\alias{getTD,aafGO-method}
\alias{getCSS,aafGO-method}
\title{Class aafGO, a class for gene ontology ids}
\description{
  An abstraction for gene ontology ids from Bioconductor data
  packages. This class is actually extends \code{aafList} and holds
  \code{aafGOItem} objects which have the actual annotation data.
}

\section{Objects from the Class}{
Objects are generally created by the \code{\link{aafGO}} constructor.
Objects can also be created manually by calls of the form
\code{new("aafGO", list(goitems))}.
}
\section{Slots}{
  \describe{
    \item{\code{.Data}:}{Object of class \code{list}}
  }
}
\section{Extends}{
Class \code{aafList}, from data part.
}

\section{Methods}{
  \describe{
    \item{getText}{
      (aafGO): Returns a comma delimeted list of the individual
      \code{aafGOItem} objects.
    }
    \item{getURL}{
      (aafGO): Returns a single URL to an AmiGO page which displays
      all the gene ontology identifiers in an hierarchical listing.
    }
    \item{getHTML}{
      (aafGO): Returns an HTML representation of each of the
      individual \code{aafGOItem} objects, concatenated together.
    }
    \item{getTD}{
      (aafGO): Returns an HTML table cell representation with the
      class set to "aafGO".
    }
    \item{getCSS}{
      (aafGOItem): Returns a line of CSS that indents GOItem paragraphs.
    }
  }
}

\author{Colin A. Smith, \email{annaffy@colinsmith.org}}

\seealso{
  \link{aafGO},
  \link{aafGOItem},
  \link{aafList}
}

\keyword{classes}

\eof
\name{aafGO}
\alias{aafGO}
\title{Constructor for aafGO objects}
\description{
  For the given \code{probeids}, constructs an \code{aafList} of
  \code{aafGO} objects containing annotation data from the
  \code{chip} data package.
}

\usage{
aafGO(probeids, chip)
}
\arguments{
  \item{probeids}{character vector containing probe ids}
  \item{chip}{name of the chip data package}
}

\value{
  An \code{aafList} of \code{aafGO} objects. NA values are returned as
  empty objects.
}

\author{Colin A. Smith, \email{annaffy@colinsmith.org}}

\examples{
probes <- ls(annaffyGO)
gos <- aafGO(probes, "annaffy")
show(gos[6:10])
}

\seealso{\link{aafGO-class}}

\keyword{ file }

\eof
\name{aafGOItem-class}
\docType{class}
\alias{aafGOItem-class}
\alias{aafGOItem}
\alias{getText,aafGOItem-method}
\alias{getURL,aafGOItem-method}
\alias{getHTML,aafGOItem-method}
\alias{show,aafGOItem-method}
\title{Class aafGOItem, a class for gene ontology id elements}
\description{
  An abstraction for gene ontology id elements from Bioconductor
  data packages. Multiple instances of this class are held by the
  wrapper class \code{aafGO}.
}

\section{Objects from the Class}{
Objects are generally created by the \code{\link{aafGO}} constructor.
Objects can also be created manually by calls of the form
\code{new("aafGOItem", id, name, type)}.
}
\section{Slots}{
  \describe{
    \item{\code{id}:}{
      Object of class \code{character} containing GO id
    }
    \item{\code{name}:}{
      Object of class \code{character} containing textual name
    }
    \item{\code{type}:}{
      Object of class \code{character} containing GO subtype
    }
    \item{\code{evid}:}{
      Object of class \code{character} containing GO evidence code
    }
  }
}
\section{Methods}{
  \describe{
    \item{getText}{
      (aafGOItem): Returns textual representation formatted
      "\code{id}: \code{name}".
    }
    \item{getURL}{
      (aafGOItem): Returns a URL to the corresponding gene ontology
      entry on AmiGO.
    }
    \item{getHTML}{
      (aafGOItem): Returns an HTML representation including the
      URL link, gene ontology name, and rollover subtype.
    }
  }
}

\author{Colin A. Smith, \email{annaffy@colinsmith.org}}

\seealso{
  \link{aafGO-class},
  \link{aafGO}
}

\keyword{classes}

\eof
\name{aafGenBank-class}
\docType{class}
\alias{aafGenBank-class}
\alias{getURL,aafGenBank-method}
\title{Class aafGenBank, a class for GenBank accession numbers}
\description{
  An abstraction for GenBank accession numbers from Bioconductor data packages.
}

\section{Objects from the Class}{
Objects are generally created by the \code{\link{aafGenBank}}
constructor.  Objects can also be created manually by calls of the
form \code{new("aafGenBank", accnum)}.
}
\section{Slots}{
  \describe{
    \item{\code{.Data}:}{Object of class \code{character}}
  }
}
\section{Extends}{
Class \code{character}, from data part.
}

\section{Methods}{
  \describe{
    \item{getURL}{
      (aafGenBank): Returns a URL to the corresponding entry in NCBI's
      GenBank database.
    }
  }
  See generic implementations of \code{\link{getText}},
  \code{\link{getHTML}}, and \code{\link{getTD}}.
}

\author{Colin A. Smith, \email{annaffy@colinsmith.org}}

\seealso{\link{aafGenBank}}

\keyword{classes}

\eof
\name{aafGenBank}
\alias{aafGenBank}
\title{Constructor for aafGenBank objects}
\description{
  For the given \code{probeids}, constructs an \code{aafList} of
  \code{aafGenBank} objects containing annotation data from the
  \code{chip} data package.
}

\usage{
aafGenBank(probeids, chip)
}
\arguments{
  \item{probeids}{character vector containing probe ids}
  \item{chip}{name of the chip data package}
}

\value{
  An \code{aafList} of \code{aafGenBank} objects. NA values are returned as
  empty objects.
}

\author{Colin A. Smith, \email{annaffy@colinsmith.org}}

\examples{
probes <- ls(annaffyACCNUM)
gbs <- aafGenBank(probes, "annaffy")
show(gbs[6:10])
}

\seealso{\link{aafGenBank-class}}

\keyword{ file }

\eof
\name{aafIntensity-class}
\docType{class}
\alias{aafIntensity-class}
\alias{aafIntensity}
\alias{getTD,aafIntensity-method}
\title{Class aafIntensity, a class for gene expression values}
\description{
  A class for displaying gene expression values with a green
  background of differing intensities.
}

\section{Objects from the Class}{
Objects are generally created by the \code{\link{aafTableInt}}
constructor.  Objects can also be created manually by calls of the
form \code{new("aafIntensity", intensity)}.
}
\section{Slots}{
  \describe{
    \item{\code{.Data}:}{Object of class \code{numeric}}
  }
}
\section{Extends}{
Class \code{numeric}, from data part.
}

\section{Methods}{
  \describe{
    \item{getTD}{
      (aafIntensity): Returns an HTML table cell with background
      varrying from white to green depending on intensity. Scaling
      is controlled by two options, \code{minIntensity} (fully
      white) and \code{maxIntensity} (fully green), usually set by
      writeHTML.
    }
  }
  See generic implementations of \code{\link{getText}},
  \code{\link{getURL}}, \code{\link{getHTML}}, and \code{\link{getCSS}}.
}

\author{Colin A. Smith, \email{annaffy@colinsmith.org}}

\seealso{\link{aafTableInt}}

\keyword{classes}

\eof
\name{aafList-class}
\docType{class}
\alias{aafList-class}
\alias{aafList}
\alias{getText,aafList-method}
\alias{getURL,aafList-method}
\alias{getHTML,aafList-method}
\alias{getTD,aafList-method}
\alias{getCSS,aafList-method}
\alias{[,aafList-method}
\alias{show,aafList-method}
\title{Class aafList, a specialized subclass of list}
\description{
  A class for lists of annotation data objects.
}

\section{Objects from the Class}{
Objects are generally created by any of the annotation data constructors
that are also part of this package. Objects can also be created manually 
by calls of the form \code{new("aafList", list)}.
}
\section{Slots}{
  \describe{
    \item{\code{.Data}:}{Object of class \code{list}}
  }
}
\section{Extends}{
Class \code{list}, from data part.
}

\section{Methods}{
  \describe{
    \item{getText}{
      (aafList): Returns a character vector containing textual
      representations of every item.
    }
    \item{getURL}{
      (aafList): Returns a character vector containing single URLs
      (if possible) of every item.
    }
    \item{getHTML}{
      (aafList): Returns a character vector containing HTML
      representations of every item.
    }
    \item{getTD}{
      (aafList): Returns a character vector containing HTML table
      cell representations of every item.
    }
    \item{getCSS}{
      (aafList): Returns getCSS() of the first item in the list.
    }
    \item{[}{
      (aafList): Returns a subset of aafList as another \code{aafList} 
      object.
    }
  }
}

\author{Colin A. Smith, \email{annaffy@colinsmith.org}}

\keyword{classes}

\eof
\name{aafLocusLink-class}
\docType{class}
\alias{aafLocusLink-class}
\alias{getURL,aafLocusLink-method}
\title{Class aafLocusLink, a class for LocusLink ids}
\description{
  An abstraction for LocusLink ids from Bioconductor data packages.
}

\section{Objects from the Class}{
Objects are generally created by the \code{\link{aafLocusLink}}
constructor.  Objects can also be created manually by calls of the
form \code{new("aafLocusLink", id)}.
}
\section{Slots}{
  \describe{
    \item{\code{.Data}:}{Object of class \code{integer}}
  }
}
\section{Extends}{
Class \code{integer}, from data part.
}

\section{Methods}{
  \describe{
    \item{getURL}{
      (aafLocusLink): Returns a URL to the corresponding entry in
      NCBI's LocusLink database. On the rare chance that more than
      one id is defined, more than one URL will be returned.
    }
  }
  See generic implementations of \code{\link{getText}},
  \code{\link{getHTML}}, and \code{\link{getTD}}.
}

\author{Colin A. Smith, \email{annaffy@colinsmith.org}}

\seealso{\link{aafLocusLink}}

\keyword{classes}

\eof
\name{aafLocusLink}
\alias{aafLocusLink}
\title{Constructor for aafLocusLink objects}
\description{
  For the given \code{probeids}, constructs an \code{aafList} of
  \code{aafLocusLink} objects containing annotation data from the
  \code{chip} data package.
}

\usage{
aafLocusLink(probeids, chip)
}
\arguments{
  \item{probeids}{character vector containing probe ids}
  \item{chip}{name of the chip data package}
}

\value{
  An \code{aafList} of \code{aafLocusLink} objects. NA values are returned as
  empty objects.
}

\author{Colin A. Smith, \email{annaffy@colinsmith.org}}

\examples{
probes <- ls(annaffyLOCUSID)
lls <- aafLocusLink(probes, "annaffy")
show(lls[6:10])
}

\seealso{\link{aafLocusLink-class}}

\keyword{ file }

\eof
\name{aafPathway-class}
\docType{class}
\alias{aafPathway-class}
\alias{getText,aafPathway-method}
\alias{getURL,aafPathway-method}
\alias{getHTML,aafPathway-method}
\alias{getTD,aafPathway-method}
\alias{getCSS,aafPathway-method}
\title{Class aafPathway, a class for KEGG pathway ids}
\description{
  An abstraction for KEGG pathway ids from Bioconductor data
  packages. This class is actually extends \code{aafList} and holds
  \code{aafPathwayItem} objects which have the actual annotation
  data.
}
\section{Objects from the Class}{
Objects are generally created by the \code{\link{aafPathway}} constructor.
Objects can also be created manually by calls of the form
\code{new("aafPathway", list(pathwayitems))}.
}
\section{Slots}{
  \describe{
    \item{\code{.Data}:}{Object of class \code{list}}
  }
}
\section{Extends}{
Class \code{aafList}, from data part.
}

\section{Methods}{
  \describe{
    \item{getText}{
      (aafGO): Returns a comma delimeted list of the individual
      \code{aafPathwayItem} objects.
    }
    \item{getURL}{
      (aafGO): Returns zero length character vector because this
      method is not valid for this class.
    }
    \item{getHTML}{
      (aafGO): Returns an HTML representation of each of the
      individual \code{aafPathwayItem} objects, concatenated together.
    }
    \item{getTD}{
      (aafGO): Returns an HTML table cell representation with the
      class set to "aafPathway".
    }
    \item{getCSS}{
      (aafGO): Returns a line of CSS which intends PathwayItem
      paragraphs.
    }
  }
}

\author{Colin A. Smith, \email{annaffy@colinsmith.org}}

\seealso{
  \link{aafPathway},
  \link{aafPathwayItem},
  \link{aafList}
}

\keyword{classes}

\eof
\name{aafPathway}
\alias{aafPathway}
\title{Constructor for aafPathway objects}
\description{
  For the given \code{probeids}, constructs an \code{aafList} of
  \code{aafPathway} objects containing annotation data from the
  \code{chip} data package.
}

\usage{
aafPathway(probeids, chip)
}
\arguments{
  \item{probeids}{character vector containing probe ids}
  \item{chip}{name of the chip data package}
}

\value{
  An \code{aafList} of \code{aafPathway} objects. NA values are returned as
  empty objects.
}

\author{Colin A. Smith, \email{annaffy@colinsmith.org}}

\examples{
probes <- ls(annaffyPATH)
pathways <- aafPathway(probes, "annaffy")
show(pathways[6:10])
}

\seealso{\link{aafPathway-class}}

\keyword{ file }

\eof
\name{aafPathwayItem-class}
\docType{class}
\alias{aafPathwayItem-class}
\alias{aafPathwayItem}
\alias{getText,aafPathwayItem-method}
\alias{getURL,aafPathwayItem-method}
\alias{getHTML,aafPathwayItem-method}
\alias{show,aafPathwayItem-method}
\title{Class aafPathwayItem, a class for KEGG pathway id elements}
\description{
  An abstraction for KEGG pathway id elements from Bioconductor
  data packages. Multiple instances of this class are held by the
  wrapper class \code{aafPathway}.
}

\section{Objects from the Class}{
Objects are generally created by the \code{\link{aafPathway}} constructor.
Objects can also be created manually by calls of the form
\code{new("aafPathwayItem", id, name, enzyme)}.
}
\section{Slots}{
  \describe{
    \item{\code{id}:}{
      Object of class \code{character} containing KEGG pathway id
    }
    \item{\code{name}:}{
      Object of class \code{character} containing textual name
    }
    \item{\code{enzyme}:}{
      Object of class \code{character} containing the Enzyme
      Commision number if applicable
    }
  }
}

\section{Methods}{
  \describe{
    \item{getText}{
      (aafPathwayItem): Returns textual representation formatted
      "\code{id}: \code{name}".
    }
    \item{getURL}{
      (aafPathwayItem): Returns a URL to the corresponding entry
      in the Kyoto Encyclopedia of Genes and Genomes database. If
      there is a corresponding EC number, it will be highlighted
      in red.
    }
    \item{getHTML}{
      (aafPathwayItem): Returns an HTML representation including the
      URL link and pathway name.
    }
  }
}

\author{Colin A. Smith, \email{annaffy@colinsmith.org}}

\seealso{
  \link{aafPathway-class},
  \link{aafPathway}
}

\keyword{classes}

\eof
\name{aafProbe-class}
\docType{class}
\alias{aafProbe-class}
\alias{getURL,aafProbe-method}
\title{Class aafProbe, a class for Probe ids}
\description{
  An abstraction for Affymetrix ProbeSet ids.
}

\section{Objects from the Class}{
Objects are generally created by the \code{\link{aafProbe}}
constructor.  Objects can also be created manually by calls of the
form \code{new("aafProbe", id)}.
}
\section{Slots}{
  \describe{
    \item{\code{.Data}:}{Object of class \code{character}}
  }
}
\section{Extends}{
Class \code{character}, from data part.
}

\section{Methods}{
  \describe{
    \item{getURL}{
      (aafProbe): Returns a URL to the annotation found in the
      Affymetrix NetAffx Analysis Center.
    }
  }
  See generic implementations of \code{\link{getText}},
  \code{\link{getHTML}}, and \code{\link{getTD}}.
}

\author{Colin A. Smith, \email{annaffy@colinsmith.org}}

\seealso{\link{aafProbe}}

\keyword{classes}

\eof
\name{aafProbe}
\alias{aafProbe}
\title{Constructor for aafProbe objects}
\description{
  For the given \code{probeids}, constructs an \code{aafList} of
  \code{aafProbe} objects.
}

\usage{
aafProbe(probeids)
}
\arguments{
  \item{probeids}{character vector containing probe ids}
}

\value{
  An \code{aafList} of \code{aafProbe} objects.
}

\author{Colin A. Smith, \email{annaffy@colinsmith.org}}

\examples{
probes <- ls(annaffySYMBOL)
probesets <- aafProbe(probes)
getURL(probesets[6:10])
}

\seealso{\link{aafProbe-class}}

\keyword{ file }

\eof
\name{aafPubMed-class}
\docType{class}
\alias{aafPubMed-class}
\alias{getURL,aafPubMed-method}
\alias{getHTML,aafPubMed-method}
\alias{getTD,aafPubMed-method}
\alias{getCSS,aafPubMed-method}
\title{Class aafPubMed, a class for PubMed ids}
\description{
  An abstraction for LocusLink ids from Bioconductor data packages.
}

\section{Objects from the Class}{
Objects are generally created by the \code{\link{aafPubMed}}
constructor.  Objects can also be created manually by calls of the
form \code{new("aafPubMed", id)}.
}
\section{Slots}{
  \describe{
    \item{\code{.Data}:}{Object of class \code{integer}}
  }
}
\section{Extends}{
Class \code{integer}, from data part.
}

\section{Methods}{
  \describe{
    \item{getURL}{
      (aafPubMed): Returns a single URL to the corresponding
      abstracts in NCBI's PubMed database.
    }
    \item{getHTML}{
      (aafPubMed): Returns an HTML link along with the number of
      abstracts.
    }
    \item{getTD}{
      (aafPubMed): Returns an HTML table cell representation with the
      class set to "aafPubMed".
    }
    \item{getCSS}{
      (aafPubMed): Returns a line of CSS which centers the PubMed
      link.
    }
  }
}

\author{Colin A. Smith, \email{annaffy@colinsmith.org}}

\seealso{\link{aafPubMed}}

\keyword{classes}

\eof
\name{aafPubMed}
\alias{aafPubMed}
\title{Constructor for aafPubMed objects}
\description{
  For the given \code{probeids}, constructs a list of
  \code{aafPubMed} objects containing annotation data from the
  \code{chip} data package.
}

\usage{
aafPubMed(probeids, chip)
}
\arguments{
  \item{probeids}{character vector containing probe ids}
  \item{chip}{name of the chip data package}
}

\value{
  An \code{aafList} of \code{aafPubMed} objects. NA values are returned as
  empty objects.
}

\author{Colin A. Smith, \email{annaffy@colinsmith.org}}

\examples{
probes <- ls(annaffyPMID)
pmids <- aafPubMed(probes, "annaffy")
show(pmids[6:10])
}

\seealso{\link{aafPubMed-class}}

\keyword{ file }

\eof
\name{aafSearchGO}
\alias{aafSearchGO}
\title{Find probe ids corresponding to GO ids}
\description{
  Searches Gene Ontology ids for corresponding probe ids in a given
  chip, optionally including descendents.
}

\usage{
aafSearchGO(chip, ids, descendents = TRUE, logic = "OR")
}
\arguments{
  \item{chip}{name of the chip data package}
  \item{ids}{numeric or character vector of GO ids}
  \item{descendents}{logical, include GO descendents?}
  \item{logic}{type of logic to use, "AND" or "OR"}
}

\value{
  A character vector of probe ids matching the search criteria.
}

\author{Colin A. Smith, \email{annaffy@colinsmith.org}}

\seealso{\link{aafSearchText}}

\keyword{ file }

\eof
\name{aafSearchText}
\alias{aafSearchText}
\title{Search metadata annotation text}
\description{
  Searches Bioconductor metadata annotation package text for specific
  strings or Perl compatible regular expressions.
}

\usage{
aafSearchText(chip, colnames, text, logic = "OR")
}
\arguments{
  \item{chip}{name of the chip data package}
  \item{colnames}{character vector of metadata column names to search}
  \item{text}{character vector of strings/regular expressons to match}
  \item{logic}{type of logic to use, "AND" or "OR"}
}

\value{
  A character vector of probe ids matching the search criteria.
}

\author{Colin A. Smith, \email{annaffy@colinsmith.org}}

\examples{
aafSearchText("annaffy", "Description", c("kinase", "interferon"))
aafSearchText("annaffy", c("Gene Ontology", "Pathway"), "ribosome")
}

\seealso{\link{aafSearchGO}}

\keyword{ file }

\eof
\name{aafSigned-class}
\docType{class}
\alias{aafSigned-class}
\alias{aafSigned}
\alias{getTD,aafSigned-method}
\alias{getCSS,aafSigned-method}
\title{Class aafSigned, a class for signed numerical data}
\description{
  A class for displaying signed numerical data with different styles
  depending on the sign.
}

\section{Objects from the Class}{
Objects are generally created by the \code{\link{aafTable}}
constructor.  Objects can also be created manually by calls of the
form \code{new("aafSigned", signedval)}.
}
\section{Slots}{
  \describe{
    \item{\code{.Data}:}{Object of class \code{numeric}}
  }
}
\section{Extends}{
Class \code{numeric}, from data part.
}

\section{Methods}{
  \describe{
    \item{getTD}{
      (aafSigned): Returns an HTML table cell with class
      differentially set based on sign.  \code{aafSignedPos} is
      used for positive values. \code{aafSignedNeg} is used for
      negative values. \code{aafSignedZero} is used for zero values.
    }
    \item{getCSS}{
      (aafSigned): Returns two lines of CSS that set the cell
      background of positive values light blue and negative values
      light red.
    }
  }
  See generic implementations of \code{\link{getText}},
  \code{\link{getURL}}, and \code{\link{getHTML}}.
}

\author{Colin A. Smith, \email{annaffy@colinsmith.org}}

\seealso{\link{aafTable}}

\keyword{classes}

\eof
\name{aafSymbol-class}
\docType{class}
\alias{aafSymbol-class}
\title{Class aafSymbol, a class for gene symbols}
\description{
  An abstraction for gene symbol from Bioconductor data packages.
}

\section{Objects from the Class}{
Objects are generally created by the \code{\link{aafSymbol}}
constructor.  Objects can also be created manually by calls of the
form \code{new("aafSymbol", description)}.
}
\section{Slots}{
  \describe{
    \item{\code{.Data}:}{Object of class \code{character} with gene symbol}
  }
}
\section{Extends}{
Class \code{character}, from data part.
}

\section{Methods}{
No methods defined with class "aafSymbol" in the signature.
See generic implementations of \code{\link{getText}}, \code{\link{getURL}},
\code{\link{getHTML}}, and \code{\link{getTD}}.
}

\author{Colin A. Smith, \email{annaffy@colinsmith.org}}

\seealso{\link{aafSymbol}}

\keyword{classes}

\eof
\name{aafSymbol}
\alias{aafSymbol}
\title{Constructor for aafSymbol objects}
\description{
  For the given \code{probeids}, constructs a list of
  \code{aafSymbol} objects containing annotation data from the
  \code{chip} data package.
}

\usage{
aafSymbol(probeids, chip)
}
\arguments{
  \item{probeids}{character vector containing probe ids}
  \item{chip}{name of the chip data package}
}

\value{
  An \code{aafList} of \code{aafSymbol} objects. NA values are returned as
  empty objects.
}

\author{Colin A. Smith, \email{annaffy@colinsmith.org}}

\examples{
probes <- ls(annaffySYMBOL)
symbols <- aafSymbol(probes, "annaffy")
show(symbols[6:10])
}

\seealso{\link{aafSymbol-class}}

\keyword{ file }

\eof
\name{aafTable-class}
\docType{class}
\alias{aafTable-class}
\alias{probeids}
\alias{probeids,aafTable-method}
\alias{probeids<-}
\alias{probeids<-,aafTable-method}
\alias{colnames}
\alias{colnames,aafTable,missing,missing-method}
\alias{colnames<-}
\alias{colnames<-,aafTable-method}
\alias{dim.aafTable}
\alias{merge.aafTable}
\alias{rbind.aafTable}
\alias{[,aafTable-method}
\alias{[[,aafTable-method}
\alias{$.aafTable}
\alias{saveHTML}
\alias{saveHTML,aafTable-method}
\alias{saveText}
\alias{saveText,aafTable-method}
\title{Class aafTable, a tabular microarray data class}
\description{
  A class for storing and flexible output of microarray data to
  HTML and text formats.
}

\section{Objects from the Class}{
Objects are generally created by any of the annotation table constructors
that are also part of this package. Objects can also be created manually 
by calls of the form \code{new("aafList", probeids, table)}.
}
\section{Slots}{
  \describe{
    \item{\code{probeids}:}{
      Object of class \code{character} containing the probe ids
      for each row of the table.
    }
    \item{\code{table}:}{
      Object of class \code{list} containing \code{aafList} objects
      all of the same length, representing the columns of the table.
      Each item in the list must have a unique name.
    }
  }
}

\section{Methods}{
  \describe{
    \item{probeids}{
      (aafTable): Returns a character vector containing the probe
      ids for each row of the table.
    }
    \item{probeids<-}{
      (aafTable): Sets the probe ids for the table rows. Can be
      set to \code{character(0)} if unknown or not applicable.
    }
    \item{colnames}{
      (aafTable): Returns a character vector containing the names
      of the columns stored in the table.
    }
    \item{colnames<-}{
      (aafTable): Set the column names for the table. Each must be
      unique.
    }
    \item{dim}{
      (aafTable): Returns the dimensions of the table.
    }
    \item{merge}{
      (aafTable, aafTable, suffixes = c(".x",".y")): Merges two
      tables together, aligning common probe ids if possible.
      Duplicate column names are given suffixes to make them unique.
      Returns the merged table.
    }
    \item{rbind}{
      (aafTable, aafTable, ...): Vertically combines tables by row.
      Requires that column names be identical and that all tables
      either have probe ids defined or not.
    }
    \item{[}{
      Returns a subset of the table based on [row, column]. Indices
      may be passed as integers or probe ids/column names.
    }
    \item{[[}{
      Returns the given table column. This also supports recursive
      subsetting to address columns, then cells, then sub-cells
      (if applicable). See \link{Extract} for more information.
    }
    \item{\$}{
      Returns the given table column.
    }
    \item{saveHTML}{
      (aafTable, filename, title = "Bioconductor Affymetrix Probe
      Listing", colnames = colnames(aafTable), range = 1:dim(aafTable)[1],
      open = FALSE, widget = FALSE):  Saves the table to HTML with
      the specified filename and title. Both the columns and the
      range of table rows can be specified. Range can either be
      specified as a character vector of probe ids or an integer
      vector of row positions. One can also specify whether to open
      the resulting file in the browser and whether to use a widget
      for column selection.
    }
    \item{saveText}{
      (aafTable, filename, header = TRUE, colnames = colnames(aafTable),
      range = 1:dim(aafTable)[1], widget = FALSE): Saves the table
      to tab delimited text with specified filename and optional
      header.  Both the columns and the range of table rows can be
      specified.  Range can either be specified as a character
      vector of probe ids or an integer vector of row positions.
      One can also specify whether to use a widget for column
      selection.
    }
  }
}

\author{Colin A. Smith, \email{annaffy@colinsmith.org}}

\seealso{
  \link{aafTable},
  \link{aafTableFrame},
  \link{aafTableAnn},
  \link{aafTableInt}
}

\keyword{classes}

\eof
\name{aafTable}
\alias{aafTable}
\title{Constructor for aafTable objects}
\description{
  Constructs an \code{aafTable} object given vectors, lists, or
  \code{aafList} objects.
}

\usage{
aafTable(..., items = list(...), colnames = names(items), 
         probeids = character(0), signed = FALSE)
}
\arguments{
  \item{...}{named arguments, one for each column}
  \item{items}{alternatively a named list of the items to be put in the table}
  \item{colnames}{character vector of column names}
  \item{probeids}{character vector of probe ids associated with each row}
  \item{signed}{boolean, should each column be colored based on the sign?}
}

\value{
  An \code{aafTable} object.
}

\author{Colin A. Smith, \email{annaffy@colinsmith.org}}

\seealso{\link{aafTable-class}}

\keyword{ file }

\eof
\name{aafTableAnn}
\alias{aafTableAnn}
\title{Constructor for aafTable objects from annotation data}
\description{
  Constructs an \code{aafTable} object given a set of probe ids
  and desired annotation types.
}

\usage{
aafTableAnn(probeids, chip, colnames = .aaf.handler(), widget = FALSE)
}
\arguments{
  \item{probeids}{character vector of probe ids}
  \item{chip}{name of the data package in which the annotation data is stored}
  \item{colnames}{character vector of annotation types}
  \item{widget}{boolean, use widget to select columns?}
}

\value{
  An \code{aafTable} object.
}

\author{Colin A. Smith, \email{annaffy@colinsmith.org}}

\seealso{\link{aafTable-class}}

\keyword{ file }

\eof
\name{aafTableFrame}
\alias{aafTableFrame}
\title{Constructor for aafTable objects from data frames}
\description{
  Constructs an \code{aafTable} object given a data frame.
}

\usage{
aafTableFrame(frame, colnames = names(df), probeids = row.names(frame), 
              signed = FALSE)
}
\arguments{
  \item{frame}{data frame to be converted to the table}
  \item{colnames}{character vector of column names}
  \item{probeids}{character vector of probe ids associated with each row}
  \item{signed}{boolean, should each column be colored based on the sign?}
}

\value{
  An \code{aafTable} object.
}

\author{Colin A. Smith, \email{annaffy@colinsmith.org}}

\seealso{\link{aafTable-class}}

\keyword{ file }

\eof
\name{aafTableInt}
\alias{aafTableInt}
\title{Constructor for aafTable objects from exprSets}
\description{
  Constructs an \code{aafTable} object containing expression values
  given an exprSet.

  In the resulting HTML table, the expression values will have
  backgrounds with varrying intensities of green depending on the
  expression measure.
}

\usage{
aafTableInt(exprSet, colnames = sampleNames(exprSet), 
            probeids = geneNames(exprSet))
}
\arguments{
  \item{exprSet}{object of class \code{exprSet}}
  \item{colnames}{character vector of column names}
  \item{probeids}{character vector of probe ids associated with each row}
}

\value{
  An \code{aafTable} object.
}

\author{Colin A. Smith, \email{annaffy@colinsmith.org}}

\seealso{
  \link{aafTable-class},
  \link{aafIntensity}
}

\keyword{ file }

\eof
\name{aafUniGene-class}
\docType{class}
\alias{aafUniGene-class}
\alias{getURL,aafUniGene-method}
\title{Class aafUniGene, a class for UniGene cluster ids}
\description{
  An abstraction for UniGene cluster ids from Bioconductor data packages.
}

\section{Objects from the Class}{
Objects are generally created by the \code{\link{aafUniGene}}
constructor.  Objects can also be created manually by calls of the
form \code{new("aafUniGene", id)}.
}
\section{Slots}{
  \describe{
    \item{\code{.Data}:}{Object of class \code{character}}
  }
}
\section{Extends}{
Class \code{character}, from data part.
}

\section{Methods}{
  \describe{
    \item{getURL}{
      (aafUniGene): Returns a URL to the corresponding entry in
      NCBI's UniGene database. On the rare chance that more than
      one id is defined, more than one URL will be returned.
    }
  }
}

\author{Colin A. Smith, \email{annaffy@colinsmith.org}}

\seealso{\link{aafUniGene}}

\keyword{classes}

\eof
\name{aafUniGene}
\alias{aafUniGene}
\title{Constructor for aafUniGene objects}
\description{
  For the given \code{probeids}, constructs a list of
  \code{aafUniGene} objects containing annotation data from the
  \code{chip} data package.
}

\usage{
aafUniGene(probeids, chip)
}
\arguments{
  \item{probeids}{character vector containing probe ids}
  \item{chip}{name of the chip data package}
}

\value{
  An \code{aafList} of \code{aafUniGene} objects. NA values are returned as
  empty objects.
}

\author{Colin A. Smith, \email{annaffy@colinsmith.org}}

\examples{
probes <- ls(annaffyUNIGENE)
ugs <- aafUniGene(probes, "annaffy")
show(ugs[6:10])
}

\seealso{\link{aafUniGene-class}}

\keyword{ file }

\eof
\name{annaffyACCNUM}
\alias{annaffyACCNUM}
\title{Subset of annotation data file for ACCNUM in the hgu95av2 package}
\description{
This is an R environment object containing key and value pairs for the mappings between probe identifiers (key) and GenBank accession numbers (value). Keys can be accessed using ls(name) and values using get(key, name) or multiget (keys, name). Values are vectors of length 1. NA is assigned to probe identifiers that can not be mapped to a GenBank accession number. Mappings between probe ids and GenBank accession numbers are assumed to be provided by users. 
}
\details{
Source data built:
LocusLink built: May 15, 2003.\url{ftp://ftp.ncbi.nih.gov/refseq/LocusLink/LL\_tmpl.gz}.
Package built: Thu May 15 06:51:05 2003
Contains 250 randomly selected probes
}
\references{
\url{ftp://ftp.ncbi.nih.gov/refseq/LocusLink/LL_tmpl.gz}
}
\examples{
xx <- ls(env = annaffyACCNUM)
get(xx[1], env = annaffyACCNUM)
}
\keyword{datasets}


\eof
\name{annaffyCHR}
\alias{annaffyCHR}
\title{Subset of annotation data file for CHR in the hgu95av2 package}
\description{
This is an R environment object containing key and value pairs for the mappings between probe identifiers (key) and the chromosome the corresponding genes reside (value). Keys can be accessed using ls(name) and values using get(key, name) or multiget(keys, name). Values are vectors of length 1. NA is assigned to probe identifiers that can not be mapped to any chromosome at this time. Mappings are obtained using data from LocusLink (www.ncbi.nlm.nih.gov/LocusLink) based on unified mappings between probe identifiers and LocusLink ids.
}
\details{
Source data built:
LocusLink built: May 15, 2003.\url{ftp://ftp.ncbi.nih.gov/refseq/LocusLink/LL\_tmpl.gz}.
Package built: Thu May 15 06:51:06 2003
Contains 250 randomly selected probes
}
\references{
\url{ftp://ftp.ncbi.nih.gov/refseq/LocusLink/LL_tmpl.gz}
}
\examples{
xx <- ls(env = annaffyCHR)
get(xx[1], env = annaffyCHR)
}
\keyword{datasets}


\eof
\name{annaffyCHRLOC}
\alias{annaffyCHRLOC}
\title{Subset of annotation data file for CHRLOC in the hgu95av2 package}
\description{
This is an R environment object containing key and value pairs for the mappings between probe identifiers (key) and chromosomal location of genes (value). Keys can be accessed using ls(name) and values using get(key, name) or multiget(keys, name). Values are named vectors of signed integers of length 1 or more depending on whether a given probe id can be mapped to a single or multiple chromosomal locations. + and - signs are used to indicate the the strand of the chromosome. The names give the chromosome number of concern. NA is assigned to probe identifiers that can not be mapped to any chromosomal location data at this time. Mappings are obtained using data from LocusLink (www.ncbi.nlm.nih.gov/LocusLink) and Human Genome Project (http://genome.cse.ucsc.edu/goldenPath) based on unified mappings between probe identifiers and LocusLink ids and LocusLink ids and gene locations. 
}
\details{
Source data built:
Human Genome Project built: 10april2003.\url{http://www.genome.ucsc.edu/goldenPath/10april2003/database/}.
Package built: Thu May 15 06:51:05 2003
Contains 250 randomly selected probes
}
\references{
\url{http://www.genome.ucsc.edu/goldenPath/10april2003/database/}
}
\examples{
xx <- ls(env = annaffyCHRLOC)
get(xx[1], env = annaffyCHRLOC)
}
\keyword{datasets}


\eof
\name{annaffyENZYME}
\alias{annaffyENZYME}
\title{Subset of annotation data file for ENZYME in the hgu95av2 package}
\description{
This is an R environment object containing key and value pairs for the mappings between probe identifiers (key) and Enzyme Commission numbers (value) with the leading EC: portion omitted. Keys can be accessed using ls(name) and values using get(key, name)
}
\details{
Source data built:
KEGG built: Release 26.0 (April 2003).\url{ftp://ftp.genome.ad.jp/pub/kegg/pathways}.
Package built: Thu May 15 06:51:08 2003
Contains 250 randomly selected probes
}
\references{
\url{ftp://ftp.genome.ad.jp/pub/kegg/pathways}
}
\examples{
xx <- ls(env = annaffyENZYME)
get(xx[1], env = annaffyENZYME)
}
\keyword{datasets}


\eof
\name{annaffyGENENAME}
\alias{annaffyGENENAME}
\title{Subset of annotation data file for GENENAME in the hgu95av2 package}
\description{
This is an R environment object containing key and value pairs for the mappings between probe identifiers (key) and gene names (value), which are  descriptions used for gene reports. Keys can be accessed using ls(name) and values using get(key, name) or multiget(keys, name). Values are vectors of  length 1. NA is assigned to probe identifiers that can not be mapped to any gene name at this time. Mappings are obtained using data from LocusLink (www.ncbi.nlm.nih.gov/LocusLink) based on unified mappings between probe identifiers and LocusLink ids.
}
\details{
Source data built:
LocusLink built: May 15, 2003.\url{ftp://ftp.ncbi.nih.gov/refseq/LocusLink/LL\_tmpl.gz}.
Package built: Thu May 15 06:51:08 2003
Contains 250 randomly selected probes
}
\references{
\url{ftp://ftp.ncbi.nih.gov/refseq/LocusLink/LL_tmpl.gz}
}
\examples{
xx <- ls(env = annaffyGENENAME)
get(xx[1], env = annaffyGENENAME)
}
\keyword{datasets}


\eof
\name{annaffyGO}
\alias{annaffyGO}
\title{Subset of annotation data file for GO in the hgu95av2 package}
\description{
This is an R environment object containing key and value pairs for the mappings between probe identifiers (key) and Gene Ontology identifiers (value) defined by Gene Ontology consortium. Keys can be accessed using ls(name) and values using get(key, name) or multiget(keys, name). Values are named vectors of length 1 or greater depending on whether a given probe identifier can be mapped to only one or more Gene Ontology ids. Names for values are the evidence codes for the GO ids. NA is assigned to probe identifiers that can not be mapped to any Gene Ontology id at this time. Mappings between Gene Ontology ids an Gene Ontology terms and other information are available in a separate data package named GO. Above mappings are obtained using data from LocusLink (www.ncbi.nlm.nih.gov/LocusLink) based on unified mappings between probe identifiers and LocusLink ids.
}
\details{
Source data built:
LocusLink built: May 15, 2003.\url{ftp://ftp.ncbi.nih.gov/refseq/LocusLink/LL\_tmpl.gz}.
Package built: Thu May 15 06:51:08 2003
Contains 250 randomly selected probes
}
\references{
\url{ftp://ftp.ncbi.nih.gov/refseq/LocusLink/LL_tmpl.gz}
}
\examples{
xx <- ls(env = annaffyGO)
get(xx[1], env = annaffyGO)
}
\keyword{datasets}


\eof
\name{annaffyLOCUSID}
\alias{annaffyLOCUSID}
\title{Subset of annotation data file for LOCUSID in the hgu95av2 package}
\description{
This is an R environment object containing key and value pairs for the mappings between  probe identifiers (key) and LocusLink ids (values). Keys can be accessed using ls(name) and values using get(key, name) multiget(keys, name). Values may be vectors of length 1 or greater depending on whether a given probe identifier can be mapped to only one or more LocusLink ids. NA is assigned to probe identifiers that can not be mapped to a LocusLink id at this time. Mappings between probe identifiers and LocusLink ids are obtained collectively using data from UniGene (\url{www.ncbi.nlm.nih.gov/UniGene}) and LocusLink (\url{www.ncbi.nlm.nih.gov/LocusLink}).
}
\details{
Source data built:
LocusLink built: May 15, 2003.\url{ftp://ftp.ncbi.nih.gov/refseq/LocusLink/LL\_tmpl.gz}.
Package built: Thu May 15 06:51:09 2003
Contains 250 randomly selected probes
}
\references{
\url{ftp://ftp.ncbi.nih.gov/refseq/LocusLink/LL_tmpl.gz}
}
\examples{
xx <- ls(env = annaffyLOCUSID)
get(xx[1], env = annaffyLOCUSID)
}
\keyword{datasets}


\eof
\name{annaffyMAP}
\alias{annaffyMAP}
\title{Subset of annotation data file for MAP in the hgu95av2 package}
\description{
This is an R environment object containing key and value pairs for the mappings between probe identifiers (key) and cytoband locations (value). Keys can be accessed using ls(name) and values using get(key, name) or multiget(keys, name). Values are vectors of length 1. NA is assigned to probe identifiers that can not be mapped to any cytoband at this time. Mappings are obtained using data from LocusLink (www.ncbi.nlm.nih.gov/LocusLink) based on unified mappings between probe identifiers and LocusLink ids.
}
\details{
Source data built:
LocusLink built: May 15, 2003.\url{ftp://ftp.ncbi.nih.gov/refseq/LocusLink/LL\_tmpl.gz}.
Package built: Thu May 15 06:51:09 2003
Contains 250 randomly selected probes
}
\references{
\url{ftp://ftp.ncbi.nih.gov/refseq/LocusLink/LL_tmpl.gz}
}
\examples{
xx <- ls(env = annaffyMAP)
get(xx[1], env = annaffyMAP)
}
\keyword{datasets}


\eof
\name{annaffyPATH}
\alias{annaffyPATH}
\title{Subset of annotation data file for PATH in the hgu95av2 package}
\description{
This is an R environment object containing key and value pairs for the mappings between probe identifiers (key) and pathway identifiers defined by KEGG (value). Keys can be accessed using ls(name) and values using get(key, name) or multiget(keys, name). Values may be vectors of length 1 or greater depending on whether a given probe identifier can be associated with only one or more pathways. NA is assigned to probe identifiers that can not be  associated with any pathway at this time. Mappings between pathway ids to pathway names can be found in a separate data package named KEGG. Above mappings are obtained using data from LocusLink (www.ncbi.nlm.nih.gov/LocusLink) and KEGG (www.genome.ad.jp/kegg).
}
\details{
Source data built:
KEGG built: Release 26.0 (April 2003).\url{ftp://ftp.genome.ad.jp/pub/kegg/pathways}.
Package built: Thu May 15 06:51:11 2003
Contains 250 randomly selected probes
}
\references{
\url{ftp://ftp.genome.ad.jp/pub/kegg/pathways}
}
\examples{
xx <- ls(env = annaffyPATH)
get(xx[1], env = annaffyPATH)
}
\keyword{datasets}


\eof
\name{annaffyPMID}
\alias{annaffyPMID}
\title{Subset of annotation data file for PMID in the hgu95av2 package}
\description{
This is an R environment object containing key and value pairs for the mappings between probe identifiers (key) and PubMed ids (value), which are subsets of publications associated with the underlying genes. Keys can be accessed using ls(name) and values using get(key, name) or multiget(keys, name). Values may be vectors of length 1 or greater depending on whether a given probe identifier can be mapped to only one or more publications. NA is assigned to probe identifiers that can not be mapped to any publication at this time. Mappings are obtained using data from LocusLink (www.ncbi.nlm.nih.gov/LocusLink) based on unified mappings between probe identifiers and LocusLink ids.
}
\details{
Source data built:
LocusLink built: May 15, 2003.\url{ftp://ftp.ncbi.nih.gov/refseq/LocusLink/LL\_tmpl.gz}.
Package built: Thu May 15 06:51:11 2003
Contains 250 randomly selected probes
}
\references{
\url{ftp://ftp.ncbi.nih.gov/refseq/LocusLink/LL_tmpl.gz}
}
\examples{
xx <- ls(env = annaffyPMID)
get(xx[1], env = annaffyPMID)
}
\keyword{datasets}


\eof
\name{annaffySUMFUNC}
\alias{annaffySUMFUNC}
\title{Subset of annotation data file for SUMFUNC in the hgu95av2 package}
\description{
This is an R environment object containing key and value pairs for the mappings between probe identifiers (key) and summaries of the function (value), which are brief descriptions of the function of the products of genes. Keys can be accessed using ls(name) and values using get(key, name) or multiget(keys, name). Values are vectors of length 1. NA is assigned to probe identifiers that can not be mapped to any summary of function at this time.  
}
\details{
Mappings were based on data provided by LocusLink

Source data built:
LocusLink built: June 23, 2003.\url{ftp://ftp.ncbi.nih.gov/refseq/LocusLink/LL\_tmpl.gz}.
Package built: Thu Jun 26 05:23:11 2003
}
\references{
\url{ftp://ftp.ncbi.nih.gov/refseq/LocusLink/LL_tmpl.gz}
}
\examples{
xx <- ls(env = annaffySUMFUNC)
get(xx[1], env = annaffySUMFUNC)
}
\keyword{datasets}


\eof
\name{annaffySYMBOL}
\alias{annaffySYMBOL}
\title{Subset of annotation data file for SYMBOL in the hgu95av2 package}
\description{
This is an R environment object containing key and value pairs for the mappings between probe identifiers (key) and gene symbols (value), which are symbols used to report genes. Keys can be accessed using ls(name) and values using get(key, name) or multiget(keys, name). Values are vectors of length 1. NA is assigned to probe identifiers that can not be mapped to any gene symbol at this time. Mappings are obtained using data from LocusLink (www.ncbi.nlm.nih.gov/LocusLink) based on unified mappings between probe identifiers and LocusLink ids.
}
\details{
Source data built:
LocusLink built: May 15, 2003.\url{ftp://ftp.ncbi.nih.gov/refseq/LocusLink/LL\_tmpl.gz}.
Package built: Thu May 15 06:51:12 2003
Contains 250 randomly selected probes
}
\references{
\url{ftp://ftp.ncbi.nih.gov/refseq/LocusLink/LL_tmpl.gz}
}
\examples{
xx <- ls(env = annaffySYMBOL)
get(xx[1], env = annaffySYMBOL)
}
\keyword{datasets}


\eof
\name{annaffyUNIGENE}
\alias{annaffyUNIGENE}
\title{Subset of annotation data file for UNIGENE in the hgu95av2 package}
\description{
This is an R environment object containing key and value pairs for the mappings between probe identifiers (key) and UniGene cluster ids (value). Keys can be accessed using ls(name) and values using get(key, name) or multiget(keys, name) . Values may be vectors of length 1 or greater depending on whether a given probe identifier can be mapped to only one or more UniGene cluster ids. NA is assigned to probe identifiers that can not be mapped to any UniGene cluster id at this time. Mappings are obtained using data from LocusLink (www.ncbi.nlm.nih.gov/LocusLink) based on unified mappings between probe identifiers and LocusLink ids.
}
\details{
Source data built:
LocusLink built: May 15, 2003.\url{ftp://ftp.ncbi.nih.gov/refseq/LocusLink/LL\_tmpl.gz}.
Package built: Thu May 15 06:51:12 2003
Contains 250 randomly selected probes
}
\references{
\url{ftp://ftp.ncbi.nih.gov/refseq/LocusLink/LL_tmpl.gz}
}
\examples{
xx <- ls(env = annaffyUNIGENE)
get(xx[1], env = annaffyUNIGENE)
}
\keyword{datasets}


\eof
\name{getCSS-methods}
\docType{methods}
\title{Methods for function getCSS}
\alias{getCSS-methods}
\alias{getCSS}
\alias{getCSS,ANY-method}
\description{
  Methods to get relevant stylesheet lines for an object.
}

\section{Methods}{\describe{
\item{object = ANY}{Returns an empty character vector.}
}}

\note{
  For information about other implementations of this method, 
  see documentation of the respective class.  
}

\seealso{
  \link{aafList-class},
  \link{aafPubMed-class},
  \link{aafGO-class},
  \link{aafPathway-class},
  \link{aafSigned-class}
}

\keyword{methods}

\eof
\name{getHTML-methods}
\docType{methods}
\title{Methods for function getHTML}
\alias{getHTML-methods}
\alias{getHTML}
\alias{getHTML,ANY-method}
\description{
  Methods to get an HTML representation of an object.
}

\section{Methods}{
  \describe{
    \item{object = ANY}{
      Returns text of \code{object} along with URL link if applicable.
      If \code{object} is floating point, it displays a fixed number
      of significant digits as specified by the \code{sigfigs}
      option (default 6).
    }
  }
}

\note{
  For information about other implementations of this method, 
  see documentation of the respective class.  
}

\seealso{
  \link{aafList-class},
  \link{aafPubMed-class},
  \link{aafGO-class},
  \link{aafGOItem-class},
  \link{aafPathway-class},
  \link{aafPathwayItem-class}
}

\keyword{methods}

\eof
\name{getTD-methods}
\docType{methods}
\title{Methods for function getTD}
\alias{getTD-methods}
\alias{getTD}
\alias{getTD,ANY-method}
\description{
  Methods to get an HTML table cell representation of an object.
}

\section{Methods}{\describe{
\item{object = ANY}{Returns <td> tag containing HTML representation of \code{object}. Sets class attribute to \code{class(object)}. }
}}

\note{
  For information about other implementations of this method, 
  see documentation of the respective class.  
}

\seealso{
  \link{aafList-class},
  \link{aafGO-class},
  \link{aafPathway-class},
  \link{aafIntensity-class}
}

\keyword{methods}

\eof
\name{getText-methods}
\docType{methods}
\title{Methods for function getText}
\alias{getText-methods}
\alias{getText}
\alias{getText,ANY-method}
\description{
  Methods to get a textual representation of an object.
}

\section{Methods}{\describe{
\item{object = ANY}{Returns a comma delimeted list of the elements in \code{list}.}
}}

\note{
  For information about other implementations of this method, 
  see documentation of the respective class.  
}

\seealso{
  \link{aafList-class},
  \link{aafCytoband-class},
  \link{aafGO-class},
  \link{aafGOItem-class},
  \link{aafPathway-class},
  \link{aafPathwayItem-class}
}

\keyword{methods}

\eof
\name{getURL-methods}
\docType{methods}
\title{Methods for function getURL}
\alias{getURL-methods}
\alias{getURL}
\alias{getURL,ANY-method}
\description{
  Methods to get a URL link to a web resource for an object.
}

\section{Methods}{\describe{
\item{object = ANY}{Returns an empty character vector.}
}}

\note{
  For information about other implementations of this method, 
  see documentation of the respective class.  
}

\seealso{
  \link{aafList-class},
  \link{aafGenBank-class},
  \link{aafLocusLink-class},
  \link{aafCytoband-class},
  \link{aafUniGene-class},
  \link{aafPubMed-class},
  \link{aafGO-class},
  \link{aafGOItem-class},
  \link{aafPathwayItem-class}
}

\keyword{methods}

\eof
\name{selectorWidget}
\alias{selectorWidget}
\title{Dialog to select items from a list}
\description{
  Presents the user with a dialog box to select items from a list.
}

\usage{
selectorWidget(options, selected = character(0), title = "Selector",
               ordersel = FALSE, ordernsel = FALSE, 
               height = max(10,min(20,length(options))))
}
\arguments{
  \item{options}{vector, options to be selected from}
  \item{selected}{vector, subset of options selected by default}
  \item{title}{character scalar, window title}
  \item{ordersel}{boolean, keep the selected items in order?}
  \item{ordernsel}{boolean, keep the not selected items in order?}
  \item{height}{scalar, height of the two listboxes}
}
\value{
  A character vector containing the selected items. If a vector of
  a different class was initially provided, it must be manually
  coerced back to the correct type.
}
\author{Colin A. Smith, \email{annaffy@colinsmith.org}}
\keyword{ file }

\eof
