.packageName <- "Biobase"
# Functions that add vignetts or other elements to the menu bar of a window.
#
# Copyright 2002 J. Zhang, all rights reserved
#

addVig2Menu <- function(itemName, menuName = "Vignettes", itemAction = ""){
    os <- .Platform$OS.type
    switch(os,
           "windows" = addVig4Win(menuName, itemName,itemAction ),
           "unix" = addVig4Unix(menuName, itemName, itemAction),
           stop("Unknown operating system"))

}
# Add menu for windows
addVig4Win <- function(menuName, itemName, itemAction){
    options(show.error.messages = FALSE)
    tryMe <- try(winMenuAddItem(menuName, itemName, itemAction))
    options(show.error.messages = TRUE)
    if(inherits(tryMe, "try-error")){
        # Menu does not exist for the item. Add menus
        addNonExisting(menuName)
        winMenuAddItem(menuName, itemName, itemAction)
    }
}

# Add menu for a window in Unix
addVig4Unix <- function(menuName, itemName, itemAction){
    # "Do not know what to do yet"
}

# Find and add all the non-existing menu elelments
addNonExisting <- function(menuName){
    temp <- menuName
    menus <- unlist(strsplit(menuName, "/"))
    counter <- 1

    # Find and add the first missing menu along the menu tree
    repeat{
        options(show.error.messages = FALSE)
        tryMe <- try(winMenuAdd(temp))
        options(show.error.messages = TRUE)
        if(inherits(tryMe, "try-error")){
            temp <- paste(menus[1:(length(menus) - counter)], sep = "",
                          collapse = "/")
            counter <- counter + 1
        }else{
            break
        }
    }
    # Add the rest menus
    if(counter > 1){
        for(i in ((length(menus) - counter + 2):length(menus))){
            temp <-  paste(menus[1:i], sep = "", collapse = "/")
            winMenuAdd(temp)
        }
    }
}

# Add click-able menu items to view the pdf files of a package
addPDF2Vig <- function(pkgName){
    pdfs <- getPkgVigs(pkgName)
    for(i in pdfs){
        item <- sub(".pdf", "", basename(i))
        addVig2Menu(item, menuName = paste("Vignettes/", pkgName, sep = ""),
                    itemAction = paste("shell.exec(\"",
                    as.character(i), "\")", sep = ""))
    }
}
##Copyright R. Gentleman, 2001
##All rights reserved

##a simple aggregator
##data are aggregated in the environment env
##if they are not there then the get assigned with
##initfun, if they are there they get aggregated with
##agfun

Aggregate <- function(x, agg)
{
    if( !inherits(agg, "aggregator") )
        stop("second argument must be an aggregator")
    if( is.null(x) || length(x) == 0 )
        return()
    if(is.character(x)) {
        for( i in 1:length(x) ) {
            nm <- x[i]
            if( !exists(nm, env=aggenv(agg), inherits=FALSE) )
                assign(nm, env=aggenv(agg), initfun(agg)(nm, x))
            else {
                v1 <- get(nm, env=aggenv(agg))
                assign(nm, aggfun(agg)(nm, v1), env=aggenv(agg))
            }
        }
    }
    else if(is.list(x)) {
        nms <- names(x)
        for( i in 1:length(x) ) {
            nm <- nms[i]
            if( !exists(nm, env=aggenv(agg), inherits=FALSE) )
                assign(nm, env=aggenv(agg), initfun(agg)(nm, x[[i]]))
            else {
                v1 <- get(nm, env=aggenv(agg))
                assign(nm, env=aggenv(agg), aggfun(agg)(nm, v1, x[[i]]))
            }
        }
    }
    else stop("bad type for Aggregate")
}

    setClass("aggregator", representation( aggenv = "environment",
                                          initfun = "function",
                                          aggfun = "function"),
             prototype = list(aggenv = new.env(hash=TRUE), initfun =
             function(name, val) 1,
             aggfun =  function(name, current, val) current+1 ))


    if( !isGeneric("aggenv") )
        setGeneric("aggenv", function(object) standardGeneric("aggenv"))

    setMethod("aggenv", "aggregator", function(object) object@aggenv)

    if( !isGeneric("initfun") )
        setGeneric("initfun", function(object) standardGeneric("initfun"))

    setMethod("initfun", "aggregator", function(object) object@initfun)

    if( !isGeneric("aggfun") )
        setGeneric("aggfun", function(object) standardGeneric("aggfun"))

    setMethod("aggfun", "aggregator", function(object) object@aggfun)

as.data.frame.exprSet <- function(x, row.names=NA, optional=NA) {
  nc.eset <- ncol(exprs(x))
  nr.eset <- nrow(exprs(x))
  gn.eset <- geneNames(x)

  if (is.null(gn.eset))
    gn.eset <- rep(NA, nr.eset)
  i.pdata <- rep(seq(1, nc.eset), rep(nr.eset, nc.eset))

  pexp <- c(exprs(x))
  
  
  rv <- do.call("data.frame", c(list(exprs=pexp, genenames=rep(gn.eset, nc.eset)),
                                lapply(pData(x), function(y, i.pdata) y[i.pdata], i.pdata))
                )
  
  return(rv)
}
#Copyright 2001, R. Gentleman, all rights reserved.

#A simple class structure for containers
#These are simply lists where the list can contain only
#objects of the specified class

setClass("container", representation( x = "list", content =
                                     "character", locked = "logical"),
         prototype = list(x=vector("list", 0), content="object",
                                     locked=FALSE),)

if( !isGeneric("content") )
    setGeneric("content", function(object)
 standardGeneric("content"))

setMethod("content", "container", function(object) object@content)

if( !isGeneric("locked") )
    setGeneric("locked", function(object)
 standardGeneric("locked"))

setMethod("locked", "container", function(object) object@locked)

setReplaceMethod("[[", "container", function(x, i, j,..., value) {
    if( locked(x) )
        stop("cannot assign into a locked container")
    cv <- class(value)
    cont <- content(x)
    if( !extends(cv, cont) )
        stop(paste("the container is class", cont,
                   "the object is class", cv, "cannot assign",
                   sep=" "))
    x@x[[i]] <- value
    x
})

setMethod("[[", "container", function(x, i, j, ...) {
    x@x[[i]]
})

setMethod("length", "container", function(x) length(x@x))

setMethod("show", "container", function(object) {
    cat("Container of ", content(object), "\n", sep="")
    print(object@x)
})

setMethod("[", "container",
    def = function(x, i, j, ..., drop = F){
      new("container", content = content(x), x = x@x[i],
          locked = locked(x))
})
copySubstitute = function(src, dest, symbolValues,
                          symbolDelimiter = "@",
                          allowUnresolvedSymbols = FALSE,
                          recursive = FALSE,
                          removeExtension = "\\.in$") {
  ## check integrity of arguments (...lots of bureaucracy)
  mess = NULL
  if (!is.list(symbolValues) && !is.vector(symbolValues))
    mess = "'symbolValues' must be a list or vector."
  if (!all(sapply(symbolValues, is.character)))
    mess = "'symbolValues' must only contain characters."
  if (is.null(names(symbolValues)) || any(names(symbolValues)==""))
    mess = "'symbolValues' must have non-empty names."
  if (!(is.character(symbolDelimiter) && length(symbolDelimiter)==1 && all(nchar(symbolDelimiter)==1)))
    mess = "'symbolDelimiter' must be a single character."
  if (!is.logical(allowUnresolvedSymbols))
    mess = "'allowUnresolvedSymbols' must be of type logical."
  if(!is.null(mess))
    stop(mess)

  ##----------------------------------------------------------------------
  ## Here the actual subsitution and copying work is done
  ## cin and cout are single files or connections
  ##----------------------------------------------------------------------
  cpSubsCon = function(cin, cout) {
    txt = readLines(cin)
    for (i in seq(along=symbolValues))
      txt = gsub(nm[i], symbolValues[[i]], txt)

    ## check for unresolved symbols
    if(!allowUnresolvedSymbols){
      re = regexpr(paste(symbolDelimiter, ".+", symbolDelimiter, sep=""), txt)
      wh = which(re>0)
      if(length(wh)>0) {
        ml   = attr(re, "match.length")
        mess = "UNRESOLVED SYMBOLS:\n"
        mess = paste(mess, paste(sapply(wh, function(i)
          paste("Line", i, ":", substr(txt[i], re[i], re[i] + ml[i]))), collapse="\n"),
          sep="")
        stop(mess)
      }
    }
    ## finito
    writeLines(text=txt, con=cout)
  }

  ##------------------------------------------------------------
  ## Substitution on filenames
  ##------------------------------------------------------------
  subsFileName = function(x) {
    res = gsub(removeExtension, "", x)
    for (i in seq(along=symbolValues))
      res = gsub(nm[i], symbolValues[[i]], res)
    return(res)
  }
  
  ##----------------------------------------------------------------------
  ## Iterate over character vectors of filenames and
  ## recursively descend into directories
  ##----------------------------------------------------------------------
  cpSubs = function(src, dest) {
    usage = paste("\n* Usage:",
      "\n* with recursive=FALSE:",
      "\n* 'src' a connection open for reading or a file name AND",
      "\n* 'dest' a connection open for writing or a file name",
      "\n* with recursive=TRUE:",
      "\n* 'src' a vector of file and directory names and 'dest' a directory name\n\n")
    if (!recursive) {
      ## {file,connection} to {file,connection}
      if ((("connection" %in% class(src) && isOpen(src, rw="r")) ||
           (is.character(src) && length(src)==1)) &&
          (("connection" %in% class(dest) && isOpen(dest, rw="w")) ||
           (is.character(dest) && length(dest)==1))) {
 
        if(is.character(dest))
          dest = subsFileName(dest)
        cpSubsCon(src, dest)
        return(invisible(NULL))
      } 
    } else {
      ## recursive: file(s) and/or directory(ies) to directory
      if (is.character(src) && is.character(dest) && length(dest==1)) {
        ## if 'dest' does not exist, create
        if (file.access(dest) != 0) {
          if (!dir.create(dest))
            stop(paste("'dest' does not exist, and it cannot be created:", dest))
        }
        ## process src
        isdir = file.info(src)$isdir
        for (k in seq(along=src)) {
          ## name of source file or directory (without path)
          tmp  = unlist(strsplit(src[k], .Platform$file.sep))
          tmp  = subsFileName(tmp[length(tmp)])
          ## name of destination file or directory (with path)
          destname = file.path(dest, tmp)
          if (isdir[k]) {
            if(!dir.create(destname))
              stop(paste("directory cannot be created:", destname))
            cpSubs(dir(src[k], full.names=TRUE), destname)
          } else {
            cpSubsCon(src[k], destname)
          }
        } ## for k
        return(invisible(NULL))
      }
    } ## if(recursive)else 
    stop(usage)
  } ## cpSubs

  ##------------------------------------------------------------
  ## Do it!
  ##------------------------------------------------------------
  nm  = paste(symbolDelimiter, names(symbolValues), symbolDelimiter, sep="")
  cpSubs(src, dest)
  
}

createPackage <- function(pkgname, destinationDir, originDir, symbolValues,
                          unlink=FALSE, quiet=FALSE)
{
  ## check arguments 
  for (a in c("pkgname", "destinationDir", "originDir"))
    if (!is.character(get(a)) || length(get(a))!=1)
      stop(paste("'", a, "' must be a character vector of length 1.", sep=""))
  
  ## check whether destinationDir, originDir exist and are directories
  for (a in c("destinationDir", "originDir"))
    if(!file.info(get(a))$isdir)
      stop(paste("'", a, "' must be a directory (", get(a), ")\n.", sep=""))

  ## locate / remove / create destination directory
  pkgdir = file.path(destinationDir, pkgname)
  if (!quiet)
    cat("Creating package in", pkgdir, "\n")
  
  if (file.exists(pkgdir)) {
    if (unlink) {
      unlink(pkgdir, recursive=TRUE)
      if (file.exists(pkgdir)) {
        stop(paste("Directory", pkgdir, "exists and could not be removed.",
                   "Please remove it manually or choose another destination directory."))
      } else {
        if(!quiet)
          cat(paste("Existing", pkgdir, "was removed.\n"))
      }
    } else {
      stop(paste("Directory", pkgdir, "exists. Please use unlink=TRUE to remove it",
                 "or choose another destination directory."))
    } ## if (unlink) else 
  }  ## if (file.exists)

  ## predefined symbols
  symbolValues = append(symbolValues, list(TODAY=date(), PKGNAME=pkgname))

  ## copy
  copySubstitute(dir(originDir, full.names=TRUE), pkgdir, symbolValues, recursive=TRUE)
    
  return(list(pkgdir=pkgdir))
}

env2list <- function(envir, recurse=FALSE) {
    if (!is.environment(envir))
        stop("envir argument is not an environment")

    ans <- list()
    ## This is basically a lot like multiget() except there isn't
    ## need for checking that what you're looking for exists as
    ## you know it does
    vals <- ls(envir)
    for (i in seq(along=vals))
        ans[[i]] <- get(vals[i], envir=envir, inherits=FALSE)
    names(ans) <- vals

    if ((recurse == TRUE)&&(!is.null(parent.env(envir))))
        ans <- c(ans, list2env(parent.env(envir), recurse))

    ans
}

list2env <- function(vals, envir, recurse=FALSE) {
    if (!is.environment(envir))
        stop("envir argument is not an environment")

    if (!is.list(vals))
        stop("vals argument is not a list")

    len <- length(vals)
    if (len == 0)
        return(envir)

    names <- names(vals)
    if (length(names) != len)
        stop("Not a properly named list")

    ## !!! Does this work the way it is intended when recurse=TRUE?
    for (i in 1:len)
        assign(names[i], vals[[i]], envir=envir, inherits=recurse)

    envir
}

copyEnv <- function(oldEnv, newEnv=new.env(parent=parent.env(oldEnv)),
                    recurse=FALSE) {
    oldVals <- env2list(oldEnv, recurse)
    newEnv <- list2env(oldVals, newEnv)
    return(newEnv)
}
# A class for microarray data

# in this representation we think of the data set being comprised of
#  matrix slot exprs: a collection of array results organized as a
#  matrix, with genes defining rows and samples defining columns.
#  data.frame slot phenoData: in the customary organization of samples
# defining rows and variables or features defining columns.  thus
# if x is an exprSet, nrow(x@phenodata) == ncol(x@exprs)
#  * character slot description: unconstrained string with information about
# the exprSet

#
# it appears that the covariates slot is a list of labels
# giving details on elements of phenodata

# A class for MIAME information is also defined.
#we try to cover all 6 MIAME entries
# 1. Experimental design 2. Array design 3. Samples 4. Hybridizations
# 5. Measurements 6. Normalization controls
# the class will contain experimenter name, laboratory, contact
# information, a single-sentence experiment title,
#an abstract describing the experiment, and URLs  and slots that are lists for
#"Samples", "Hybridizations", "Normalization controls",
#the remaing two are already covered.
# we add a slot (also a list) to keep track of pre-processing
#
#more info: http://www.mged.org/Workgroups/MIAME/miame_1.1.html

##data class for accompanying data
  setClass("phenoData", representation(pData="data.frame",
                                       varLabels="list"),
           prototype=list(pData=data.frame(matrix(nr=0,nc=0)),
             varLabels=list()),
           validity =  function(object) {
                                dm <- dim(object@pData)
                                if(dm[2] != length(object@varLabels) )
                                   return(FALSE)
                                return(TRUE)
                       }
           )

  if( !isGeneric("pData") )
    setGeneric("pData", function(object) standardGeneric("pData"))
  setMethod("pData", "phenoData",
            function(object) object@pData)


  if( !isGeneric("varLabels") )
      setGeneric("varLabels", function(object)
      standardGeneric("varLabels"))

  setMethod("varLabels", "phenoData",
            function(object) object@varLabels)

  setMethod("[", "phenoData", function(x, i, j, ..., drop=FALSE) {
      if( missing(drop) ) drop<-FALSE
      vL <- varLabels(x)
      if( missing(j) ) {
          if( missing(i) )
              pD <- x@pData
          else
              pD <- x@pData[i, ,drop=FALSE]
     }
      else {
          vL <- vL[j]
          if(missing(i) )
              pD <- x@pData[,j,drop=drop]
         else
             pD <- x@pData[i, j,drop=FALSE]
      }
      new("phenoData", pData=pD, varLabels=vL)})

  setMethod("[[", "phenoData", function(x, i, j, ...)
      x@pData[[i]])

  setReplaceMethod("[[", "phenoData", function(x, i, j, ..., value) {
      x@pData[[i]] <- value
      x})

  setMethod("show", "phenoData",
            function(object) {
                dm <- dim(object@pData)
                cat("\t phenoData object with ", dm[2], " variables",
            sep="")
                cat(" and ", dm[1], " cases\n", sep="")
                vL <- object@varLabels
                cat("\t varLabels\n")
                nm <- names(vL)
                  for(i in seq(along=vL) )
                    cat("\t\t", nm[[i]], ": ", vL[[i]], "\n", sep="")
            })



##data class for MIAME information
  setClass("MIAME", representation(name="character",
                                   lab="character",
                                   contact="character",
                                   title="character",
                                   abstract="character",
                                   url="character",
                                   samples="list",
                                   hybridizations="list",
                                   normControls="list",
                                   preprocessing="list",
                                   other="list"),
           prototype=list(name="",lab="",contact="",
             title="",abstract="",url="",
             samples=list(),hybridizations=list(),normControls=list(),
             preprocessing=list(),other=list()))

  ##show method
  setMethod("show", "MIAME",
            function(object) {
              tmp <- c("samples","hybridizations","normalization controls","preprocessing")
              Index <-c(length(object@samples)>0,
                        length(object@hybridizations)>0,
                        length(object@normControls)>0,
                        length(object@preprocessing)>0)
              cat("Experimenter name:",object@name,"\n")
              cat("Laboratory:",object@lab,"\n")
              cat("Contact information:",object@contact,"\n")
              cat("Title:",object@title,"\n")
              cat("URL:",object@url,"\n")
              if(object@abstract!="")
                cat("\nA",length(strsplit(object@abstract," ")[[1]]),
                    "word abstract is available. Use 'abstract' method.\n")
              else cat("No abstract available.\n")
              if(any(Index)) cat("\nInformation is available on:",
                                 paste(tmp[Index],collapse=", "),"\n")
            })

  ##abstract method
  if( !isGeneric("abstract") )
    setGeneric("abstract", function(object)
               standardGeneric("abstract"))

  setMethod("abstract","MIAME",function(object) object@abstract)

   ##samples method
  if( !isGeneric("samples") )
    setGeneric("samples", function(object)
               standardGeneric("samples"))

  setMethod("samples","MIAME",function(object) object@samples)

   ##hybridizations method
  if( !isGeneric("hybridizations") )
    setGeneric("hybridizations", function(object)
               standardGeneric("hybridizations"))

  setMethod("hybridizations","MIAME",function(object) object@hybridizations)

   ##normControls method
  if( !isGeneric("normControls") )
    setGeneric("normControls", function(object)
               standardGeneric("normControls"))

  setMethod("normControls","MIAME",function(object) object@normControls)

   ##preproc method
  if( !isGeneric("preproc") )
    setGeneric("preproc", function(object)
               standardGeneric("preproc"))

  setMethod("preproc","MIAME",function(object) object@preprocessing)

  ##otherInfo method
  if( !isGeneric("otherInfo") )
    setGeneric("otherInfo", function(object)
               standardGeneric("otherInfo"))

  setMethod("otherInfo","MIAME",function(object) object@other)

  ##expinfo method
  if( !isGeneric("expinfo") )
    setGeneric("expinfo", function(object)
               standardGeneric("expinfo"))

  setMethod("expinfo","MIAME",function(object){
    tmp <- c(object@name,
             object@lab,
             object@contact,
             object@title,
             object@url)
    names(tmp) <- c("name","lab","contact","title","url")
    return(tmp)
  })

  ##trick so that the old exprSet and Plobs works
  setClassUnion("characterORMIAME", c("MIAME", "character"))


  ## Class union for the exprs and se.exprs slots of exprSet
  if (!isClass("exprMatrix"))
    setClassUnion("exprMatrix", c("matrix"))

  ##data class for expression arrays
  setClass("exprSet", representation(exprs="exprMatrix",
                                     se.exprs = "exprMatrix",
                                     phenoData="phenoData",
                                     description="characterORMIAME",
                                     annotation="character",
                                     notes="character") ,
           prototype=list(exprs=matrix(nr=0,nc=0),
             se.exprs = matrix(nr=0,nc=0),
             description=new("MIAME"),
             annotation="",
             notes=""))

  ##define a method to update exprsSet from previous versions
  if( !isGeneric("update2MIAME") )
    setGeneric("update2MIAME", function(object)
  standardGeneric("update2MIAME"))

  setMethod("update2MIAME", "exprSet",
            function(object){
              if(class(description(object))=="MIAME")
                cat("This object is up to date.\n")
              else{
                cat("For now we will keep old description in the experiment title.\nConsider defining an object of class MIAME with more information\n")
                description(object) <- new("MIAME",title=description(object))
              }
              object
            })

  ##define a generic for obtaining the data
  if( !isGeneric("exprs") )
    setGeneric("exprs", function(object) standardGeneric("exprs"))
  setMethod("exprs", "exprSet", function(object) object@exprs)

  ##RI: define a genric for obtaining the se's
  if( !isGeneric("se.exprs") )
    setGeneric("se.exprs", function(object) standardGeneric("se.exprs"))
  setMethod("se.exprs", "exprSet", function(object) object@se.exprs)

  if( !isGeneric("exprs<-") )
    setGeneric("exprs<-", function(object, value)
               standardGeneric("exprs<-"))

  setReplaceMethod("exprs", "exprSet",
                   function(object, value) {
                     object@exprs <- value
                     return(object)
                   })

  if( !isGeneric("se.exprs<-") )
    setGeneric("se.exprs<-", function(object, value)
               standardGeneric("se.exprs<-"))

  setReplaceMethod("se.exprs", "exprSet",
                   function(object, value) {
                     object@se.exprs <- value
                     return(object)
                   })

  ##RI: Added this so i can access notes
  ##method for notes (accessor and replacement)
  if( !isGeneric("notes") )
    setGeneric("notes", function(object)
               standardGeneric("notes"))
  setMethod("notes", "exprSet", function(object)
            object@notes)

  if( !isGeneric("notes<-") )
    setGeneric("notes<-", function(object, value)
               standardGeneric("notes<-"))

  setReplaceMethod("notes", "exprSet", function(object, value) {
    object@notes <- value
    object
  })

  ##method for MIAME description
  if( !isGeneric("description") )
    setGeneric("description", function(object)
               standardGeneric("description"))
  setMethod("description", "exprSet", function(object)
            object@description)

  ##replace method for description
  if( !isGeneric("description<-") )
    setGeneric("description<-", function(object, value)
               standardGeneric("description<-"))

  setReplaceMethod("description", "exprSet", function(object, value) {
    object@description <- value
    object
  })

  ##method for abstract
  setMethod("abstract", "exprSet",
            function(object) abstract(description(object)))

  ##method for phenoData
  if( !isGeneric("phenoData") )
    setGeneric("phenoData", function(object)
               standardGeneric("phenoData"))
  setMethod("phenoData", "exprSet", function(object)
            object@phenoData)

  if( !isGeneric("phenoData<-") )
    setGeneric("phenoData<-", function(object, value)
               standardGeneric("phenoData<-"))

  setReplaceMethod("phenoData", c("exprSet", "phenoData"),
                   function(object, value) {
                       object@phenoData <- value
                       object })

  ##method for pData
  setMethod("pData", "exprSet",
            function(object) pData(phenoData(object)))


  ##replace method for pData
  if( !isGeneric("pData<-") )
    setGeneric("pData<-", function(object, value)
               standardGeneric("pData<-"))

  setReplaceMethod("pData", "exprSet", function(object, value) {
    ph <- phenoData(object)
    pData(ph) <- value
    phenoData(object) <- ph
    object
  })

  ##[[ method
  setReplaceMethod("[[", "exprSet", function(x, i, j, ..., value) {
    pD <- phenoData(x)
    pD@pData[[i]] <- value
    phenoData(x) <- pD
    x})

  setMethod("[[", "exprSet", function(x, i, j, ...)
      phenoData(x)[[i]] )


###RI: this is a simple a pData replace for phenoData. i need it for affy.
  setReplaceMethod("pData", "phenoData", function(object, value){
    object@pData <- value
    object
  })


  if( !isGeneric("sampleNames") )
    setGeneric("sampleNames", function(object)
               standardGeneric("sampleNames"))
  setMethod("sampleNames", "exprSet",
            function(object) {
              if (! is.null(colnames(exprs(object))))
                colnames(exprs(object))
              else
                row.names(pData(object))
            })

  if( !isGeneric("geneNames") )
    setGeneric("geneNames", function(object)
               standardGeneric("geneNames"))
  setMethod("geneNames", "exprSet", function(object)
            row.names(exprs(object)))

  if( !isGeneric("geneNames<-") )
    setGeneric("geneNames<-", function(object, value)
               standardGeneric("geneNames<-"))

  setReplaceMethod("geneNames", "exprSet", function(object, value) {
    es <- exprs(object)
    row.names(es) <- value
    exprs(object) <- es
    object
  })

  ##a varLabels method for exprSets
  setMethod("varLabels", "exprSet",
            function(object) phenoData(object)@varLabels)

  ## annotation: read and replace. WH, 11 Mar 2003
  if(!isGeneric("annotation") )
    setGeneric("annotation", function(object)
               standardGeneric("annotation"))
  setMethod("annotation", "exprSet",
     definition = function(object) object@annotation)

  if(!isGeneric("annotation<-") )
    setGeneric("annotation<-", function(object, value)
               standardGeneric("annotation<-"))

  setReplaceMethod("annotation", signature="exprSet",
     definition =  function(object, value) {
                     object@annotation <- value
                     return(object)
                   })

  ## [
  setMethod("[", "exprSet", function(x, i, j, ..., drop=FALSE) {
    if( missing(j) )
	pdata <- phenoData(x)
    else
        pdata <- phenoData(x)[j,, ..., drop=FALSE]
    haveSES <- nrow(se.exprs(x)) > 0
    if(missing(j) ) {
        if( missing(i) ) {
            nexprs <- exprs(x)
            if( haveSES )
                nses <- se.exprs(x)
        }
        else {
            nexprs <- exprs(x)[i, ,drop=FALSE]
            if( haveSES )
                nses <- se.exprs(x)[i, ,drop=FALSE]
        }
    }
    else {
      if( missing(i) ) {
          nexprs <- exprs(x)[,j, drop=FALSE]
          if( haveSES )
              nses <- se.exprs(x)[,j, drop=FALSE]
      }
      else {
          nexprs <- exprs(x)[i, j, drop=FALSE]
          if( haveSES )
              nses <- se.exprs(x)[i, j, drop=FALSE]
      }
    }
    exprs(x) <- nexprs
    if( haveSES )
        se.exprs(x) <- nses
    phenoData(x) <- pdata
    x
  })

  setMethod("show", "exprSet", function(object ) {
    dm <-dim(exprs(object))
    ngenes <- dm[1]
    nsamples <- dm[2]
    cat("Expression Set (exprSet) with \n\t", ngenes, " genes\n\t", sep="")
    cat(nsamples, "samples\n\t")
    show(phenoData(object))
  })

  setGeneric("iter", function(object, covlab, f) standardGeneric("iter"))
                                        #
  setMethod("iter", signature(object="exprSet", covlab="missing",
                              f="function"),
            function(object, covlab, f)
            apply(exprs(object), 1, f))
                                        #
  setMethod("iter", signature(object="exprSet", covlab="missing",
                              f="list"),
            function(object,covlab,f) {
              flist <- f
              llen <- length(flist)
              out <- matrix(NA,nr=nrow(exprs(object)), nc=llen )
              lnames <- names(flist)
              if(is.null(lnames)) lnames <- paste("l",1:llen,sep="")
              for (i in 1:llen)
                out[,i] <- apply(exprs(object),1,flist[[i]])
              dimnames(out) <- list(row.names(exprs(object)),lnames)
              out
            })

  setMethod("iter", signature(object="exprSet", covlab="character",
                              f="function"),
            function(object, covlab, f) {
              ## f assumed to be a function of two arguments,
              ## first is a stratum identifier to be used
              ## in evaluating a statistical contrast by f
              varnames <- names(phenoData(object)@pData)
              if (!(any(match(covlab,varnames))))
                stop("the requested covariate is not in the exprSet")
              fc <- function(x) function(y) f(x,y)
              f2app <- fc(phenoData(object)@pData[[covlab]])
              iter(object,f=f2app)
            })


  ##split
  if( !isGeneric("split") )
    setGeneric("split")

  setMethod("split", signature(x="exprSet", f="vector"),
            function(x, f) {
              lenf <- length(f)
              exs <- exprs(x)
              pD <- phenoData(x)
              aN <- annotation(x)
                if( (nrow(exs) %% lenf == 0 ) ) {
                  splitexprs <- lapply(split(1:nrow(exs), f),
                                         function(ind) exs[ind, , drop =
                                                           FALSE])
                  nsplit<-length(splitexprs)
                  for(i in 1:nsplit) {
                    ## Create the new exprSet with the same class as
                    ## the original one - SDR
                    tmp <- x
                    exprs(tmp) <- splitexprs[[i]]
                    phenoData(tmp) <- pD
                    annotation(tmp) <- aN
                    se.exprs(tmp) <- matrix(nr=0,nc=0)
                    description(tmp) <- new("MIAME")
                    notes(tmp) <- ""
                    splitexprs[[i]] <- tmp
                    rm(tmp)
                  }
                  return(splitexprs)
                }  ##split the expressions
              if( (nrow(pData(x)) %% lenf ==0) ) {
                npD <- split(pD, f)
                nEx <- lapply(split(1:ncol(exs), f),
                              function(ind) exs[,ind,drop=FALSE])
                nsplit <- length(npD)
                for( i in 1:nsplit) {
                  ## Create the new exprSet with the same class as
                  ## the original one - SDR
                  tmp <- x
                  exprs(tmp) <- nEx[[i]]
                  phenoData(tmp) <- npD[[i]]
                  annotation(tmp) <- aN
                  npD[[i]] <- tmp
                  rm(tmp)
                }
                return(npD)
              }
              else
                stop("could not split")
            })


  setMethod("split", signature(x="phenoData", f="vector"),
            function(x, f) {
              lenf <- length(f)
              pD <- pData(x)
              vL <- varLabels(x)
              if( (nrow(pD) %% lenf ==0) ) {
                npD <- split(pD, f)
                nsplit <- length(npD)
                for( i in 1:nsplit)
                  npD[[i]] <- new("phenoData", pData = npD[[i]],
                                  varLabels=vL)
                return(npD)
              }
              else
                stop("could not split")
            })

###write table for exprSet. makes a table with gene names in first column
###chip names in first row
###apart from quote=FALSE and sep="\t"
###everything else is the same as write.table
  if( !isGeneric("write.exprs") )
    setGeneric("write.exprs", function(x,...) standardGeneric("write.exprs"))
  setMethod("write.exprs", signature(x="exprSet"),
            function(x,file = "tmp.txt",
                     append = FALSE, quote = FALSE, sep = "\t",
                     eol = "\n", na = "NA", dec = ".", row.names = TRUE,
                     col.names = TRUE, qmethod = c("escape", "double"))
            write.table(exprs(x),file = file, append = append,
                        quote = quote,
                        sep = sep,eol = eol, na = na, dec = dec,
                        row.names = row.names, col.names = col.names,
                        qmethod = qmethod))


  if( !isGeneric("exprs2excel") )
    setGeneric("exprs2excel", function(x,...) standardGeneric("exprs2excel"))
  setMethod("exprs2excel", signature(x="exprSet"),
            function(x,file = "tmp.csv",
                     append = FALSE, quote = FALSE, sep = ",",
                     eol = "\n", na = "NA", dec = ".", row.names = TRUE,
                     col.names = NA, qmethod = c("escape", "double"))
            write.table(exprs(x),file = file, append = append,
                        quote = quote,
                        sep = sep,eol = eol, na = na, dec = dec,
                        row.names = row.names, col.names = col.names,
                        qmethod = qmethod))



##not quite the right semantics
##but it is a start

"$.exprSet" <- function(eset, val)
    (pData(eset))[[as.character(val)]]

"$.phenoData" <- function(x, val, ...)
    (pData(x))[[as.character(val)]]

esApply <- function(X, MARGIN, FUN, ...) {
    if (class(X) != "exprSet") stop("arg1 must be of class exprSet")
    e1 <- new.env(parent=environment(FUN))
    multiassign(names(pData(X)), pData(X), env=e1)
    environment(FUN) <- e1
    apply(exprs(X), MARGIN, FUN, ...)
}


#Copyright R. Gentleman, 2001
#multiput and multiget
#ideally these will be internalized at some point

#FIXME: I think Luke's Dynamic variables should be used rather than
#the on.exit kludge

multiget <- function(x, pos=-1, envir=as.environment(pos), mode =
                     "any",inherits = TRUE, iffail=NA)
{
    lenx <- length(x)
    ans <- vector("list", length=lenx)
    if( ! is.environment(envir) )
        stop("envir argument is not an environment")
    options(show.error.messages = FALSE)
    on.exit(options(show.error.messages = TRUE))
    for(i in 1:lenx)
        if( is.list(x) )
            ans[[i]] <- try(get(x[[i]],pos,envir, mode, inherits))
        else
            ans[[i]] <- try(get(x[i],pos,envir, mode, inherits))
    options(show.error.messages = TRUE)
    on.exit(NULL)

    failfun <- function(x) {
        cx <- class(x)
        if( !is.null(cx) && cx == "try-error")
            TRUE
        else
            FALSE
    }
    failed <- sapply(ans, failfun)
    ans[failed] <- iffail

    names(ans) <- x
    ans
}

multiassign <- function (x, value, envir = parent.frame(), inherits =
                         FALSE)
{
    if( ! is.environment(envir) )
        stop("envir argument is not an environment")
    if( missing(value) ) {
        nx <- names(x)
        if( any(nchar(nx) == 0) )
            stop("value is missing and x does not have named components")
        value <- x
        x <- nx
    }
    lenx <- length(x)
    for(i in 1:lenx) {
        i2 <- (i-1)%%lenx+1
        if( is.list(x) ) {
            if( is.list(value) )
                assign(x[[i]], value[[i2]], envir=envir,
                         inherits=inherits)
            else
                assign(x[[i]], value[i2], envir=envir,
                         inherits=inherits)
        }
        else {
            if( is.list(value) )
                assign(x[i], value[[i2]], envir=envir,
                         inherits=inherits)
            else
                assign(x[i], value[i2], envir=envir,
                         inherits=inherits)
        }
    }
}
testBioCConnection <- function() {
    ## Stifle the "connected to www.... garbage output
    curNetOpt <- getOption("internet.info")
    on.exit(options(internet.info=curNetOpt), add=TRUE)
    options(internet.info=3)

    ## First check to make sure they have HTTP capability.  If they do
    ## not, there is no point to this exercise.
    http <- as.logical(capabilities(what="http/ftp"))
    if (http == FALSE)
        return(FALSE)

    ## find out where we think that bioC is
    bioCoption <- getOption("BIOC")
    if (is.null(bioCoption))
        bioCoption <- "http://www.bioconductor.org"

    ## Now check to see if we can connect to the BioC website
    biocURL <- url(paste(bioCoption,"/main.html",sep=""))
    options(show.error.messages=FALSE)
    test <- try(readLines(biocURL)[1])
    options(show.error.messages=TRUE)
    if (inherits(test,"try-error"))
        return(FALSE)
    else
        close(biocURL)

    return(TRUE)
}

getPkgVigs <- function(package=NULL) {
    pkgs <- .packages()

    if( !is.null(package) ) {
        if( !is.character(package) )
            stop("`package' must be a character vector of package names")

        rows <- match(package, pkgs)
        if( all(is.na(rows)) )
            stop("packages: ", paste(package,collapse=", "),
                 " are not loaded")
        if( any(is.na(rows)) )
            warning("packages ", paste(package[is.na(rows)], collapse=", "),
                    " are not loaded")
        pkgs <- pkgs[rows[!is.na(rows)]]
    }
    vigDirs <- file.path(.find.package(pkgs), "doc/00Index.dcf")

    vigs <- lapply(vigDirs, function(x){
        if (file.exists(x)) {
            vigs <- read.dcf(x)
            if (nrow(vigs) > 0) {
                vigPaths <- file.path(dirname(x),colnames(vigs))
                vigNames <- as.character(vigs)
                names(vigPaths) <- vigNames
                vigPaths
            } # else NULL
        }# else NULL
    })

    unlist(vigs[sapply(vigs, function(x) !is.null(x) && (length(x) > 0))])
}

openVignette <- function(package=NULL) {
    vigFiles <- getPkgVigs(package)
    names <- names(vigFiles)
    ##indent a little
    names <- paste("",names)
    ##FIXME: why set names to NULL?
    names(vigFiles) <- NULL
    index <- menu(names, title="Please select (by number) a vignette")

    if (index > 0) {
        ## Need to switch on the file extension
        ext <- strsplit(vigFiles[index],"\\.")[[1]]
        switch(ext[length(ext)],
               "pdf" = openPDF(vigFiles[index]),
               "html"= browseURL(paste("file://",vigFiles[index],sep="")),
               stop("Don't know how to handle this vignette")
               )
    }
}
setOptionPdfViewer <- function(viewer,verbose=FALSE) {
    if (missing(viewer)) {
        viewer <- getOption("pdfviewer")
        if (is.null(viewer)) {
            for (x in c("xpdf","acroread","acroread4")) {
                viewer<-system(paste("which",x),intern=TRUE,
                               ignore.stderr=TRUE)
                if( length(viewer) > 0 && file.exists(viewer) )
                    break
                ## It is important to break here due to an OSX problem
                viewer <- character()
            }
            if (length(viewer) == 0) {
                warning("No available PDF viewer found on system")
                return(FALSE)
            }
            if (verbose == TRUE)
                note(paste("Selecting PDF viewer",viewer))
        }
    }

    ## Probably can get away with a few less steps here
    bioOpt <- getOption("BioC")
    bioBase <- bioOpt$Base
    bioBase$pdfViewer <- viewer
    bioOpt$Base <- bioBase
    options("BioC"=bioOpt)
    return(TRUE)
}

openPDF <- function(file, bg=TRUE) {
    OST <- .Platform$OS.type
    if (OST=="windows") {
        shell.exec(file)
    }
    else if (OST == "unix") {
        bioCOpt <- getOption("BioC")
        pdf <- bioCOpt$Base$pdfViewer
        if (is.null(pdf)) {
            warning(paste("pdfViewer is set to:",pdf,
                          "which does not seem to exist.  Please",
                          "run the command setOptionPdfViewer()"))
            return(FALSE)
        }
        cmd <- paste(pdf,file)
        if( bg )
          cmd <- paste(cmd, "&")
        system(cmd)
    }
    return(TRUE)
}


package.version <- function(pkg, lib.loc = NULL) {

    curWarn <- getOption("warn")
    on.exit(options(warn=curWarn),add=TRUE)
    options(warn=-1)

    desc <- package.description(pkg, lib.loc, "Version")
    if (is.na(desc))
        stop(paste("Package",pkg,"does not exist"))
    desc

}
read.MIAME <- function(filename=NULL,widget=getOption("BioC")$Base$use.widgets,...){

  if(!is.null(filename)){
    miame <- scan(filename,what="c",quiet=TRUE,sep="\n",...)
    return(new("MIAME",name=miame[1],lab=miame[2],contact=miame[3],title=miame[4],abstract=miame[5],url=miame[6]))
  }
  else
    if(widget){
      require(tkWidgets) || stop("Requires tkWidgets")
      tmp <- tkMIAME()
      return(new("MIAME",
                 name=tmp$ExperimentName,
                 lab=tmp$LabName,
                 contact=tmp$ContactInfo,
                 title=tmp$ExperimentTitle,
                 abstract=tmp$Description,
                 url=tmp$URL))
    }

    else{
      return(new("MIAME"))
    }
}

# A function to read exprSet from text a text file. The function
# assumes that exprs and se.exprs are tab separated text files.
# The first 6 arguments are for text files to be read in for the slots
# of an exprSet object. seps is for the separators used for exprs and
# se.exprs files. length(seps) = 1 if all two files share
# the same separator or length(seps) = 2 if not. seps are ordered in
# the order as the two arguments appear in the argument list. A
# header is assumed to exist for exprs file. Only exprs is required
# and the others are optional. phenodata has to be the name of a rda
# file that is a list with a pData and a varLabels element.
#


read.exprSet <- function(exprs, se.exprs, phenoData, annotation,
                        description, notes, seps = "\t" ){
    if(missing(exprs)){
        stop("exprs can not be missing!")
    }
    # Read exprs
    geneData <- as.matrix(read.table(exprs, header = TRUE,
                                  sep = seps[1], as.is =  TRUE))
    # Read se.exprs using the right separator
    if(!missing(se.exprs)){
        se.exprs <- as.matrix(read.table(se.exprs, header = FALSE,
                                     sep = ifelse(length(seps == 1),
                                     seps, seps[2]), as.is = TRUE))
    }
    # If phenoData is missing provide a default
    if(missing(phenoData)){
        phenoData <- read.phenoData(NULL, colnames(geneData), FALSE)
    }else{
        phenoData <- read.phenoData(phenoData, colnames(geneData), FALSE)
    }
    # Each gene should have a coVariate?
    if(nrow(pData(phenoData)) != ncol(geneData)){
        warning("Gene and covariate numbers do not match")
    }
    eSet <- new("exprSet", exprs = geneData, phenoData = phenoData)
    if(!missing(se.exprs)){
        se.exprs(eSet) <- se.exprs
    }
    if(!missing(annotation)){
        annotation(eSet) <-  readLines(annotation)
    }
    if(!missing(description)){
        description(eSet) <- read.MIAME(description, FALSE)
    }else{
        description(eSet) <- new("MIAME")
    }
    if(!missing(notes)){
        notes(eSet) <- readLines(notes)
    }

    return(eSet)
}

read.phenoData <- function(filename=NULL,sampleNames=NULL,widget=getOption("BioC")$Base$use.widgets,...){
  
  if(!is.null(filename)){
    pData <- read.table(filename,...)
    if(!is.null(sampleNames)) row.names(pData) <- sampleNames
    varLabels <- as.list(rep("read from file",ncol(pData)))
    names(varLabels) <- names(pData)
    return(new("phenoData",pData=pData,varLabels=varLabels))
  }
  else{
    if(widget){
      require(tkWidgets)
      if(is.null(sampleNames)) stop("To use widget you must supply sampleNames.\n")
      tmp <- tkphenoData(sampleNames)
      pdata <- data.frame(tmp$pData)
      varlabels <- as.list(tmp$varLabels[,1])
      names(varlabels) <- rownames(tmp$varLabels)
      return(new("phenoData",pData=pdata,varLabels=varlabels))
    }
    else{
      if(is.null(sampleNames)){
        return(new("phenoData")) ##return a blank!
      }
      else{
        pdata <- data.frame(sample=1:length(sampleNames),row.names=sampleNames)
        return(new("phenoData",pData=pdata,varLabels=list(sample="arbitrary numbering")))
      }
    }
  }
}
# temporarily store this utility here

.buildBiobaseOpts <- function() {
    if (is.null(getOption("BioC"))) {
        BioC <- list()
        class(BioC) <- "BioCOptions"
        options("BioC"=BioC)
    }

    Base <- list()
    class(Base) <- "BioCPkg"
    Base$urls <- list( bioc = "http://www.bioconductor.org")
    ##RI: I added this to make my life easier. Should it be TRUE?
    ##AJR: NO.  I've run across a few cases when it would completely
    ##     break functionality, i.e. when tcltk isn't part of the R
    ##     package (on weird, and development-based machines
    Base$use.widgets=FALSE

    BioC <- getOption("BioC")
    BioC$Base <- Base
    options("BioC"=BioC)
}

.getPDFOption <- function() {
    OS <- .Platform$OS.type
    if (OS == "unix") {
        if (setOptionPdfViewer() == FALSE) {
            note(paste("To manually set your viewer, run the",
                       "command 'setOptionPdfViewer(<path>),",
                       "where <path> is a path to a valid PDF",
                       "viewer program."))
        }
    }
}

dumpPackTxt <- function (package)
{
# stolen from "library" to get descriptive
# info out on stdout
#
    pkgpath <- .find.package(package)
#    outFile <- tempfile("Rlibrary")
#    outConn <- file(outFile, open = "w")
    docFiles <- file.path(pkgpath, c("TITLE", "DESCRIPTION",
        "INDEX"))
    headers <- c("", "Description:\n\n", "Index:\n\n")
    footers <- c("\n", "\n", "")
    for (i in which(file.exists(docFiles))) {
        writeLines(headers[i], sep = "")
        writeLines(readLines(docFiles[i]) )
        writeLines(footers[i], sep = "")
    }
#    close(outConn)
#    file.show(outFile, delete.file = TRUE)
}

##we need to be more careful about the where argument. If any of these
##function calls load a library, where is wrong from there on...
.First.lib <- function(libname, pkgname, where) {
    ## Define the function inside of .First.lib, as other packages
    ## might be defining this function and we don't want to cause mask
    ## messages.
    checkPkgDeps <- function(pkg) {
        reqVers <- function(req)
            return(gsub("(>)|(<)|(=)|([[:space:]])","",req))

        reqOper <- function(req)
            return(gsub("[^><=]","",req))


        buildDepMtrx <- function(deps) {
            ## Takes in a vector of dependencies, separates any version
            ## requirements.  Returns a matrix, col1 is the packes, col2
            ## is any algebraic requirements
            if (is.null(deps))
                return(NULL)

            deps <- gsub("\\)","",deps)
            deps <- strsplit(deps,"\\(")
            ## Now have a list, some w/ 1 els some w/ more (those w/ reqs)
            deps <- lapply(deps, function(x){
                if (length(x) == 1)
                    return(c(x, ""))
                else
                    return(x)
            }
                           )
            pkgs <- lapply(deps,function(x){return(x[1])})
            reqs <- lapply(deps,function(x){return(x[2])})
            depMtrx <- cbind(matrix(unlist(pkgs)),matrix(unlist(reqs)))
            ## Kill any trailing whitespace on entries
            for (i in 1:2)
                depMtrx[,i] <- gsub("[[:space:]]$","",depMtrx[,i])

            if (depMtrx[1,1] == "NA")
                depMtrx <- NULL

            return(depMtrx)
        }

        if (!is.na(desc <- package.description(pkg,fields="Depends")))
            depMtrx <- buildDepMtrx(strsplit(desc,",[[:space:]]*")[[1]])
        else
            stop(paste("No DESCRIPTION file for package",pkg))

        for (i in 1:nrow(depMtrx)) {
            if (depMtrx[i,1] == "R") {
                ## Should we even check the R version?  DOesn't R do this?
            }
            else {
                depName <- depMtrx[i,1]
                depVers <- depMtrx[i,2]
                outString <- paste("You can not load package",pkg,
                                   "as\nit requires you to have package")

                if (!(depName %in% installed.packages()[,1])) {
                    ## Does not have this package
                    outString <- paste(outString,depName,
                                       "which is not currently installed.")
                    stop(outString)
                }
                if (depVers != "") {
                    ## have a package version
                    outString <- paste(outString, depName, "with a version",
                                       reqOper(depVers),
                                       reqVers(depVers),
                                       "which is not\ncurrently installed.")

                    vers <- package.description(depName,fields="Version")

                    comp <- compareVersion(vers, reqVers(depVers))
                    ## oper can currently be either '>=' or '<='
                    oper <- reqOper(depVers)

                    if ((oper == ">=")&&(comp < 0))
                        stop(outString)
                    else if ((oper == "<=")&&(comp > 0))
                        stop(outString)
                }
            }
        }
    }

    checkPkgDeps(pkgname)
    ## still need methods for 1.6.x users


    .buildBiobaseOpts()
    .getPDFOption()

    cat("Welcome to Bioconductor \n")
    cat("\t Vignettes contain introductory material.  To view, \n")
    cat("\t simply type: openVignette() \n")
    ##        cat("\t to see the available vignettes\n")
    cat("\t For details on reading vignettes, see\n")
    cat("\t the openVignette help page.\n")

    ##set up repository management
    require("reposTools", quietly=TRUE) || stop ("Package reposTools required")


    ##set up menus -- windows only for now
    if( .Platform$OS.type == "windows" && .Platform$GUI == "Rgui" ) {
        addPDF2Vig("Biobase")
    }

}
