ABPkgsBuilder <- function(basePath ="/scratch/homes/jzhang/affymappings",
                          baseMapType = "gb",
                          pkgPath = "/scratch/homes/jzhang/datapkgs",
                          version = "1.2.1",
                          author = list(author = "Jianhua Zhang",
                          maintainer = "jzhang@jimmy.harvard.edu"),
                          pattern = "_.*\\.txt$", pkgOnly = FALSE,
                          makeXML = TRUE){

    srcs <- getPkgFileNames(basePath, pattern = pattern)
    orgs <- unique(unlist(sapply(names(srcs), getOrganism),
                          use.names = FALSE))
    # Get all the source files so that they are downloaded only once
    # for all the data package
    llSrc <- loadFromUrl(getSrcUrl("LL"))
    ugSrc <- list()
    gpSrc <- list()
    keggSrc <- list()
    hgid <- list()
    path <- file.path(.path.package("AnnBuilder"), "data")
    makeSrcInfo()
    for(i in orgs){
        ugSrc[[i]] <- loadFromUrl(getSrcUrl("UG", i))
        gp <- GP(srcUrl = getSrcUrl("GP", organism = i), organism = i)
        options(show.error.messages = FALSE)
        strand <- try(getStrand(gp))
        options(show.error.messages = TRUE)
        if(inherits(strand, "try-error")){
            stop(paste("Failed to parse Golden Path data because of:\n\n",
                       strand))
        }else{
            gpSrc[[i]] <- strand
        }
        kegg <- KEGG(srcUrl = getSrcUrl("KEGG", organism = i), organism = i)
        keggSrc[[i]] <- mapLL2ECNPName(kegg)
        hgid[[i]] <- getLL2IntID(procHomoData(getSrcUrl("HG")), i)
    }
    go <- GO(srcUrl = getSrcUrl("GO"))
    goData <- readData(go, xml = TRUE, fromWeb = TRUE)
    switch(baseMapType,
           gb = baseParser <- getBaseParsers("gb"),
           ug = baseParser <- getBaseParsers("ug"),
           ll = TRUE,
           stop("Invalid imput for baseMapType"))

    for(i in names(srcs)){
        baseName <- srcs[[i]]["ID"]
        otherSrc <- srcs[[i]][names(srcs[[i]]) != "ID"]
        organism <- getOrganism(i)

        base <- try(matrix(scan(baseName, what = "", sep = "\t", quote = "",
                                quiet = TRUE), ncol = 2, byrow = TRUE))
        options(show.error.messages = TRUE)
        if(inherits(base, "try-error")){
            stop(paste("Base file", baseName,
                       "may not be valid or have two columns"))
        }
        colnames(base) <- c("PROBE", "ACC")
        ll <- LL(srcUrl = llSrc,
             parser = baseParser["LL"], baseFile = baseName)
        ug <- UG(srcUrl = ugSrc[[organism]],
             parser = baseParser["UG"], baseFile = baseName,
             organism = organism)
        if(baseMapType != "ll"){
            unified <- unifyMappings(base, ll, ug, otherSrc, FALSE)
        }else{
            unified <- baseName
        }
        # Using the unified mapping as the base file, the data file
        # (ll_tmpl) can be parsed to get annoation data from LocusLink
        # using the correct parser
        parser(ll) <- file.path(path, "llParser")
        baseFile(ll) <- unified
        options(show.error.messages = FALSE)
        annotation <- try(parseData(ll, ncol = 15, fromWeb = FALSE,
                         mergeKey = FALSE))
        options(show.error.messages = TRUE)
        if(inherits(annotation, "try-error")){
            stop(paste("Parsing LocusLink data failed because of:\n\n",
                       annotation))
        }
        colnames(annotation) <- c("PROBE", "ACCNUM", "LOCUSID", "UNIGENE",
                                  "GENENAME", "SYMBOL","CHR", "MAP",
                                  "PMID", "GRIF", "SUMFUNC", "GO",
                                  "OMIM", "NM", "NP")

        annotation <- merge(annotation, gpSrc[[organism]],
                            by = "LOCUSID",  all.x = TRUE)
        annotation <- merge(annotation, keggSrc[[organism]]$llpathname,
                            by = "LOCUSID", all.x = TRUE)
        annotation <- merge(annotation, keggSrc[[organism]]$llec,
                            by = "LOCUSID", all.x = TRUE)
        annotation <- merge(annotation, hgid[[organism]],
                            by = "LOCUSID", all.x = TRUE)

        if(makeXML){
            multC <- colnames(annotation)[is.element(colnames(annotation),
                                                     getMultiColNames())]
            typeC <- colnames(annotation)[is.element(colnames(annotation),
                                                     getTypeColNames())]
            XMLOut <- file.path(pkgPath, paste(i, ".xml", sep = ""))
            fileToXML(targetName = i, outName = XMLOut,
                      inName = annotation, colNames = "",
                      idColName = "PROBE", multColNames = multC,
                      typeColNames = typeC, isFile = FALSE, version = version)
        }
        # Create a data package with no data
        createEmptyDPkg(pkgName = i, pkgPath, force = TRUE)
        annotation <- as.matrix(annotation)
        # Write data to the package for one to one mappings
        for(j in getUniColNames()){
            env <- new.env(hash = TRUE, parent = NULL)
            if(j == "LOCUSID"){
                multiassign(annotation[,"PROBE"],
                            as.integer(annotation[,j]), env)
            }else{
                multiassign(annotation[,"PROBE"],
                            as.vector(annotation[,j]), env)
            }
            assign(paste(i, j, sep = ""), env)
            save(list = paste(i, j, sep = ""), file = file.path(pkgPath,
                      i, "data", paste(i, j, ".rda", sep = "")))
        }
        # Write data to the package for one to many mappings with
        # multiple mappings separated by a ";"
        for(j in intersect(colnames(annotation), getMultiColNames())){
            env <- new.env(hash = TRUE, parent = NULL)
            multiassign(annotation[,"PROBE"],
                        lapply(annotation[,j], splitEntry), env)
            assign(paste(i, j, sep = ""), env)
            save(list = paste(i, j, sep = ""), file = file.path(pkgPath,
                      i, "data", paste(i, j, ".rda", sep = "")))
        }
        # Write the mappings between probe ids and evidence provided by LL
        # and  CHRORI and CHRLOC. All of them have a "@" separating some
        # descriptive values for an entry. Multiple entries are separated
        # by ";"
        for(j in intersect(colnames(annotation), c("GO", "CHRLOC"))){
            env <- new.env(hash = TRUE, parent = NULL)
            if(j == "CHRLOC"){
                multiassign(annotation[, "PROBE"],
                        lapply(annotation[, j], twoStepSplit,
                               asNumeric = TRUE), env)
            }else{
                multiassign(annotation[, "PROBE"],
                            lapply(annotation[, j], twoStepSplit), env)
            }
            assign(paste(i, j, sep = ""), env)
            save(list = paste(i, j, sep = ""), file = file.path(pkgPath,
                      i, "data", paste(i, j, ".rda", sep = "")))
        }
        # Get the reverse mapping between PubMed, pathway, enzyme and probe
        # ids. Column names have to be modified for each reverse mapping
        env <- cols2Env(annotation[,c("PMID", "PROBE")],
                    colNames = c("PMID", "PROBE"), keyColName = "PMID")
        assign(paste(i, "PMID2PROBE", sep = ""), env)
        save(list = paste(i, "PMID2PROBE", sep = ""),
             file =  file.path(pkgPath, i, "data",
             paste(i, "PMID2PROBE.rda", sep = "")))
        env <- cols2Env(annotation[,c("PATH", "PROBE")],
                        colNames = c("PATH", "PROBE"), keyColName = "PATH")
        assign(paste(i, "PATH2PROBE", sep = ""), env)
        save(list = paste(i, "PATH2PROBE", sep = ""),
             file =  file.path(pkgPath, i, "data",
             paste(i, "PATH2PROBE.rda", sep = "")))
        env <- cols2Env(annotation[,c("ENZYME", "PROBE")],
                        colNames = c("ENZYME", "PROBE"),
                        keyColName = "ENZYME")
        assign(paste(i, "ENZYME2PROBE", sep = ""), env)
        save(list = paste(i, "ENZYME2PROBE", sep = ""),
             file =  file.path(pkgPath, i, "data",
             paste(i, "ENZYME2PROBE.rda", sep = "")))
        # Reassign a parser and base file to parse LocusLink data to map
        # GO ids to probe ids
        parser(ll) <- file.path(path, "GO2ProbeParser")
        options(show.error.messages = FALSE)
        go2Probe <- try(parseData(ll, ncol = 3, fromWeb = FALSE))
        options(show.error.messages = TRUE)
        if(inherits(go2Probe, "try-error")){
            stop(paste("Failed to parse LocusLink data because of:\n\n",
                       go2Probe))
        }
        if(is.null(go2Probe)){
            warning("No mappings between GO and probe ids found")
        }else if(nrow(go2Probe) == 1){
            go2Probe <- matrix(go2Probe[, 1:2], nrow = 1)
            colnames(go2Probe) <- c("GO", "GO2PROBE")
            env <- new.env(hash = TRUE, parent = NULL)
            assign(go2Probe["GO"], go2Probe["GO2Probe"], env)
        }else{
            colnames(go2Probe) <- c("GO", "GO2PROBE", "COUNTS")
            # Drop column counts for now. Will get ride of the extra column
            go2Probe <- go2Probe[, c("GO", "GO2PROBE")]
            # Remove "" for GO id resulting from mapping probe ids to NA GO id
            go2Probe <- go2Probe[go2Probe[,1] != "", ]
            # Write GO id to probe id mappings to data package
            env <- new.env(hash = TRUE, parent = NULL)
            multiassign(as.vector(go2Probe[, 1]),
                        sapply(go2Probe[, 2], splitEntry), env)
            assign(paste(i, "GO2PROBE", sep = ""), env)
            save(list = paste(i, "GO2PROBE", sep = ""),
                 file =  file.path(pkgPath, i, "data",
                 paste(i, "GO2PROBE.rda", sep = "")))
        }
        # Get the mappings between GO ids to all the probe ids (including
        # children)
        go2All <- mapGO2AllProbe(go2Probe, goData, "", sep = ";", all = TRUE)
        # Write GO id to all probe ids mappings to data package
        env <- new.env(hash = TRUE, parent = NULL)
        multiassign(names(go2All), sapply(go2All, splitEntry), env)
        assign(paste(i, "GO2ALLPROBES", sep = ""), env)
        save(list = paste(i, "GO2ALLPROBES", sep = ""),
             file =  file.path(pkgPath, i, "data",
             paste(i, "GO2ALLPROBES.rda", sep = "")))
        go2All <- cbind(names(go2All), go2All)
        colnames(go2All) <- c("GO", "GO2ALLPROBES")
        if(makeXML){
            if(!is.null(go2All) && !is.null(go2Probe)){
                mergedGO <- as.matrix(merge(as.matrix(go2All),
                           as.matrix(go2Probe), by = "GO", all.x = TRUE))
                XMLByNum <- file.path(pkgPath, paste(i, "ByNum.xml",
                                                 sep = ""))
                fileToXML(targetName = "GOByNum", outName = XMLByNum,
                          inName = mergedGO, colNames = "",
                          idColName = "GO", multColNames = c("GO2ALLPROBES",
                          "GO2PROBE"), typeColNames = "",
                          isFile = FALSE, version = version)
            }
        }
        writeAccessory(i, pkgPath, organism, version, author)
        # Write the quality control data
        # Quality control
        getDPStats(baseName, i, pkgPath)
        writeMan4QC(i, pkgPath)
    }
    if(!pkgOnly){
        for(j in c("human", "mouse", "rat")){
            chrLengths <- findChrLength(j, getSrcUrl("GP", j))
            for(k in getPkgNameByOrg(pkgPath, j)){
                writeOrganism(k, pkgPath, j)
                writeChrLength(k, pkgPath, chrLengths)
                writeManPage(k, pkgPath, "CHRLENGTHS", organism = j,
                             src = "gp", isEnv = FALSE)
                writeManPage(k, pkgPath, "ORGANISM", organism = j,
                             src = "", isEnv = FALSE)
            }
        }
        GOPkgBuilder(pkgPath = pkgPath, version = version, author = author)
        KEGGPkgBuilder(pkgPath = pkgPath, version = version, author = author)
    }
    # Clearn up
}

getPkgFileNames <- function(basePath, pattern = "_.*\\.txt$"){
    # Create a list for holding the data
    idNSrcs <- list()

    pkgNames <- list.files(basePath, pattern = pattern, full.names = TRUE)

    for(i in unique(gsub("_.*\\.txt", "", pkgNames))){
        files4OnePkg <- pkgNames[grep(paste("^", i, "_.*", sep = ""),
                                     pkgNames)]
        if(length(grep("_ID", files4OnePkg)) > 0){
            files4OnePkg <- c(files4OnePkg[grep("_ID", files4OnePkg)],
                              files4OnePkg[-grep("_ID", files4OnePkg)])
            names(files4OnePkg) <- c("ID",
                                  LETTERS[1:length(files4OnePkg) - 1])
            idNSrcs[[basename(i)]] <- files4OnePkg
        }
    }
    return(idNSrcs)
}

getOrganism <- function(orgChar){
    orgChar <- substr(orgChar, 1, 2)
    switch(toupper(orgChar),
           HG = ,
           HU = return("human"),
           MO = ,
           MG = return("mouse"),
           RA = ,
           RG = return("rat"),
           stop(paste("Unknown entry for orgChar:", orgChar)))
}

getPkgNameByOrg <- function(pkgPath, organism){
    switch(toupper(organism),
           HUMAN = init <- "h",
           MOUSE = init <- "mg\|mo",
           RAT = init <- "rg\|ra",
           stop(paste("Ubknown organism", organism)))

    pkgNames <- list.files(pkgPath, pattern = init)

    return(setdiff(pkgNames, pkgNames[grep("\\.xml$", pkgNames)]))
}

putDPkgsToWeb <- function(pkgPath = "/scratch/homes/jzhang/datapkgs",
                    webPath = "/scratch/homes/jzhang/bioC/webstuff/data",
                    organism = "human", version = "1.2.1", what = "gz"){

    if(organism == "GO" || organism == "KEGG"){
        destDir <- file.path(webPath, "misc", "files", version)
    }else{
        destDir <- file.path(webPath, organism, "files", version)
    }

    if(!file.exists(destDir)){
        dir.create(destDir)
    }

    system(paste("cp", file.path(pkgPath, paste(substr(organism, 1, 1),
                           paste("*.", what, sep = ""), sep = "")), destDir))
}

writeDataHTML <- function(cin =
    "/scratch/homes/jzhang/madman/Rpacks/AnnBuilder/inst/scripts/data.html",
    cout = "/scratch/homes/jzhang/bioC/webstuff/data/data.html",
                          symbolValues = list(VERSION = "1.2.1")){
    copySubstitute(cin, cout, symbolValues)
}
